# -*- coding: utf-8 -*-
import re

from collections import defaultdict
from datetime import date, datetime
from django.utils.translation import ugettext as _
from six import string_types

from events.abc.models import AbcService
from events.staff.models import StaffGroup, StaffOffice


STARTREK_FIELDS = defaultdict(lambda: BaseField())


def register(field_type):
    STARTREK_FIELDS[str(field_type)] = field_type


class TrackerFieldError(Exception):
    def __init__(self, message):
        message = ' '.join([_('Ошибка формирования значения для поля трекера.'), message])
        super().__init__(message)


class BaseField(object):
    slug = None

    def get_error_message(self, field, value):
        args = {'field': field, 'type': self.slug, 'value': value}
        return _('Ожидается поле %(field)s типа %(type)s, пришло значение "%(value)s"') % args

    def get(self, field, value):
        args = {'field': field, 'type': self.slug}
        raise TrackerFieldError(_('Поле %(field)s типа %(type)s еще не реализовано') % args)

    def __str__(self):
        return self.slug


class ArrayField(BaseField):
    slug = 'array'

    def __init__(self, item):
        self.item = item

    def get(self, field, value):
        if not isinstance(value, (tuple, list)):
            value = [value]
        return self.flatten(self.item.get(field, i) for i in value)

    def from_iterable(self, value):
        for it in value:
            if isinstance(it, (tuple, list)):
                for subit in it:
                    if subit is not None:
                        yield subit
            elif it is not None:
                yield it

    def flatten(self, value):
        return list(self.from_iterable(value))

    def __str__(self):
        return '%s/%s' % (self.slug, self.item.slug)


class StringField(BaseField):
    slug = 'string'

    def get(self, field, value):
        return value


class TextField(StringField):
    slug = 'text'


class IntegerField(BaseField):
    slug = 'integer'
    int_pattern = re.compile(r'^-?\d+$')

    def get(self, field, value):
        if isinstance(value, (int, float)):
            return int(value)
        if isinstance(value, string_types) and self.int_pattern.match(value):
            return int(value)
        raise TrackerFieldError(self.get_error_message(field, value))


class FloatField(BaseField):
    slug = 'float'
    float_pattern = re.compile(r'^-?\d+(?:\.\d+)?$')

    def get(self, field, value):
        if isinstance(value, (int, float)):
            return float(value)
        if isinstance(value, string_types):
            value = value.replace(',', '.')
            if self.float_pattern.match(value):
                return float(value)
        raise TrackerFieldError(self.get_error_message(field, value))


class MoneyField(BaseField):
    slug = 'money'
    unit_types = ['RUB', 'USD', 'EUR', 'UAH', 'TRY']

    def get(self, field, value):
        if isinstance(value, (int, float)):
            return value
        if isinstance(value, string_types):
            params = {}
            for part in value.split():
                if IntegerField.int_pattern.match(part):
                    params['amount'] = int(part)
                elif FloatField.float_pattern.match(part):
                    params['amount'] = float(part)
                elif part in self.unit_types:
                    params['unit'] = part
            if 'amount' in params:
                return params
        raise TrackerFieldError(self.get_error_message(field, value))


class DateField(BaseField):
    slug = 'date'
    iso_pattern = re.compile(r'^[12]\d{3}-[01]\d-[0-3]\d$')
    js_pattern = re.compile(r'^([0-3]\d)\.([01]\d)\.([12]\d{3})$')

    def get(self, field, value):
        if isinstance(value, datetime):
            return value.date().isoformat()
        if isinstance(value, date):
            return value.isoformat()
        if isinstance(value, string_types):
            if self.iso_pattern.match(value):
                return value
            m = self.js_pattern.match(value)
            if m:
                return '-'.join(m.groups()[::-1])
        raise TrackerFieldError(self.get_error_message(field, value))


class DateTimeField(BaseField):
    slug = 'datetime'
    datetime_pattern = re.compile(r'^[12]\d{3}-[01]\d-[0-3]\d(?:T[0-2]\d:[0-5]\d:[0-5]\d(?:\.\d{3,6})?(?:[+-]\d\d:?\d\d|Z)?)?$')

    def get(self, field, value):
        if isinstance(value, datetime):
            return self.localize(value).isoformat()
        if isinstance(value, date):
            return value.isoformat()
        if isinstance(value, str) and self.datetime_pattern.match(value):
            return value
        raise TrackerFieldError(self.get_error_message(field, value))

    def localize(self, value):
        from django.utils import timezone
        if not value.tzinfo:
            return timezone.get_default_timezone().localize(value)
        return value


class TimeTrackingField(BaseField):
    slug = 'timetracking'
    validate_pattern = re.compile(r'^(?:\d{1,3}[wW]\s*)?(?:\d{1,3}[dD]\s*)?(?:\d{1,3}[hH]\s*)?(?:\d{1,3}[mM]\s*)?(?:\d{1,3}[sS]\s*)?$')
    names = 'weeks days hours minutes seconds delim'.split()

    def get(self, field, value):
        if isinstance(value, string_types):
            if self.validate_pattern.match(value):
                return self.to_startrek(value)
        raise TrackerFieldError(self.get_error_message(field, value))

    def to_startrek(self, value):
        params = dict(list(zip(self.names, [''] * len(self.names))))
        time_section = False
        for i in value.split():
            if i[-1] in 'wW':
                params['weeks'] = i.upper()
            elif i[-1] in 'dD':
                params['days'] = i.upper()
            elif i[-1] in 'hH':
                params['hours'] = i.upper()
                time_section = True
            elif i[-1] in 'mM':
                params['minutes'] = i.upper()
                time_section = True
            elif i[-1] in 'sS':
                params['seconds'] = i.upper()
                time_section = True
        if time_section:
            params['delim'] = 'T'
        return 'P{weeks}{days}{delim}{hours}{minutes}{seconds}'.format(**params)


class UriField(StringField):
    slug = 'uri'


class UserField(StringField):
    slug = 'user'

    def get(self, field, value):
        # делаем из строки с логинами разделенными запятой массив логинов
        # стрипаем пробелы и игнорируем пустые значения
        if isinstance(value, str) and value.find(',') != -1:
            g = (login.strip() for login in value.split(','))
            return [login for login in g if login != '']
        return value


class TranslationField(StringField):
    slug = 'translation'


class IssueField(StringField):
    slug = 'issue'


class VersionField(StringField):
    slug = 'version'


class ComponentField(StringField):
    slug = 'component'


class ProjectField(StringField):
    slug = 'project'


class ResolutionField(StringField):
    slug = 'resolution'


class PriorityField(StringField):
    slug = 'priority'


class MaillistField(StringField):
    slug = 'maillist'


class JsonField(BaseField):
    def get(self, field, value):
        if isinstance(value, string_types):
            if value.isdigit():
                return {'id': int(value)}
            else:
                return {'slug': value}
        return value


class GoalField(JsonField):
    slug = 'goal'


class BoardField(IntegerField):
    slug = 'board'


class ObjectField(BaseField):
    model = None
    field_name = None

    def get_object(self, value):
        return self.model.objects.filter(name=value).first()

    def get(self, field, value):
        obj_id = None
        if isinstance(value, string_types):
            if value.isdigit():
                obj_id = int(value)
            else:
                obj = self.get_object(value)
                if obj:
                    obj_id = getattr(obj, self.field_name, None)
        if obj_id is not None:
            return {'id': obj_id}


class GroupField(ObjectField):
    slug = 'group'
    model = StaffGroup
    field_name = 'staff_id'


class ServiceField(ObjectField):
    slug = 'service'
    model = AbcService
    field_name = 'abc_id'

    def get_object(self, value):
        return (
            self.model.objects.filter(is_deleted=False)
            .filter_exact(value, 'name').first()
        )


class SprintField(JsonField):
    slug = 'sprint'


class OfficeField(ObjectField):
    slug = 'office'
    model = StaffOffice
    field_name = 'staff_id'

    def get_object(self, value):
        return self.model.objects.filter_exact(value, 'name').first()


register(ArrayField(ComponentField()))
register(ArrayField(GoalField()))
register(ArrayField(GroupField()))
register(ArrayField(MaillistField()))
register(ArrayField(ServiceField()))
register(ArrayField(SprintField()))
register(ArrayField(StringField()))
register(ArrayField(TranslationField()))
register(ArrayField(UserField()))
register(ArrayField(VersionField()))
register(ArrayField(BoardField()))
register(DateField())
register(DateTimeField())
register(FloatField())
register(IntegerField())
register(IssueField())
register(MoneyField())
register(PriorityField())
register(ProjectField())
register(ResolutionField())
register(StringField())
register(TextField())
register(TimeTrackingField())
register(TranslationField())
register(UriField())
register(UserField())
register(GroupField())
register(ServiceField())
register(OfficeField())
