# -*- coding: utf-8 -*-
import logging

from django.db import models
from django.db.models.query import QuerySet

from events.accounts.org_ctx import get_org_dir_id
from events.common_app.managers import ProxyToQuerySetMixin, SyncWithSourceMixin
from events.common_app.startrek.client import get_startrek_client
from events.startrek.conditions import StartrekFieldConditions


logger = logging.getLogger(__name__)


class StartrekFieldQuerySet(QuerySet):
    def filter_by_fit_for_search_text(self, search_text):
        return self.filter(StartrekFieldConditions.is_fit_for_search_text(search_text=search_text))

    def filter_by_slugs(self, slug_list):
        return self.filter(StartrekFieldConditions.filter_by_slugs(slug_list))

    def filter_optional_fields(self):
        return self.filter(StartrekFieldConditions.filter_optional_fields())


class StartrekFieldManager(ProxyToQuerySetMixin, SyncWithSourceMixin, models.Manager):
    sync_resource_name = 'startrek_field'

    def get_queryset(self):
        return StartrekFieldQuerySet(self.model, using=self._db)

    def sync_with_source(self):
        logger.info('[%s] Started sync with source.', self.sync_resource_name)

        fields = []
        startrek_client = get_startrek_client(dir_id=get_org_dir_id())
        for i in startrek_client.get_fields():
            schema = i.get('schema') or {}
            schema_type = schema.get('type')
            schema_items = schema.get('items')
            if schema_type == 'array':
                schema_type = f'{schema_type}/{schema_items}'
            if schema.get('required', None) is None:
                schema['required'] = False
            field = {
                'id': i.get('id'),
                'name': i.get('name'),
                'order': i.get('order'),
                'readonly': i.get('readonly'),
                'schema': schema,
                'self': i.get('self'),
                'version': i.get('version'),
                'options': False,
                'optionValues': None,
            }
            fields.append(field)

        self.process_response(fields)
        logger.info(
            '[%s] Processed (len: %s).', self.sync_resource_name, len(fields)
        )
