# -*- coding: utf-8 -*-
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db.models import Q
from collections import defaultdict
from guardian import shortcuts
from guardian.models import UserObjectPermission, GroupObjectPermission

from events.accounts.models import User, Organization, create_organization
from events.staff.models import StaffGroup
from events.surveyme.models import Survey, SurveyGroup, SurveyQuestion
from events.celery_app import app


class UserNotFound(Exception):
    pass


class OrganizationNotFound(Exception):
    pass


def _return_error(message):
    return {'status': 'error', 'message': message}


def _return_success():
    return {'status': 'success'}


def _return_inprogress(task_id):
    return {'status': 'inprogress', 'task_id': task_id}


def _is_user_in_org_group(user, org):
    return user.groups.filter(pk=org.o2g.group.pk).exists()


def _find_user(uid=None, cloud_uid=None, email=None):
    if email and not (uid or cloud_uid):
        try:
            user = User.objects.get(email=email)
        except User.DoesNotExist:
            raise UserNotFound(f'Пользователь с эмейлом {email} не найден')
    elif uid or cloud_uid:
        condition = Q()
        if uid:
            condition |= Q(uid=uid)
        if cloud_uid:
            condition |= Q(cloud_uid=cloud_uid)
        users = list(User.objects.filter(condition))
        if not users:
            raise UserNotFound(f'Пользователь с uid={uid}, cloud_uid={cloud_uid} не найден')
        if len(users) > 1:
            raise UserNotFound(f'Найдено несколько пользователей с uid={uid}, cloud_uid={cloud_uid}')
        user = users[0]
    else:
        raise UserNotFound('Не переданы реквизиты для поиска пользователя')
    return user


def _change_owner(survey_id, uid=None, cloud_uid=None, email=None):
    try:
        survey = Survey.objects.using(settings.DATABASE_ROLOCAL).get(pk=survey_id)
    except Survey.DoesNotExist:
        return _return_error(f'Форма {survey_id} не существует')

    try:
        user = _find_user(uid, cloud_uid, email)
    except UserNotFound as e:
        return _return_error(str(e))

    if survey.org:
        if not _is_user_in_org_group(user, survey.org):
            url = f'https://catalog.ws.yandex-team.ru/organizations/{survey.org.dir_id}/'
            return _return_error(f'Пользователь {user.email} не состоит в организации {url}')

    shortcuts.assign_perm('change_survey', user, survey)
    shortcuts.remove_perm('change_survey', survey.user, survey)

    survey.user = user
    survey.save(update_fields=['user'])
    return _return_success()


def change_owner(survey_id, uid=None, cloud_uid=None, email=None):
    from events.support import tasks
    if email and not (uid or cloud_uid):
        async_result = tasks.change_owner.delay(survey_id, uid, cloud_uid, email)
        return _return_inprogress(async_result.task_id)
    return _change_owner(survey_id, uid, cloud_uid, email)


def _find_organization(dir_id):
    try:
        org = Organization.objects.get(dir_id=dir_id)
    except Organization.DoesNotExist:
        org = create_organization(dir_id)

    if not org:
        raise OrganizationNotFound(f'Не удалось создать организацию с кодом {dir_id}')
    return org


def change_organization(survey_id, dir_id):
    try:
        survey = Survey.objects.using(settings.DATABASE_ROLOCAL).get(pk=survey_id)
    except Survey.DoesNotExist:
        return _return_error(f'Форма {survey_id} не существует')

    if dir_id:
        try:
            org = _find_organization(dir_id)
        except OrganizationNotFound as e:
            return _return_error(str(e))

        if not _is_user_in_org_group(survey.user, org):
            url = f'https://catalog.ws.yandex-team.ru/organizations/{org.dir_id}/'
            return _return_error(f'Автор {survey.user.email} формы {survey_id} не состоит в организации {url}')
    else:
        org = None

    survey.org = org
    survey.save(update_fields=['org'])
    return _return_success()


def get_task_info(task_id):
    task = app.AsyncResult(task_id)
    is_ready = task.ready()
    is_failed = task.failed()
    if is_ready:
        try:
            result = task.get()
        except Exception as e:
            if is_failed:
                result = {'error': str(e)}
            else:
                raise
    else:
        result = None
    return {
        'task_id': task_id,
        'ready': is_ready,
        'failed': is_failed,
        'result': result,
    }


def get_question_info(row):
    return {
        'id': row.pk,
        'label': row.label,
        'slug': row.param_slug,
        'answer_type': row.answer_type__slug,
        'page': row.page,
        'position': row.position,
        'group_id': row.group_id,
        'is_required': row.param_is_required,
        'is_hidden': row.param_is_hidden,
        'is_deleted': row.is_deleted,
        'translations': row.translations,
    }


def get_questions_info(survey_id):
    qs = (
        SurveyQuestion.with_deleted_objects.all()
        .filter(survey_id=survey_id)
        .values_list(
            'pk', 'label', 'param_slug', 'answer_type__slug', 'page', 'position', 'group_id',
            'param_is_required', 'param_is_hidden', 'is_deleted', 'translations', named=True,
        )
        .order_by('page', 'position')
    )
    return [
        get_question_info(row)
        for row in qs
    ]


def get_user_info(row):
    user_info = {
        'id': row.user__pk,
        'email': row.user__email,
        'uid': row.user__uid,
    }
    if settings.IS_BUSINESS_SITE:
        user_info['cloud_uid'] = row.user__cloud_uid
    return user_info


def get_group_info(row, staff_groups):
    if row.group__name not in staff_groups:
        return {'name': row.group__name}
    return staff_groups[row.group__name]


def get_user_permissions(survey_id, survey_group_id, result=None):
    cts = ContentType.objects.get_for_models(Survey, SurveyGroup)
    condition = Q(content_type=cts[Survey], object_pk=str(survey_id))
    if survey_group_id:
        condition |= Q(content_type=cts[SurveyGroup], object_pk=str(survey_group_id))
    perms = (
        UserObjectPermission.objects.filter(condition)
        .values_list(
            'user__pk', 'user__uid', 'user__cloud_uid', 'user__email',
            'permission__codename', named=True,
        )
    )
    result = result or defaultdict(list)
    for row in perms:
        result[row.permission__codename].append(get_user_info(row))
    return result


def get_staff_groups(group_names):
    staff_groups = {}
    if not settings.IS_BUSINESS_SITE:
        group_ids = []
        for group_name in group_names:
            if group_name.startswith('group:'):
                group_id = group_name[6:]
                if group_id.isdigit():
                    group_ids.append(group_id)
        if group_ids:
            qs = (
                StaffGroup.objects.filter(staff_id__in=group_ids)
                .values_list('staff_id', 'name', 'url', named=True)
            )
            for row in qs:
                staff_groups[f'group:{row.staff_id}'] = {
                    'id': row.staff_id,
                    'name': row.name,
                    'role': row.url,
                }
    return staff_groups


def get_group_permissions(survey_id, survey_group_id, result=None):
    cts = ContentType.objects.get_for_models(Survey, SurveyGroup)
    condition = Q(content_type=cts[Survey], object_pk=str(survey_id))
    if survey_group_id:
        condition |= Q(content_type=cts[SurveyGroup], object_pk=str(survey_group_id))
    perms = list(
        GroupObjectPermission.objects.filter(condition)
        .values_list('group__name', 'permission__codename', named=True)
    )
    staff_groups = get_staff_groups(row.group__name for row in perms)
    result = result or defaultdict(list)
    for row in perms:
        result[row.permission__codename].append(get_group_info(row, staff_groups))
    return result


def get_survey_permissions_info(survey_id, survey_group_id):
    result = get_user_permissions(survey_id, survey_group_id)
    return get_group_permissions(survey_id, survey_group_id, result)


def get_survey_info(survey_id):
    fields = [
        'pk', 'name', 'slug', 'group_id', 'date_created', 'date_published',
        'date_unpublished', 'is_published_external', 'is_public',
        'is_ban_detected', 'is_deleted', 'translations',
        'user__pk', 'user__uid', 'user__cloud_uid', 'user__email',
    ]
    if settings.IS_BUSINESS_SITE:
        fields.append('org__dir_id')
    row = (
        Survey.with_deleted_objects.all()
        .values_list(*fields, named=True)
        .get(pk=survey_id)
    )
    return {
        'id': row.pk,
        'name': row.name,
        'slug': row.slug,
        'dir_id': getattr(row, 'org__dir_id', None),
        'group_id': row.group_id,
        'date_created': row.date_created,
        'date_published': row.date_published,
        'date_unpublished': row.date_unpublished,
        'author': get_user_info(row),
        'is_published': row.is_published_external,
        'is_public': row.is_public,
        'is_ban_detected': row.is_ban_detected,
        'is_deleted': row.is_deleted,
        'translations': row.translations,
        'permissions': get_survey_permissions_info(row.pk, row.group_id),
        'questions': get_questions_info(row.pk),
    }
