# -*- coding: utf-8 -*-
from collections import defaultdict
from operator import itemgetter
from urllib.parse import quote_plus
from django.conf import settings
from django.db.models import Q
from django.utils.translation import ugettext as _
from events.surveyme.models import SurveyQuestion


class BaseValue:
    slug = None

    def __init__(self, answer_data):
        self.answer_data = answer_data

    def get_raw_value(self, question, **kwargs):
        stop_iter = kwargs.get('stop_iter', False)
        if question.group_id and not stop_iter:
            group_value = self.answer_data.get(question.group_id, {}).get('value') or []
            for i, field_set in enumerate(group_value):
                for child in field_set:
                    if child and child.get('question', {}).get('id') == question.pk:
                        yield i, child.get('value')
        else:
            yield 0, self.answer_data.get(question.pk, {}).get('value')

    def get_delimiter(self, question, **kwargs):
        return ', '

    def get_value(self, question, **kwargs):
        unfold_group = kwargs.get('unfold_group', False)
        delimiter = self.get_delimiter(question, **kwargs)
        list_value = [
            (idx, self.cast(raw_value, question, **kwargs))
            for idx, raw_value in self.get_raw_value(question, **kwargs)
        ]
        if unfold_group:
            return {
                f'{question.param_slug}__{idx}': value
                for idx, value in list_value
            }
        return delimiter.join(map(itemgetter(1), list_value))

    def cast(self, raw_value, question, **kwargs):
        raise NotImplementedError


class TextValue(BaseValue):
    def cast(self, raw_value, question, **kwargs):
        if isinstance(raw_value, str):
            return raw_value
        return ''


class ShortTextValue(TextValue):
    slug = 'answer_short_text'


class LongTextValue(TextValue):
    slug = 'answer_long_text'


class NameValue(TextValue):
    slug = 'answer_name'


class SurnameValue(TextValue):
    slug = 'answer_surname'


class PhoneValue(TextValue):
    slug = 'answer_phone'


class UrlValue(TextValue):
    slug = 'answer_url'


class EmailValue(TextValue):
    slug = 'answer_non_profile_email'


class NumberValue(BaseValue):
    slug = 'answer_number'

    def cast(self, raw_value, question, **kwargs):
        if isinstance(raw_value, int):
            return str(raw_value)
        return ''


class DateValue(BaseValue):
    slug = 'answer_date'

    def cast(self, raw_value, question, **kwargs):
        if isinstance(raw_value, dict):
            return '{begin} - {end}'.format(**raw_value)
        if isinstance(raw_value, str):
            return raw_value
        return ''


class FilesValue(BaseValue):
    slug = 'answer_files'

    def get_file_value(self, file_item, show_filenames):
        if isinstance(file_item, dict):
            if show_filenames:
                return file_item.get('name')

            path = file_item.get('path')
            if path:
                return 'https://{domain}{prefix}/files?path={path}'.format(
                    domain=settings.DEFAULT_FRONTEND_DOMAIN,
                    prefix=settings.FILE_PATH_PREFIX,
                    path=quote_plus(path),
                )

    def get_delimiter(self, question, **kwargs):
        return '\n'

    def cast(self, raw_value, question, **kwargs):
        delimiter = self.get_delimiter(question, **kwargs)
        show_filenames = kwargs.get('show_filenames', False)
        if isinstance(raw_value, list):
            value = [
                self.get_file_value(file_item, show_filenames)
                for file_item in raw_value
            ]
            return delimiter.join(value)
        return ''


class PaymentValue(BaseValue):
    slug = 'answer_payment'

    def cast(self, raw_value, question, **kwargs):
        answer_id = kwargs.get('answer_id')
        if isinstance(raw_value, dict):
            amount = raw_value.get('amount')
            if amount:
                return 'Оплата заказа {order} на сумму {amount} руб'.format(
                    order=answer_id,
                    amount=amount,
                )
        return ''


class ChoicesValue(BaseValue):
    slug = 'answer_choices'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._labels = None

    def _get_labels(self, question):
        data_source = question.param_data_source
        if data_source == 'survey_question_choice':
            labels = {
                str(choice.pk): choice.get_label()
                for choice in question.surveyquestionchoice_set.all()
            }
        elif data_source == 'survey_question_matrix_choice':
            labels = {
                str(title.pk): title.get_label()
                for title in question.surveyquestionmatrixtitle_set.all()
            }
        else:
            labels = {}
        return labels

    def get_labels(self, question):
        # оптимизация на случай если вопрос является частью серии
        # для чтобы не ходить лишний раз в базу за текстами
        # вариантов ответов
        if self._labels is None:
            self._labels = self._get_labels(question)
        return self._labels

    def get_from_labels(self, key, item, labels):
        if key in labels:
            return labels[key]
        return item.get('text')

    def get_choice_value(self, choice_item, labels):
        key = choice_item.get('key')
        if key:
            return self.get_from_labels(key, choice_item, labels)
        else:
            row = choice_item.get('row', {})
            col = choice_item.get('col', {})
            if row and col:
                row_text = self.get_from_labels(row.get('key'), row, labels)
                col_text = self.get_from_labels(col.get('key'), col, labels)
                return '"{row}": {col}'.format(
                    row=row_text,
                    col=col_text,
                )

    def get_delimiter(self, question, **kwargs):
        if question.param_data_source == 'survey_question_matrix_choice':
            return '\n'
        return ', '

    def get_value(self, question, **kwargs):
        self.labels = self.get_labels(question)
        return super().get_value(question, **kwargs)

    def cast(self, raw_value, question, **kwargs):
        delimiter = self.get_delimiter(question, **kwargs)
        if isinstance(raw_value, list):
            return delimiter.join(
                self.get_choice_value(choice_item, self.labels)
                for choice_item in raw_value
            )
        return ''


class BooleanValue(BaseValue):
    slug = 'answer_boolean'

    def cast(self, raw_value, question, **kwargs):
        if raw_value is True:
            return _('Yes')
        elif raw_value is False:
            return _('No')
        return ''


class GroupValue(BaseValue):
    slug = 'answer_group'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.value_objs = {}

    def get_value_obj(self, question):
        value_obj = None
        if question.pk in self.value_objs:
            value_obj = self.value_objs[question.pk]
        else:
            value_class = value_classes.get(question.answer_type.slug)
            if value_class is not None:
                value_obj = value_class(None)
            self.value_objs[question.pk] = value_obj
        return value_obj

    def get_generic_value(self, answer_data, question, **kwargs):
        # оптимизация для серии вопросов, чтобы не получилось так
        # что на каждый списочный вопрос мы ходим в базу за
        # названиями вариантов ответа
        value_obj = self.get_value_obj(question)
        if value_obj:
            value_obj.answer_data = answer_data
            return value_obj.get_value(question, **kwargs)

    def get_value(self, question, **kwargs):
        only_with_value = kwargs.get('only_with_value', False)
        groups = kwargs.get('groups', {})
        children = groups.get(question.pk, [])
        _, raw_value = next(self.get_raw_value(question, **kwargs))
        if raw_value:
            group_label = question.get_label()
            value = []
            for field_set in raw_value:
                children_data = {
                    child.get('question', {}).get('id'): child
                    for child in field_set or []
                    if child
                }
                children_value = []
                for child in children:
                    child_value = self.get_generic_value(children_data, child, stop_iter=True, **kwargs)
                    if child_value or not only_with_value:
                        children_value.append('{label} - {value}'.format(
                            label=child.get_label(),
                            value=child_value,
                        ))

                if children_value or not only_with_value:
                    value.append('{group_label}\n{children_value}'.format(
                        group_label=group_label,
                        children_value='\n'.join(children_value),
                    ))
            return '\n\n'.join(value)
        return ''


value_classes = {
    value_class.slug: value_class
    for value_class in (
        ShortTextValue,
        LongTextValue,
        NameValue,
        SurnameValue,
        PhoneValue,
        UrlValue,
        EmailValue,
        NumberValue,
        DateValue,
        FilesValue,
        PaymentValue,
        ChoicesValue,
        BooleanValue,
        GroupValue,
    )
}


def get_generic_value(answer_data, question, **kwargs):
    value_class = value_classes.get(question.answer_type.slug)
    if value_class:
        return value_class(answer_data).get_value(question, **kwargs)


def get_profile_value(profile, attr_name):
    result = getattr(profile, attr_name, None)
    if result:
        return str(result)
    return ''


def get_value(answer, question, **kwargs):
    return get_generic_value(answer.as_dict(), question, **kwargs)


def get_value_for_question(answer, question_id, **kwargs):
    questions_qs = (
        SurveyQuestion.objects.using(settings.DATABASE_ROLOCAL)
        .select_related('answer_type')
        .filter(Q(answer_type__is_read_only=False) | Q(answer_type__slug='answer_group'))
        .filter(Q(pk=question_id) | Q(group_id=question_id))
        .filter(survey_id=answer.survey_id)
    )
    question, group = None, []
    for q in questions_qs:
        if q.pk == question_id:
            question = q
        else:
            group.append(q)

    if question:
        groups = {}
        if group:
            groups[question_id] = sorted(group, key=lambda q: q.position)

        kwargs['groups'] = groups
        kwargs['answer_id'] = answer.pk
        kwargs['only_with_value'] = True

        return get_value(answer, question, **kwargs)


def get_value_for_questions(answer, question_ids=None, **kwargs):
    questions_qs = (
        SurveyQuestion.objects.using(settings.DATABASE_ROLOCAL)
        .select_related('answer_type')
        .prefetch_related(
            'surveyquestionchoice_set',
            'surveyquestionmatrixtitle_set',
        )
        .filter(Q(answer_type__is_read_only=False) | Q(answer_type__slug='answer_group'))
        .filter(survey_id=answer.survey_id)
    )
    if question_ids:
        questions_qs = questions_qs.filter(Q(pk__in=question_ids) | Q(group_id__in=question_ids))

    questions, groups = [], defaultdict(list)
    for question in questions_qs.order_by('page', 'position'):
        if not question.group_id:
            questions.append(question)
        else:
            groups[question.group_id].append(question)

    if questions:
        kwargs['groups'] = groups
        kwargs['answer_id'] = answer.pk

        data = [
            (question.get_label(), get_value(answer, question, **kwargs))
            for question in questions
        ]
        if kwargs.get('only_with_value', False):
            data = [
                (label, value)
                for label, value in data
                if value
            ]
        return data


def find_question_data(answer_data, question_id):
    if isinstance(answer_data, list):
        for it in answer_data:
            if not it:
                continue
            if question_id == it.get('question', {}).get('id'):
                yield it
            else:
                slug = it.get('question', {}).get('answer_type', {}).get('slug')
                if slug == 'answer_group':
                    for fieldset in it.get('value') or []:
                        yield from find_question_data(fieldset, question_id)
