# -*- coding: utf-8 -*-
import django_filters

from django.conf import settings
from django.db.models import Q
from guardian.shortcuts import get_objects_for_user
from rest_framework.filters import BaseFilterBackend

from events.rest_framework_contrib import queryset_filters
from events.rest_framework_contrib.permissions import HasChangePermission
from events.surveyme.models import (
    AnswerType,
    ProfileSurveyAnswer,
    Survey,
    SurveyGroup,
    SurveyQuestion,
    SurveyStateConditionNode,
    SurveyStyleTemplate,
    SurveyText,
    SurveyQuestionShowConditionNode,
    SurveySubmitConditionNode,
    SURVEY_OWNERSHIP_MYFORMS,
    SURVEY_OWNERSHIP_SHARED,
    SURVEY_STATUS_PUBLISHED,
    SURVEY_STATUS_UNPUBLISHED,
)


class SurveyGroupFilter(django_filters.FilterSet):
    id__in = queryset_filters.NumberInFilter(field_name='id', lookup_expr='in')
    ordering = django_filters.OrderingFilter(fields=(
        ('id', 'id'),
        ('name', 'name'),
    ))

    class Meta:
        model = SurveyGroup
        fields = (
            'id',
        )


class SurveyFilter(django_filters.FilterSet):
    search = queryset_filters.MethodFilter('filter_by_fit_for_search_text')
    id__in = queryset_filters.NumberInFilter(field_name='id', lookup_expr='in')
    group__in = queryset_filters.NumberInFilter(field_name='group', lookup_expr='in')
    status = django_filters.CharFilter(method='filter_by_publication_status')
    ownership = django_filters.CharFilter(method='filter_by_ownership')
    org = django_filters.CharFilter(method='filter_by_org')
    user_uid = django_filters.CharFilter(method='filter_by_user_uid')
    ordering = django_filters.OrderingFilter(fields=(
        ('id', 'id'),
        ('date_created', 'date_created'),
        ('date_updated', 'date_updated'),
        ('name', 'name'),
        ('answercount__count', 'answers_count'),
    ))

    def filter_by_user_uid(self, queryset, name, value):
        return queryset.filter(user__uid=value)

    def filter_by_org(self, queryset, name, value):
        conditions = Q()
        orgs = set(value.split(','))
        if 'mine' in orgs:
            conditions |= Q(org_id__isnull=True, user=self.request.user)
            orgs.remove('mine')
        if orgs:
            conditions |= Q(org_id__dir_id__in=orgs)
        if conditions:
            queryset = queryset.filter(conditions)
        return queryset

    def filter_by_ownership(self, queryset, name, value):
        if value == SURVEY_OWNERSHIP_MYFORMS:
            return queryset.filter(user=self.request.user)
        elif value == SURVEY_OWNERSHIP_SHARED:
            return queryset.filter(~Q(user=self.request.user))
        return queryset

    def filter_by_publication_status(self, queryset, name, value):
        if value == SURVEY_STATUS_PUBLISHED:
            return queryset.filter(is_published_external=True)
        elif value == SURVEY_STATUS_UNPUBLISHED:
            return queryset.filter(is_published_external=False)
        return queryset

    class Meta:
        model = Survey
        fields = (
            'id',
            'type',
            'status',
            'ownership',
            'group',
            'org',
        )


class AnswerTypeFilter(django_filters.FilterSet):
    class Meta:
        model = AnswerType
        fields = (
            'is_allow_choices',
            'is_could_be_used_in_conditions',
            'is_read_only'
        )


class SurveyQuestionFilter(django_filters.FilterSet):
    survey = django_filters.CharFilter(field_name='survey', lookup_expr='exact')

    class Meta:
        model = SurveyQuestion
        fields = (
            'survey',
            'answer_type',
            'position',
            'page',
        )


class ProfileSurveyAnswerSurveyFilter(django_filters.FilterSet):
    class Meta:
        model = Survey
        fields = (
            'type',
        )


class ProfileSurveyAnswerFilter(django_filters.FilterSet):
    search = queryset_filters.MethodFilter('filter_by_fit_for_search_text')
    id__in = django_filters.Filter(
        method=lambda qs, value: qs.filter(**{'id__in': value.split(',')}) if value is not None else qs
    )
    survey = django_filters.CharFilter(field_name='survey', lookup_expr='exact')

    class Meta:
        model = ProfileSurveyAnswer
        fields = (
            'survey',
            'user',
        )


class SurveyStateConditionNodeFilter(django_filters.FilterSet):
    survey = django_filters.CharFilter(field_name='survey', lookup_expr='exact')

    class Meta:
        model = SurveyStateConditionNode
        fields = (
            'survey',
        )


class UserProfileSurveyAnswerSurveyFilter(django_filters.FilterSet):
    class Meta:
        model = Survey
        fields = (
            'type',
        )


class UserProfileSurveyAnswerFilter(django_filters.FilterSet):
    survey = django_filters.CharFilter(field_name='survey', lookup_expr='exact')

    class Meta:
        model = ProfileSurveyAnswer
        fields = (
            'survey',
        )


class SurveyStyleTemplateFilter(django_filters.FilterSet):
    class Meta:
        model = SurveyStyleTemplate
        fields = (
            'type',
        )


class SurveyTextFilter(django_filters.FilterSet):
    survey = django_filters.CharFilter(field_name='survey', lookup_expr='exact')

    class Meta:
        model = SurveyText
        fields = (
            'survey',
        )


class SurveyQuestionShowConditionNodeFilter(django_filters.FilterSet):
    survey_question__survey = django_filters.CharFilter(method='filter_survey')

    def filter_survey(self, qs, field_name, value):
        return qs.filter(survey_question__survey=value)

    class Meta:
        model = SurveyQuestionShowConditionNode
        fields = (
            'survey_question',
            'survey_question__survey',
        )


class SurveySubmitConditionNodeFilter(django_filters.FilterSet):
    survey = django_filters.CharFilter(field_name='survey', lookup_expr='exact')

    class Meta:
        model = SurveySubmitConditionNode
        fields = (
            'survey',
            'survey_id',
        )


class SurveyFilterBackend(BaseFilterBackend):
    def filter_queryset(self, request, queryset, view):
        if request.user.is_superuser:
            return queryset
        return self._filter_queryset(request, queryset, view)

    def get_objects_for_user(self, request, permissions, queryset=None):
        return get_objects_for_user(
            request.user, permissions, klass=queryset,
            use_groups=True, any_perm=True,
            with_superuser=False, accept_global_perms=False,
        )

    def _filter_queryset(self, request, queryset, view):
        from events.surveyme.models import SurveyGroup
        if settings.IS_BUSINESS_SITE:
            user_qs = queryset.filter(org_id__isnull=True, user=request.user)
            if request.orgs:
                survey_qs = self.get_objects_for_user(
                    request, 'surveyme.change_survey',
                    queryset.filter(org__dir_id__in=request.orgs),
                )
                group_qs = self.get_objects_for_user(
                    request, 'surveyme.change_surveygroup',
                    SurveyGroup.objects.filter(org__dir_id__in=request.orgs),
                )
                queryset = user_qs | survey_qs | queryset.filter(group__in=group_qs)
            else:
                queryset = user_qs
        else:
            survey_qs = self.get_objects_for_user(request, 'surveyme.change_survey', queryset)
            group_qs = self.get_objects_for_user(request, 'surveyme.change_surveygroup')
            queryset = survey_qs | queryset.filter(group__in=group_qs)

        return queryset


class SurveyGroupFilterBackend(SurveyFilterBackend):
    def _filter_queryset(self, request, queryset, view):
        if settings.IS_BUSINESS_SITE:
            if request.orgs:
                queryset = self.get_objects_for_user(
                    request, 'surveyme.change_surveygroup',
                    queryset.filter(org__dir_id__in=request.orgs),
                )
            else:
                queryset = queryset.none()
        else:
            queryset = self.get_objects_for_user(request, 'surveyme.change_surveygroup', queryset)
        return queryset


class BaseFormFilterBackend(BaseFilterBackend):
    def filter_queryset(self, request, queryset, view):
        survey_id = request.query_params.get('survey')
        if not survey_id:
            return queryset.none()

        try:
            survey = Survey.objects.get(pk=survey_id)
        except Survey.DoesNotExist:
            return queryset.none()

        if not self._check_survey(request, view, survey):
            return queryset.none()

        return self._filter_queryset(queryset, survey_id)

    def _check_survey(self, request, view, survey):
        perm = HasChangePermission()
        return perm.has_object_permission(request, view, survey)

    def _filter_queryset(self, queryset, survey_id):
        return queryset.filter(survey=survey_id)


class SurveyQuestionFilterBackend(BaseFormFilterBackend):
    pass


class ProfileSurveyAnswerFilterBackend(BaseFormFilterBackend):
    pass


class SurveyTextFilterBackend(BaseFormFilterBackend):
    pass


class SurveyQuestionConditionNodeFilterBackend(BaseFormFilterBackend):
    pass


class SurveyQuestionConditionFilterBackend(BaseFormFilterBackend):
    def _filter_queryset(self, queryset, survey_id):
        return queryset.filter(survey_question__survey=survey_id)
