# -*- coding: utf-8 -*-
import binascii
import logging
import os
import re
import sys

from celery.exceptions import TimeoutError
from copy import copy
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db import transaction
from django.db.models import Q, F
from django.http import Http404, HttpResponse
from django.utils import timezone
from django.utils.translation import ugettext as _
from rest_framework import status, viewsets, serializers, views, generics, mixins
from rest_framework.decorators import action
from rest_framework.exceptions import PermissionDenied, ValidationError, AuthenticationFailed
from rest_framework.generics import get_object_or_404
from rest_framework.parsers import JSONParser, MultiPartParser
from rest_framework.response import Response
from rest_framework.settings import api_settings
from unidecode import unidecode
from urllib.parse import urlencode

from events.surveyme.api_admin.v2.filters import (
    ProfileSurveyAnswerFilter,
    SurveyFilter,
    SurveyGroupFilter,
    SurveyQuestionFilter,
    SurveyStateConditionNodeFilter,
    SurveyStyleTemplateFilter,
    SurveyTextFilter,
    UserProfileSurveyAnswerFilter,
    SurveyQuestionShowConditionNodeFilter,
    SurveySubmitConditionNodeFilter,
    SurveyFilterBackend,
    SurveyGroupFilterBackend,
    ProfileSurveyAnswerFilterBackend,
    SurveyQuestionConditionFilterBackend,
    SurveyQuestionConditionNodeFilterBackend,
    SurveyQuestionFilterBackend,
    SurveyTextFilterBackend,
)
from events.surveyme.serializers import AnswerResultsSerializer
from events.surveyme.api_admin.v2.serializers import (
    AnswerParamsSerializer,
    AnswerTypeSerializer,
    ExportAnswersSerializer,
    HideNotificationsSerializer,
    MoveSurveyQuestionPageSerializer,
    PositionSerializer,
    ProfileSurveyAnswerSerializer,
    SimpleProfileSurveyAnswerSerializer,
    SurveyAccessDeserializer,
    SurveyAccessSerializer,
    SurveyAgreementSerializer,
    SurveyDetailSerializer,
    SurveyGroupSerializer,
    SurveyListSerializer,
    SurveyShortSerializer,
    SurveyQuestionChoiceSerializer,
    SurveyQuestionCopySerializer,
    SurveyQuestionHintTypeSerializer,
    SurveyQuestionPageSerializer,
    SurveyQuestionRestore,
    SurveyQuestionSerializer,
    SurveyQuestionShowConditionNodeSerializer,
    SurveyStateConditionNodeSerializer,
    SurveyStyleTemplateSerializer,
    SurveySubmitConditionNodeSerializer,
    SurveyTemplateSerializer,
    SurveyTextSerializer,
    SurveyQuizSerializer,
    SurveyBanSerializer,
)
from events.surveyme.copiers import SurveyQuestionCopier, SurveyCopier
from events.surveyme.logic.access import (
    get_change_permission,
    get_survey_access,
    request_roles,
    set_survey_access,
)
from events.surveyme.logic.questions import move_page_questions, delete_page_questions
from events.surveyme.logic.query import is_true
from events.surveyme.models import (
    AnswerType,
    ProfileSurveyAnswer,
    Survey,
    SurveyAgreement,
    SurveyGroup,
    SurveyQuestion,
    SurveyQuestionChoice,
    SurveyQuestionHintType,
    SurveyQuestionShowConditionNode,
    SurveyQuestionShowConditionNodeItem,
    SurveyStateConditionNode,
    SurveyStyleTemplate,
    SurveySubmitConditionNode,
    SurveySubmitConditionNodeItem,
    SurveyTemplate,
    SurveyText,

    ANOTHER_GROUP_QUESTION_IN_LOGIC,
    NO_GROUP_QUESTIONS_IN_LOGIC,
)
from events.surveyme.preview import get_preview_class_type
from events.surveyme import tasks
from events.surveyme.utils import (
    check_survey_tickets_count_consistency,
)
from events.surveyme_integration.models import (
    HookSubscriptionNotification,
    HookSubscriptionNotificationCounter,
    StartrekSubscriptionData,
)

from events.rest_framework_contrib.pagination import (
    LargePagination,
    FetchPagination,
    OneItemPagination,
)
from events.rest_framework_contrib.permissions import (
    ANY,
    IsSuperUser,
    IsSupport,
    TakeoutClientOnly,
    AntispamClientsOnly,
    IsAuthenticated,
    HasChangePermission,
)
from events.rest_framework_contrib.parsers import FormParser
from events.rest_framework_contrib.mixins import (
    InternalGenericApiViewV2Mixin,
    InternalGenericApiViewV2MixinWithPermissions,
    ManyLookupFieldsMixin,
    TranslationViewMixin,
    DetailSerializerMixin,
)

from events.accounts.models import User, get_or_create_organization, UserKarmaError, UserKarmaWarn
from events.common_app.utils import get_language_or_default, get_tld
from events.data_sources.sources.external_table import DB_EXTERNAL_TABLE_SOURCES
from events.data_sources.tasks import sync_external_table_with_db
from events.followme.api_admin.v2.views import ContentFollowerMixin
from events.guardian_contrib.api_admin.v2.views_api import UsersWithPermissionsMixin
from events.history.models import HistoryRawEntry
from events.history.mixins import HistoryMixin
from events.surveyme_integration.api_admin.v2.serializers import HookSubscriptionNotificationShortSerializer
from events.tanker.tasks import run_translation_process, update_translations
from events.celery_app import app

logger = logging.getLogger(__name__)


class ExportNotReadyError(Exception):
    pass


class ExportResult(object):
    TASK_TIMEOUT = 5  # sec

    def __init__(self, instance):
        self._eager_results = getattr(instance, '_eager_results', {})

    def get(self, task_id):
        if settings.IS_TEST:
            return self._eager_results.get(task_id)
        task = app.AsyncResult(task_id)
        if not task.ready():
            raise ExportNotReadyError
        try:
            return task.get(timeout=self.TASK_TIMEOUT)
        except TimeoutError:
            logger.warn('Celery task %s timeout', task_id)


class SurveyStatsMixin(object):
    def rebuild_if_needed(self, survey):
        from events.countme.utils import rebuild_counters, check_if_need_rebuild_counters
        if check_if_need_rebuild_counters(survey):
            rebuild_counters(survey.pk)

    def to_integer(self, value, default):
        try:
            value = int(value)
            return value if value > 0 else default
        except (ValueError, TypeError):
            return default

    def get_pagination_params(self, request):
        return {
            'page': self.to_integer(request.query_params.get('page'), 1),
            'page_size': self.to_integer(request.query_params.get('page_size'), LargePagination.page_size),
        }

    def get_next_url(self, request, page=None, page_size=None):
        page = page or 1
        params = request.query_params.copy()
        params['page'] = page + 1
        if page_size:
            params['page_size'] = page_size
        return f'{request.path}?{urlencode(params)}'

    @action(url_path='stats', detail=True)
    def stats(self, request, *args, **kwargs):
        from events.countme.stats import get_stats_info
        survey = self.get_object()
        self.rebuild_if_needed(survey)
        params = self.get_pagination_params(request)
        data = get_stats_info(survey, **params)
        next_url = None
        if len(data['questions']) == params['page_size']:
            next_url = self.get_next_url(request, **params)
        data['links'] = {'next-url': next_url}
        return Response(data=data)

    @action(url_path='stats-detail', detail=True)
    def stats_detail(self, request, *args, **kwargs):
        return self.stats(request, *args, **kwargs)


class SurveyAccessMixin(object):
    def _set_access(self, obj, user):
        ct = ContentType.objects.get_for_model(obj)
        perm = get_change_permission(ct)
        request_roles(self.request.user, perm, users=[user], content_type=ct, object_pk=obj.pk)

    @action(methods=['get', 'post'], serializer_detail_class=SurveyAccessSerializer, detail=True)
    def access(self, request, *args, **kwargs):
        """
        /admin/api/v2/surveys/{id}/access
        формат ответа GET
        {
            "type": "common",  // "common" | "owner" | "restricted"
            "users": [  // список пользователей, только для type == "restricted"
                {
                    "uid": 100,
                    "login": "vpupkin",
                },
                ...
            ],
            "groups": [ 962 ],
            "role": "viewfile_survey"
        }

        формат тела запроса POST
        {
            "type": "common",  // "common" | "owner" | "restricted"
            "users": [  // список пользователей, только для type == "restricted"
                100,  // uid пользователя
                101,
                ...
            ],
            "groups": [ 962 ],
            "role": "viewfile_survey" // change_survey -- default
        }
        """
        survey_or_group = self.get_object()
        self.pre_save_check_permissions(survey_or_group)
        if request.method.lower() == 'post':
            deserializer = SurveyAccessDeserializer(data=request.data)
            if not deserializer.is_valid():
                return Response(deserializer.errors, status=status.HTTP_400_BAD_REQUEST)
            data = deserializer.validated_data
            set_survey_access(request.user, survey_or_group, data['type'], data.get('users'), data.get('groups'))
            HistoryRawEntry.objects.create_entry(survey_or_group)
            return Response()

        data = get_survey_access(survey_or_group)
        if 'type' in data:
            return Response(SurveyAccessSerializer(data).data)
        return Response({
            role: SurveyAccessSerializer(child).data
            for role, child in data.items()
        })


class SurveyGroupViewSet(
        HistoryMixin,
        SurveyAccessMixin,
        DetailSerializerMixin,
        InternalGenericApiViewV2MixinWithPermissions,
        viewsets.ModelViewSet):

    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [SurveyGroupFilterBackend]
    serializer_class = SurveyGroupSerializer
    serializer_detail_class = SurveyGroupSerializer
    queryset = (
        SurveyGroup.objects.all()
        .select_related('user')
    )
    check_permissions_on_create = False
    pagination_class = LargePagination
    filter_class = SurveyGroupFilter

    def perform_create(self, serializer):
        super().perform_create(serializer)
        surveygroup = serializer.instance
        self._set_access(surveygroup, self.request.user)
        surveygroup.user = self.request.user
        surveygroup.save()

    def perform_destroy(self, obj):
        tasks.reject_roles(self.request.user.pk, SurveyGroup, obj.pk, deleted_only=False)
        HistoryRawEntry.objects.create_entry(obj)
        obj.delete()

    def get_queryset(self, *args, **kwargs):
        queryset = super().get_queryset(*args, **kwargs)
        if self.request.user.is_superuser:
            return queryset
        if settings.IS_BUSINESS_SITE:
            user = self.request.user
            if self.request.orgs:
                queryset = queryset.filter(Q(org__dir_id__in=self.request.orgs) | Q(org_id__isnull=True, user=user))
            else:
                queryset = queryset.filter(org_id__isnull=True, user=user)
        return queryset

    @action(methods=['post'], url_path='change-surveys', detail=True)
    def add_surveys(self, request, *args, **kwargs):
        obj = self.get_object()
        surveys = request.data.get('surveys')
        if isinstance(surveys, str):
            surveys = surveys.split(',')

        action = request.data.get('action')
        if not (surveys and action):
            raise serializers.ValidationError({"detail": _('You should pass surveys and action in request')})

        with transaction.atomic():
            surveys_objects = Survey.objects.filter(pk__in=surveys).select_related('group')
            if action == 'remove':
                surveys_objects = surveys_objects.filter(group_id=obj.id)

            if action == 'add':
                obj.survey_set.add(*surveys_objects)
            elif action == 'remove':
                obj.survey_set.remove(*surveys_objects)
            else:
                raise serializers.ValidationError({"action": _('Unknown action: {}'.format(action))})
            Survey.objects.filter(pk__in=(s.pk for s in surveys_objects)).update(date_updated=timezone.now())

            for survey in surveys_objects:
                HistoryRawEntry.objects.create_entry(survey)

        return Response(data={'status': 'ok'})


class SaveNodeWithItemsMixin(object):
    node_item_model_class = None
    node_item_related_name_to_node = None

    def save_node_with_related_items(self, data):
        save_nodes_with_ids = []
        saved_related_objects_data = {}
        for i, node_data in enumerate(data.get('nodes', []), start=1):
            node = self.get_or_create_node(node_data, position=i)
            save_nodes_with_ids.append(node.id)
            for key, value in self.save_related_objects(node_data, node).items():
                if key not in saved_related_objects_data:
                    saved_related_objects_data[key] = []
                saved_related_objects_data[key] += value

        self.remove_unnecessary_nodes(data, save_nodes_with_ids)
        self.remove_unnecessary_related_objects(data, saved_related_objects_data)
        return {
            'save_nodes_with_ids': save_nodes_with_ids,
            'saved_related_objects': saved_related_objects_data
        }

    def get_root_model_instance(self, data):
        raise NotImplementedError

    def save_related_objects(self, node_data, node):
        return {
            'items': self.save_node_items(node_data, node)
        }

    def save_node_items(self, node_data, node):
        save_node_items_with_ids = []
        for j, node_item_data in enumerate(node_data['items'], start=1):
            if j == 1:
                node_item_data['operator'] = 'and'
            node_item_data[self.node_item_related_name_to_node] = node
            node_item = self.get_or_create_node_item(node_item_data, position=j)
            save_node_items_with_ids.append(node_item.id)
        return save_node_items_with_ids

    def get_or_create_node(self, node_data, position):
        node = self.get_model()(**self.prepare_node_data(node_data, position))
        node.save()
        return node

    def prepare_node_data(self, node_data, position):
        return {
            'survey_question_id': node_data.get('survey_question'),
            'id': node_data.get('id') or None,
            'position': position
        }

    def get_or_create_node_item(self, node_item_data, position):
        prepared_node_item_data = copy(node_item_data)
        prepared_node_item_data['survey_question_id'] = prepared_node_item_data.get('survey_question')
        prepared_node_item_data['content_type_attribute_id'] = prepared_node_item_data.get('content_type_attribute')
        prepared_node_item_data['position'] = position
        for key in ['survey_question', 'survey_question_choice', 'content_type_attribute']:
            prepared_node_item_data.pop(key, None)
        for key in list(prepared_node_item_data.keys()):
            if key.startswith('_'):
                del prepared_node_item_data[key]
        node_item = self.node_item_model_class(**prepared_node_item_data)
        node_item.save()
        return node_item

    def remove_unnecessary_related_objects(self, data, saved_objects_data):
        self.remove_unnecessary_node_items(data, saved_objects_data.get('items', []))

    def remove_unnecessary_nodes(self, data, save_nodes_with_ids):
        self.get_queryset_nodes_for_remove(data, save_nodes_with_ids).delete()

    def remove_unnecessary_node_items(self, data, save_node_items_with_ids):
        self.get_queryset_node_items_for_remove(data, save_node_items_with_ids).delete()

    def get_queryset_nodes_for_remove(self, data, save_nodes_with_ids):
        raise NotImplementedError

    def get_queryset_node_items_for_remove(self, data, save_node_items_with_ids):
        raise NotImplementedError


class PermissionDeniedDetailed(PermissionDenied):
    def __init__(self, obj, *args, **kwargs):
        super(PermissionDeniedDetailed, self).__init__(*args, **kwargs)
        self.detail = {
            'detail': self.detail,
        }
        access_data = get_survey_access(obj)
        perm = get_change_permission(obj)
        if perm.codename in access_data:
            access_data = access_data[perm.codename]
        if access_data.get('type') == 'restricted':
            self.detail['users'] = [
                {
                    'uid': user.uid,
                    'cloud_uid': user.cloud_uid,
                    'display_name': user.get_name_and_surname_with_fallback(),
                    'email': user.email,
                }
                for user in access_data.get('users') or []
            ]
        if obj.user:
            self.detail.update({
                'author': {
                    'uid': obj.user.uid,
                    'cloud_uid': obj.user.cloud_uid,
                    'display_name': obj.user.get_name_and_surname_with_fallback(),
                    'email': obj.user.email,
                },
            })


class SurveyMixin:
    def get_object(self):
        # В отличии от стандартного кода не применяем метод filter_queryset
        queryset = self.get_queryset()

        # Perform the lookup filtering.
        lookup_url_kwarg = self.lookup_url_kwarg or self.lookup_field

        assert lookup_url_kwarg in self.kwargs, (
            'Expected view %s to be called with a URL keyword argument '
            'named "%s". Fix your URL conf, or set the `.lookup_field` '
            'attribute on the view correctly.' %
            (self.__class__.__name__, lookup_url_kwarg)
        )

        filter_kwargs = {self.lookup_field: self.kwargs[lookup_url_kwarg]}
        obj = get_object_or_404(queryset, **filter_kwargs)
        # May raise a permission denied
        try:
            self.check_object_permissions(self.request, obj)
        except PermissionDenied:
            raise PermissionDeniedDetailed(obj)
        return obj

    def get_queryset(self, *args, **kwargs):
        queryset = super().get_queryset(*args, **kwargs)
        if self.request.user.is_superuser:
            return queryset
        if settings.IS_BUSINESS_SITE:
            user = self.request.user
            if self.request.orgs:
                queryset = queryset.filter(Q(org__dir_id__in=self.request.orgs) | Q(org_id__isnull=True, user=user))
            else:
                queryset = queryset.filter(org_id__isnull=True, user=user)
        return queryset


class SurveyViewSet(TranslationViewMixin,
                    SurveyMixin,
                    SurveyStatsMixin,
                    ContentFollowerMixin,
                    HistoryMixin,
                    SurveyAccessMixin,
                    SaveNodeWithItemsMixin,
                    UsersWithPermissionsMixin,
                    DetailSerializerMixin,
                    InternalGenericApiViewV2MixinWithPermissions,
                    viewsets.ModelViewSet):
    node_item_model_class = SurveySubmitConditionNodeItem
    node_item_related_name_to_node = 'survey_submit_condition_node'
    serializer_class = SurveyShortSerializer
    serializer_detail_class = SurveyDetailSerializer
    serializer_list_class = SurveyListSerializer
    queryset = (
        Survey.objects.all()
        .select_related(
            'org',
            'answercount',
            'user',
            'group__user',
            'group__org',
        )
    )
    queryset_detail = (
        queryset.select_related('content_type', 'styles_template', 'org', )
        .prefetch_related(
            'agreements',
            'surveystateconditionnode_set__items',
            'hooks__items',
            'hooks__triggers',
            'hooks__subscriptions__headers',
            'hooks__subscriptions__questions',
            'hooks__subscriptions__json_rpc',
            'hooks__subscriptions__surveyvariable_set',
            'hooks__subscriptions__json_rpc__params',
        )
    )
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [SurveyFilterBackend]
    filter_class = SurveyFilter
    parser_classes = (JSONParser, FormParser, MultiPartParser)
    check_permissions_on_create = False
    _eager_results = {}

    def get_serializer_class(self):
        if self.action == 'list':
            return self.serializer_list_class
        elif self._is_request_to_detail_endpoint():
            return self.serializer_detail_class
        return self.serializer_class

    def _is_request_to_detail_endpoint(self):
        if self.request.query_params.get('detailed', '').lower() in ('0', 'false', 'no'):
            return False
        return super(SurveyViewSet, self)._is_request_to_detail_endpoint()

    def pre_save_check_permissions(self, obj):
        self._check_permission_to_change_group(obj)
        if not obj.pk and not obj.group_id:
            return
        super(SurveyViewSet, self).pre_save_check_permissions(obj)

    def _check_permission_to_change_group(self, obj):
        original_obj = self.get_original_object(obj)
        if obj.pk and obj.group_id != original_obj.group_id:
            perm = 'surveyme.change_surveygroup'
            if not self.request.user.has_perm(perm, obj.group):
                raise PermissionDenied

    def perform_create(self, serializer):
        current_language = get_language_or_default()
        serializer.validated_data['language'] = current_language
        super().perform_create(serializer)
        survey = serializer.instance
        user = self.request.user

        if settings.IS_BUSINESS_SITE:
            survey.is_published_external = True
            survey.date_published = timezone.now()

        self._set_access(survey, user)
        survey.user = user
        survey.save()

        # добавляем создателя формы в наблюдатели
        user.follow(survey)

    def perform_update(self, serializer):
        save_logs_for_statbox = serializer.instance.save_logs_for_statbox
        super(SurveyViewSet, self).perform_update(serializer)
        obj = serializer.instance

        update_fields = ['date_updated']
        if 'is_published_external' in serializer.validated_data:
            if serializer.validated_data['is_published_external']:
                obj.date_published = timezone.now()
                obj.date_unpublished = None
                update_fields += ['date_published', 'date_unpublished']
            else:
                obj.date_unpublished = timezone.now()
                update_fields += ['date_unpublished']
        obj.save(update_fields=update_fields)

        if obj.save_logs_for_statbox != save_logs_for_statbox:
            if obj.save_logs_for_statbox:
                obj.prepare_answers_to_export()
            else:
                obj.remove_answers_from_export()

        # проверяем нобходимость создания билетов для резервирования (FORMS-493)
        check_survey_tickets_count_consistency(obj)

    def get_serializer(self, *args, **kwargs):
        serializer_class = self.get_serializer_class()
        kwargs['context'] = self.get_serializer_context()
        kwargs['context']['survey'] = kwargs.pop('survey', None)
        return serializer_class(*args, **kwargs)

    def update(self, request, *args, **kwargs):
        partial = kwargs.pop('partial', False)
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data, partial=partial, survey=instance)
        serializer.is_valid(raise_exception=True)

        if 'is_published_external' in serializer.validated_data:
            if serializer.validated_data['is_published_external']:
                try:
                    request.user.check_karma()
                except (UserKarmaError, UserKarmaWarn) as e:
                    result = {'code': e.code, 'detail': _('Плохая карма, публикация форм запрещена')}
                    return Response(result, status=status.HTTP_400_BAD_REQUEST)
        self.perform_update(serializer)

        if getattr(instance, '_prefetched_objects_cache', None):
            # If 'prefetch_related' has been applied to a queryset, we need to
            # forcibly invalidate the prefetch cache on the instance.
            instance._prefetched_objects_cache = {}

        return Response(serializer.data)

    @action(url_path='tickets', detail=True)
    def tickets_info(self, request, *args, **kwargs):
        data = self.get_object().get_tickets_status()
        if data:
            return Response(data=data)
        else:
            raise Http404()

    @action(url_path='translations', detail=True)
    def get_translations(self, request, *args, **kwargs):
        from events.tanker.utils import get_translation_status
        obj = self.get_object()
        translation_status = get_translation_status(obj.pk)
        if not translation_status:
            raise Http404()
        return Response(data=translation_status)

    @action(methods=['post'], url_path='restore', detail=True)
    def restore(self, request, *args, **kwargs):
        queryset = Survey.with_deleted_objects.get_queryset()
        obj = queryset.get(**kwargs)
        if obj.user_id != request.user.pk:
            self.check_object_custom_permissions(
                request,
                obj,
                [IsAuthenticated, ANY(IsSuperUser, HasChangePermission, IsSupport)],
            )
        obj.is_deleted = False
        update_fields = ['is_deleted']
        if obj.slug:
            obj.slug = re.sub(r'-[\da-f]{6}$', '', obj.slug)
            update_fields.append('slug')
        obj.save(update_fields=update_fields)

        HistoryRawEntry.objects.create_entry(obj)

        args = (request.user.pk, Survey, obj.pk)
        tasks.request_roles.apply_async(args)

        return Response({'status': 'ok'})

    @action(methods=['post'], url_path='run-translation-process', detail=True)
    def run_translation_process(self, request, *args, **kwargs):
        languages = self.request.data.get('languages')
        if not languages:
            return Response({'error_messages': 'languages required'}, status=status.HTTP_400_BAD_REQUEST)

        obj = self.get_object()
        task = run_translation_process.delay(obj.pk, languages)
        data = {'task_id': task.task_id}
        return Response(data, status=status.HTTP_202_ACCEPTED)

    @action(methods=['post'], url_path='check-translations', detail=True)
    def check_translations(self, request, *args, **kwargs):
        obj = self.get_object()
        task = update_translations.delay(obj.pk)
        data = {'task_id': task.task_id}
        return Response(data, status=status.HTTP_202_ACCEPTED)

    @action(methods=['get'], url_path='questions-count', detail=True)
    def questions_count(self, request, *args, **kwargs):
        obj = self.get_object()
        return Response({'questions_count': obj.questions_count})

    @transaction.atomic
    @action(methods=['post'], url_path='copy', detail=True)
    def create_copy_of_form(self, request, *args, **kwargs):
        old_survey = self.get_object()
        self.pre_save_check_permissions(old_survey)
        if old_survey.is_ban_detected:
            return Response(_('Нельзя скопировать забаненную форму'), status=status.HTTP_400_BAD_REQUEST)

        user = self.request.user
        new_survey = SurveyCopier(old_survey).copy(user.pk)

        self._set_access(new_survey, user)

        HistoryRawEntry.objects.create_entry(new_survey)

        data = {'survey_id': new_survey.pk}
        return Response(data)

    @action(methods=['post'], url_path='import', detail=False)
    def import_survey(self, request, *args, **kwargs):
        from events.surveyme.survey_importer import SurveyImporter
        user = self.request.user
        user_karma_warn = False
        try:
            user.check_karma()
        except UserKarmaError as e:
            result = {'code': e.code, 'detail': str(e)}
            return Response(result, status=status.HTTP_400_BAD_REQUEST)
        except UserKarmaWarn as e:
            user_karma_warn = True
            logger.warn('%s', e)

        try:
            survey_data = self._get_survey_data(request)
        except SurveyTemplate.DoesNotExist:
            return Response(
                'No template with such pk exists',
                status=status.HTTP_400_BAD_REQUEST
            )
        dir_id = None
        if 'org_id' in request.data:
            dir_id = request.data['org_id']

        with transaction.atomic():
            survey = SurveyImporter(survey_data).import_survey(user_id=user.pk, dir_id=dir_id)
            self._set_access(survey, user)

        if settings.IS_BUSINESS_SITE:
            if 'import_from_template' not in request.data and not user_karma_warn:
                survey.is_published_external = True
                survey.date_published = timezone.now()
                survey.save(update_fields=['is_published_external', 'date_published'])

        HistoryRawEntry.objects.create_entry(survey)

        data = {
            'status': 'success',
            'result': {
                'survey': {
                    'id': survey.pk
                }
            }
        }
        return Response(data)

    @action(methods=['post'], url_path='import-tracker', detail=False)
    def import_tracker_template(self, request, *args, **kwargs):
        from events.surveyme.survey_importer import SurveyImporter

        if 'import_from_template' not in request.data:
            raise ValidationError({'import_from_template': [_('This field is required.')]})
        if settings.IS_BUSINESS_SITE and 'org_id' not in request.data:
            raise ValidationError({'org_id': [_('This field is required.')]})

        dir_id = request.data.get('org_id')
        try:
            survey_data = self._get_survey_data(request)
        except SurveyTemplate.DoesNotExist:
            raise ValidationError({'import_from_template': [_('Not found.')]})

        if settings.IS_BUSINESS_SITE:
            org = get_or_create_organization(dir_id)
            group = org.o2g.group
            if request.user not in group.user_set.all():
                raise AuthenticationFailed

        with transaction.atomic():
            survey = SurveyImporter(survey_data).import_survey(request.user.pk, dir_id=dir_id)
            self._set_access(survey, request.user)

        fields = []
        is_published_external = request.data.get('is_published_external', True)
        if is_published_external:
            survey.is_published_external = True
            fields.append('is_published_external')

        if 'name' in request.data:
            survey.name = request.data['name']
            survey.set_translated_field('name', survey.name)
            fields.append('name')
            fields.append('translations')

        if 'queue' in request.data:
            startrek_subscriptions = StartrekSubscriptionData.objects.filter(
                subscription__survey_hook__survey_id=survey.pk,
                subscription__service_type_action__service_type__slug='startrek',
            )
            startrek_subscriptions.update(queue=request.data['queue'])

        if fields:
            survey.save(update_fields=fields)
        HistoryRawEntry.objects.create_entry(survey)

        return Response({
            'status': 'success',
            'result': {'survey': {'id': survey.pk}},
        })

    def _get_survey_data(self, request):
        import_from_template = request.data.get('import_from_template')
        if import_from_template:
            params = {}
            if isinstance(import_from_template, int) or import_from_template.isdigit():
                params['pk'] = import_from_template
            else:
                params['slug'] = import_from_template
            template = SurveyTemplate.objects.get(**params)
            return template.data
        else:
            return request.data

    def _prepare_import_result(self, survey, request):
        return {
            'survey': SurveyDetailSerializer(instance=survey, context={'request': request}).data,
            'questions': [
                SurveyQuestionSerializer(instance=question, context={'request': request}).data
                for question in survey.surveyquestion_set.select_related('answer_type')
            ]
        }

    @transaction.atomic
    @action(methods=['post'], url_path='move-page', detail=True)
    def move_page(self, request, *args, **kwargs):
        serializer = MoveSurveyQuestionPageSerializer(data=request.data)
        if serializer.is_valid():
            survey = self.get_object()
            page_from = serializer.validated_data.get('page')
            page_to = serializer.validated_data.get('to')

            try:
                move_page_questions(survey, page_from, page_to)
            except ValidationError as e:
                return Response(str(e), status=status.HTTP_400_BAD_REQUEST)

            return Response('')
        else:
            return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @transaction.atomic
    @action(methods=['delete'], url_path='page', detail=True)
    def delete_page(self, request, *args, **kwargs):
        serializer = SurveyQuestionPageSerializer(data=request.data)
        if serializer.is_valid():
            survey = self.get_object()
            page = serializer.validated_data.get('page')

            try:
                delete_page_questions(survey, page)
            except ValidationError as e:
                return Response(str(e), status=status.HTTP_400_BAD_REQUEST)

            return Response('')
        else:
            return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @action(methods=['post'], url_path='hide-notifications', detail=True)
    def hide_notifications(self, request, *args, **kwargs):
        survey = self.get_object()
        serializer = HideNotificationsSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        subscription = serializer.data.get('subscription')
        with transaction.atomic():
            queryset = (
                HookSubscriptionNotification.objects.all()
                .filter(
                    survey=survey,
                    status='error',
                    is_visible=True,
                )
            )
            counters_queryset = HookSubscriptionNotificationCounter.objects.filter(survey=survey)
            if subscription:
                queryset = queryset.filter(subscription=subscription)
                counters_queryset = counters_queryset.filter(subscription=subscription)
            result = queryset.update(is_visible=False)
            counters_queryset.update(errors_count=0, date_updated=timezone.now())

        data = {'status': 'success', 'count': result}
        return Response(data)

    def _save_submit_conditions(self, survey, data, meta_data):
        for node in data.get('nodes', []):
            node = self._prepare_node_for_import(node, survey)
            node['items'] = [self._prepare_node_item_for_import(item, meta_data) for item in node.get('items', [])]
        self.save_node_with_related_items(data)

    def _prepare_node_for_import(self, node, survey):
        node['survey_id'] = survey.pk
        return self._remove_fields_from_item(node, ('id', 'pk'))

    def _prepare_node_item_for_import(self, node_item, meta_data):
        node_item['survey_question'] = meta_data.get('questions_map', {}).get(node_item.get('survey_question'))
        node_item['survey_question_choice'] = meta_data.get('choices_map', {}).get(node_item.get('survey_question_choice'))
        if ('%s' % node_item.get('value')).isdigit():
            new_value = meta_data.get('choices_map', {}).get(int(node_item.get('value')))
            if new_value:
                node_item['value'] = str(new_value)
        return self._remove_fields_from_item(node_item, ('id', 'pk', 'survey_submit_condition_node'))

    def _remove_fields_from_item(self, item, fields):
        for field in [f for f in fields if f in item]:
            del item[field]
        return item

    def get_model(self):
        return SurveySubmitConditionNode

    def prepare_node_data(self, node_data, position):
        return {
            'id': node_data.get('id') or None,
            'position': position,
            'survey_id': node_data['survey_id'],
        }

    def get_queryset_nodes_for_remove(self, data, save_nodes_with_ids):
        return SurveySubmitConditionNode.objects.none()

    def get_queryset_node_items_for_remove(self, data, save_node_items_with_ids):
        return SurveySubmitConditionNodeItem.objects.none()

    def perform_destroy(self, obj):
        self.pre_delete_check_permissions(obj)
        # скрываем форму вместо того чтобы удалить
        obj.is_deleted = True
        if obj.slug:
            unique_code = binascii.hexlify(os.urandom(3))
            obj.slug = '%s-%s' % (obj.slug, unique_code.decode('utf-8'))
        obj.save()

        HistoryRawEntry.objects.create_entry(obj)
        obj.remove_answers_from_export()

        args = (self.request.user.pk, Survey, obj.pk)
        tasks.reject_roles.apply_async(args, countdown=settings.REJECT_ROLES_COUNTDOWN)

    def get_int_array(self, data):
        for item in data.split(','):
            value = item.strip()
            if value.isdigit():
                yield int(value)

    @action(methods=['post'], url_path='export', detail=True)
    def export(self, request, *args, **kwargs):
        survey = self.get_object()
        serializer = ExportAnswersSerializer(data=request.data)

        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data
        export_columns = data.get('export_columns', {})

        columns = export_columns.get('answer_fields', [])
        columns.extend(export_columns.get('user_fields', []))

        tld = get_tld(request.headers.get('referer'))

        task_data = {
            'user_uid': str(request.user.uid),
            'survey_id': survey.pk,
            'answers_pks': data.get('pks'),
            'questions_pks': export_columns.get('questions', []),
            'columns': columns,
            'format': data.get('export_format', 'xlsx'),
            'upload': data.get('upload', 'mds'),
            'upload_files': data.get('upload_files', False),
            'started_at': data.get('date_started'),
            'finished_at': data.get('date_finished'),
            'limit': data.get('limit'),
            'tld': tld,
        }

        logger.info('run export answers %s', task_data)
        task = tasks.export_survey_answers.delay(**task_data)
        if settings.IS_TEST:
            self._eager_results[task.id] = task.get()
        response = {
            'task_id': task.id
        }

        survey.date_exported = timezone.now()
        survey.save(update_fields=['date_exported'])

        return Response(response, status=status.HTTP_202_ACCEPTED)

    @action(url_path='export-results', detail=True)
    def export_results(self, request, *args, **kwargs):
        from events.surveyme.export_answers import MDSDownloader
        survey = self.get_object()
        task_id = request.query_params.get('task_id')
        task = ExportResult(self)
        try:
            task_data = task.get(task_id)
        except ExportNotReadyError:
            headers = {
                'Retry-After': '5',
                'Content-Location': request.build_absolute_uri(),
            }
            return Response({}, headers=headers, status=status.HTTP_202_ACCEPTED)

        # провряем что данные экспортированны из текущей формы
        if not task_data or task_data.get('survey_id') != survey.pk:
            raise Http404

        content_type = task_data.get('content_type')
        status_code = task_data.get('status_code', status.HTTP_200_OK)
        response = HttpResponse(content_type=content_type, status=status_code)

        if status_code == status.HTTP_200_OK:
            content_disposition = 'attachment; filename="{file_name}"'.format(
                file_name=unidecode(task_data['file_name']),
            )
            response['Content-Disposition'] = content_disposition
            response.write(MDSDownloader().download(task_data['path']))
        elif status_code == status.HTTP_302_FOUND:
            response['Location'] = task_data['path']

        return response

    @action(methods=['post'], url_path='change-style', detail=True)
    def change_style(self, request, *args, **kwargs):
        obj = self.get_object()
        template_id = request.data.get('template_id', False)
        if template_id is False:
            raise serializers.ValidationError({"detail": _('В запросе отсутствует template_id')})
        elif not template_id:
            if obj.styles_template_id:
                obj.styles_template = None
                obj.save()
        else:
            try:
                styles_template = SurveyStyleTemplate.objects.get(pk=template_id)
            except SurveyStyleTemplate.DoesNotExist:
                raise serializers.ValidationError({"detail": _('Шаблона с указанным template_id не существует')})
            else:
                if styles_template != obj.styles_template:
                    obj.styles_template = styles_template
                    obj.save()

        HistoryRawEntry.objects.create_entry(obj)

        return Response(data={'status': 'ok'})

    @action(methods=['get'], url_path='notifications', detail=True)
    def notifications(self, request, *args, **kwargs):
        survey = self.get_object()
        queryset = (
            HookSubscriptionNotification.objects.select_related('user')
            .filter(status='error', is_visible=True, survey=survey)
            .defer(
                'celery_task_id', 'context', 'date_finished',
                'date_next_processing', 'date_retry', 'date_updated',
                'is_visible', 'max_retries', 'notification_id',
                'response', 'retries', 'status', 'survey_group',
                'trigger_data', 'trigger_slug',
                'user__password', 'user__last_login', 'user__is_superuser',
                'user__first_name', 'user__last_name', 'user__email',
                'user__is_staff', 'user__is_active', 'user__date_joined',
            )
            .order_by('-date_created')
        )

        paginator = FetchPagination()
        page = paginator.paginate_queryset(queryset, request)
        serializer = HookSubscriptionNotificationShortSerializer(page, many=True)
        return paginator.get_paginated_response(serializer.data)

    @action(methods=['get'], url_path='answers', detail=True)
    def answers(self, request, *args, **kwargs):
        survey = self.get_object()
        queryset = (
            ProfileSurveyAnswer.objects.select_related('survey')
            .filter(survey=survey)
            .order_by('-pk')
        )
        params = AnswerParamsSerializer(data=request.query_params)
        params.is_valid(raise_exception=True)

        if 'started' in params.validated_data:
            queryset = queryset.filter(date_created__gte=params.validated_data['started'])
        if 'finished' in params.validated_data:
            queryset = queryset.filter(date_created__lte=params.validated_data['finished'])

        survey.date_exported = timezone.now()
        survey.save(update_fields=['date_exported'])

        paginator = OneItemPagination()
        page = paginator.paginate_queryset(queryset, request)
        context = {'request': request, 'view': self}
        serializer = AnswerResultsSerializer(page, many=True, context=context)
        return paginator.get_paginated_response(serializer.data)


class SurveyAgreementViewSet(InternalGenericApiViewV2Mixin, viewsets.ReadOnlyModelViewSet):
    serializer_class = SurveyAgreementSerializer
    queryset = SurveyAgreement.objects.all()


class SurveyTemplateViewSet(InternalGenericApiViewV2Mixin, viewsets.ReadOnlyModelViewSet):
    serializer_class = SurveyTemplateSerializer
    permission_classes = []
    queryset = (
        SurveyTemplate.objects.all()
        .filter(is_personal=False)
        .select_related(
            'image',
            'image_admin',
        )
    )


class AnswerTypeViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = AnswerTypeSerializer
    queryset = (
        AnswerType.objects.all()
        .filter(app_types=settings.APP_TYPE)
        .prefetch_related(
            'allow_settings',
            'required_settings',
            'validator_types',
        )
    )


class SurveyQuestionChoiceViewSet(TranslationViewMixin,
                                  HistoryMixin,
                                  InternalGenericApiViewV2Mixin,
                                  viewsets.ModelViewSet):
    serializer_class = SurveyQuestionChoiceSerializer
    queryset = SurveyQuestionChoice.objects.all().select_related('label_image')


class BadRequest(Response):
    status_code = status.HTTP_400_BAD_REQUEST


class SurveyQuestionViewSet(TranslationViewMixin,
                            HistoryMixin,
                            InternalGenericApiViewV2MixinWithPermissions,
                            viewsets.ModelViewSet):
    serializer_class = SurveyQuestionSerializer
    queryset = (
        SurveyQuestion.objects.all()
        .select_related(
            'label_image',
            'validator_type',
        )
        .prefetch_related(
            'answer_type__allow_settings',
            'answer_type__required_settings',
            'answer_type__validator_types',
            'surveyquestionchoice_set',
            'surveyquestionmatrixtitle_set',
            'show_condition_nodes',
            'surveyquestionchoice_set__label_image',
        )
    )
    filter_class = SurveyQuestionFilter
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [SurveyQuestionFilterBackend]
    pagination_class = LargePagination

    def get_object(self):
        # В отличии от стандартного кода не применяем метод filter_queryset
        queryset = self.get_queryset()

        # Perform the lookup filtering.
        lookup_url_kwarg = self.lookup_url_kwarg or self.lookup_field

        assert lookup_url_kwarg in self.kwargs, (
            'Expected view %s to be called with a URL keyword argument '
            'named "%s". Fix your URL conf, or set the `.lookup_field` '
            'attribute on the view correctly.' %
            (self.__class__.__name__, lookup_url_kwarg)
        )

        filter_kwargs = {self.lookup_field: self.kwargs[lookup_url_kwarg]}
        obj = get_object_or_404(queryset, **filter_kwargs)

        # May raise a permission denied
        self.check_object_permissions(self.request, obj.survey)

        return obj

    @action(methods=['post'], url_path='copy', detail=True)
    def copy_question(self, request, *args, **kwargs):
        old_question = self.get_object()
        if old_question.answer_type.slug == 'answer_payment':
            return BadRequest({'errors': _('answer_payment_field_must_be_single')})

        copier = SurveyQuestionCopier()
        with transaction.atomic():
            instance = copier.copy(old_question)[0]
        new_question = self.get_queryset().get(pk=instance.pk)
        data = SurveyQuestionCopySerializer(new_question).data

        HistoryRawEntry.objects.create_entry(instance)

        return Response(data, status=status.HTTP_201_CREATED)

    @action(methods=['post'], url_path='move-to-position', detail=True)
    def move_to_position(self, request, *args, **kwargs):
        # todo: test me
        question = self.get_object()
        position_serializer = PositionSerializer(data=request.data)
        if position_serializer.is_valid():
            try:
                page = position_serializer.validated_data.get('page')
                position = position_serializer.validated_data.get('position')
                question.change_position(position, page=page, group_id=question.group_id)
            except ValidationError as e:
                return BadRequest(e.detail)
            else:
                return Response('')
        else:
            return BadRequest(position_serializer.errors)

    def perform_create(self, serializer):
        page = serializer.validated_data.pop('page', 1)
        position = serializer.validated_data.pop('position', sys.maxsize)
        group_id = serializer.validated_data.pop('group_id', None)

        super(SurveyQuestionViewSet, self).perform_create(serializer)
        obj = serializer.instance
        obj.change_position(position, page=page, group_id=group_id)

        external_table_sync = DB_EXTERNAL_TABLE_SOURCES.get(obj.param_data_source)
        if external_table_sync:
            sync_external_table_with_db(external_table_sync.name, obj.param_data_source_params)

    def perform_update(self, serializer):
        obj = self.get_object()
        page = serializer.validated_data.pop('page', obj.page)
        position = serializer.validated_data.pop('position', obj.position)
        group_id = serializer.validated_data.pop('group_id', obj.group_id)

        super(SurveyQuestionViewSet, self).perform_update(serializer)
        new_obj = serializer.instance
        new_obj.change_position(position, page=page, group_id=group_id)

        external_table_sync = DB_EXTERNAL_TABLE_SOURCES.get(new_obj.param_data_source)
        if external_table_sync:
            sync_external_table_with_db(external_table_sync.name, new_obj.param_data_source_params)

    def perform_destroy(self, obj):
        self.pre_delete_check_permissions(obj)
        survey = obj.survey
        HistoryRawEntry.objects.create_entry(obj)
        if obj.answer_type.kind == 'profile':
            # удаляем профильные вопросы
            obj.delete()
        else:
            # скрываем вопрос вместо того чтобы удалить
            obj.is_deleted = True
            obj.param_slug = '%s_%s' % (obj.answer_type.slug, obj.pk)
            update_fields = ['is_deleted', 'param_slug']
            obj.save(update_fields=update_fields)
            obj.group_questions.update(is_deleted=True)
        survey.sync_questions()

    @action(methods=['post'], url_path='preview', detail=True)
    def preview(self, request, *args, **kwargs):
        question = self.get_object()
        field_class = get_preview_class_type(question.answer_type.slug)
        field = field_class(question, **request.data)
        try:
            data = field.as_dict()
            status_code = status.HTTP_200_OK
        except ValidationError as exc:
            data = {'error_message': str(exc)}
            status_code = status.HTTP_400_BAD_REQUEST
        if not data:
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
        return Response(data=data, status=status_code)

    def _move_to_new_page(self, request, page, survey, obj):
        new_page = is_true(request.query_params.get('new_page', ''))
        if new_page:
            survey.surveyquestion_set.filter(
                page__gte=page
            ).exclude(pk=obj.pk).update(page=F('page') + 1)

    @action(methods=['post'], url_path='restore', detail=True)
    def restore(self, request, *args, **kwargs):
        """
        Восстанновление вопроса
        """
        serializer = SurveyQuestionRestore(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.data

        queryset = SurveyQuestion.with_deleted_objects.get_queryset()
        obj = queryset.get(**kwargs)
        survey = obj.survey
        if survey.user_id != request.user.pk:
            self.check_object_custom_permissions(
                request,
                survey,
                [IsAuthenticated, ANY(IsSuperUser, HasChangePermission, IsSupport)],
            )
        survey = obj.survey
        update_fields = []

        obj.is_deleted = False
        update_fields.append('is_deleted')
        self._move_to_new_page(
            request=request, page=data.get('page', obj.page),
            survey=survey, obj=obj,
        )
        survey.surveyquestion_set.filter(
            page=data.get('page', obj.page),
            position__gte=data.get('position', obj.position)
        ).update(position=F('position') + 1)
        if 'page' in data:
            obj.page = data['page']
            update_fields.append('page')
        if 'position' in data:
            obj.position = data['position']
            update_fields.append('position')

        obj.save(update_fields=update_fields)
        obj.group_questions.update(is_deleted=False)
        survey.sync_questions()

        HistoryRawEntry.objects.create_entry(obj)

        return Response({'status': 'ok'})


class ProfileSurveyAnswerViewSet(InternalGenericApiViewV2MixinWithPermissions,
                                 viewsets.ModelViewSet):
    serializer_class = ProfileSurveyAnswerSerializer
    queryset = (
        ProfileSurveyAnswer.objects.all()
        .select_related(
            'user',
        )
        .prefetch_related(
            'keys__bundle',
        )
    )
    filter_class = ProfileSurveyAnswerFilter
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [ProfileSurveyAnswerFilterBackend]
    ordering = ('date_created', )


class SurveyTextViewSet(TranslationViewMixin,
                        HistoryMixin,
                        InternalGenericApiViewV2MixinWithPermissions,
                        viewsets.ModelViewSet):
    serializer_class = SurveyTextSerializer
    queryset = SurveyText.objects.all()
    filter_class = SurveyTextFilter
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [SurveyTextFilterBackend]
    pagination_class = LargePagination

    def perform_update(self, serializer):
        obj = serializer.instance
        if obj.value != serializer.validated_data.get('value'):
            super().perform_update(serializer)


class SurveyStateConditionNodeViewSet(InternalGenericApiViewV2MixinWithPermissions, viewsets.ModelViewSet):
    serializer_class = SurveyStateConditionNodeSerializer
    queryset = SurveyStateConditionNode.objects.all()
    filter_class = SurveyStateConditionNodeFilter

# triggers


class SurveyConditionNodeBaseViewsSet(InternalGenericApiViewV2MixinWithPermissions,
                                      SaveNodeWithItemsMixin,
                                      viewsets.ModelViewSet):
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [SurveyQuestionConditionNodeFilterBackend]

    def get_model(self):
        return getattr(self, 'model', self.queryset.model)

    @action(methods=['post'], url_path='save-with-items', detail=False)
    def save_with_items(self, request, *args, **kwargs):
        root_instance = self.get_root_model_instance(request.data)
        self._check_permissions_for_request(root_instance)
        self._validate_data(request.data)
        saved_nodes_data = self.save_node_with_related_items(request.data)

        HistoryRawEntry.objects.create_entry(root_instance)

        return Response(saved_nodes_data)

    def _validate_data(self, data):
        pass

    def get_root_model_instance(self, data):
        raise NotImplementedError


class SurveyQuestionShowConditionNodeViewSet(SurveyConditionNodeBaseViewsSet):
    # todo: doc me
    node_item_model_class = SurveyQuestionShowConditionNodeItem
    serializer_class = SurveyQuestionShowConditionNodeSerializer
    queryset = (
        SurveyQuestionShowConditionNode.objects.all()
        .filter(items__survey_question__is_deleted=False)
        .prefetch_related('items')
        .distinct()
    )
    node_item_related_name_to_node = 'survey_question_show_condition_node'
    filter_class = SurveyQuestionShowConditionNodeFilter
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [SurveyQuestionConditionFilterBackend]

    def _validate_data(self, data):
        questions_ids = {
            item.get('survey_question') for node in data['nodes']
            for item in node['items']
        }
        questions_obj = SurveyQuestion.objects.select_related('answer_type').filter(pk__in=questions_ids)
        group_id = self.instance.group_id

        for question in questions_obj:
            if question.answer_type.slug == 'answer_group':
                raise serializers.ValidationError({'error_message': [NO_GROUP_QUESTIONS_IN_LOGIC]})
            elif group_id and question.group_id and group_id != question.group_id:
                raise serializers.ValidationError({'error_message': [ANOTHER_GROUP_QUESTION_IN_LOGIC]})

    def get_root_model_instance(self, data):
        self.instance = SurveyQuestion.objects.get(pk=data.get('survey_question'))
        return self.instance

    def get_queryset_nodes_for_remove(self, data, save_nodes_with_ids):
        survey_question_id = int(data['survey_question'])
        return (self.get_model().objects
                                .filter(survey_question_id=survey_question_id)
                                .exclude(id__in=save_nodes_with_ids))

    def get_queryset_node_items_for_remove(self, data, save_node_items_with_ids):
        survey_question_id = int(data['survey_question'])
        return (self.node_item_model_class.objects
                    .filter(**{self.node_item_related_name_to_node + '__survey_question_id': survey_question_id})  # todo: test me
                    .exclude(id__in=save_node_items_with_ids))


class UserProfileSurveyAnswersViewSet(InternalGenericApiViewV2MixinWithPermissions, viewsets.ReadOnlyModelViewSet):
    serializer_class = SimpleProfileSurveyAnswerSerializer
    queryset = ProfileSurveyAnswer.objects.prefetch_related('survey__content_object')
    filter_class = UserProfileSurveyAnswerFilter
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [ProfileSurveyAnswerFilterBackend]


class SurveySubmitConditionNodeViewSet(SurveyConditionNodeBaseViewsSet):
    # todo: doc me
    node_item_model_class = SurveySubmitConditionNodeItem
    serializer_class = SurveySubmitConditionNodeSerializer
    queryset = (
        SurveySubmitConditionNode.objects.all()
        .filter(items__survey_question__is_deleted=False)
        .prefetch_related('items')
        .distinct()
    )
    node_item_related_name_to_node = 'survey_submit_condition_node'
    filter_class = SurveySubmitConditionNodeFilter

    def get_root_model_instance(self, data):
        return Survey.objects.get(pk=data.get('survey_id'))

    def get_queryset_nodes_for_remove(self, data, save_nodes_with_ids):
        survey_id = data['survey_id']
        return (self.get_model().objects
                                .filter(survey_id=survey_id)
                                .exclude(id__in=save_nodes_with_ids))

    def get_queryset_node_items_for_remove(self, data, save_node_items_with_ids):
        survey_id = data['survey_id']
        return (self.node_item_model_class.objects
                    .filter(**{self.node_item_related_name_to_node + '__survey_id': survey_id})  # todo: test me
                    .exclude(id__in=save_node_items_with_ids))

    def prepare_node_data(self, node_data, position):
        return {
            'survey_id': node_data.get('survey_id'),
            'id': node_data.get('id') or None,
            'position': position
        }


class SurveyQuestionHintTypeViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = SurveyQuestionHintTypeSerializer
    queryset = SurveyQuestionHintType.objects.all()


class TakeoutView(InternalGenericApiViewV2Mixin, views.APIView):
    permission_classes = [ANY(IsSuperUser, TakeoutClientOnly)]
    parser_classes = (JSONParser, FormParser)

    def _create_task(self, uid):
        try:
            profile = User.objects.get(uid=uid)
            task = tasks.takeout.delay(profile.pk)
            return {
                'status': 'ok',
                'job_id': task.id,
            }
        except User.DoesNotExist:
            return None

    def _poll_task(self, task_id):
        task = app.AsyncResult(task_id)
        if task:
            if task.ready():
                task_data = task.get(task_id)
                file_link = 'https://{domain}/v1/files/?path={path}'.format(
                    domain=settings.DOMAIN_NAME,
                    path=task_data.get('path'),
                )
                return {
                    'status': 'ok',
                    'file_links': [file_link],
                }
            elif task.status == 'SENT':
                return {
                    'status': 'pending',
                }

    def post(self, request, *args, **kwargs):
        logger.info('takeout request %s', request.data)
        data = None
        try:
            if 'job_id' in self.request.data:
                task_id = self.request.data.get('job_id')
                data = self._poll_task(task_id)
            elif 'uid' in self.request.data:
                uid = self.request.data.get('uid')
                data = self._create_task(uid)

            if not data:
                data = {
                    'status': 'no_data',
                }
        except Exception as e:
            logger.exception('takeout error %s', str(e))
            data = {
                'status': 'error',
                'error': str(e),
            }

        if isinstance(data, HttpResponse):
            return data
        return Response(data, status=status.HTTP_200_OK)


class SurveyStyleTemplateViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = SurveyStyleTemplateSerializer
    queryset = SurveyStyleTemplate.objects.all()
    filter_class = SurveyStyleTemplateFilter


class SurveyQuizView(SurveyMixin,
                     HistoryMixin,
                     InternalGenericApiViewV2MixinWithPermissions,
                     generics.RetrieveUpdateAPIView):
    queryset = (
        Survey.objects.all()
        .prefetch_related(
            'surveyquestion_set',
        )
    )
    serializer_class = SurveyQuizSerializer


class SurveyBanView(InternalGenericApiViewV2Mixin,
                    mixins.RetrieveModelMixin,
                    mixins.UpdateModelMixin,
                    generics.GenericAPIView):
    permission_classes = [ANY(IsSuperUser, AntispamClientsOnly, IsSupport)]
    queryset = Survey.objects.all()
    serializer_class = SurveyBanSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        obj = self.get_object()
        HistoryRawEntry.objects.create_entry(obj)
        return self.partial_update(request, *args, **kwargs)


class AnswerResultsView(ManyLookupFieldsMixin,
                        InternalGenericApiViewV2MixinWithPermissions,
                        generics.RetrieveAPIView):
    lookup_fields = ('pk', 'secret_code')
    queryset = ProfileSurveyAnswer.objects.select_related('survey').all()
    serializer_class = AnswerResultsSerializer
