# -*- coding: utf-8 -*-
from django.conf import settings
from django.db.models import Q

from events.accounts.utils import is_anonymous
from events.common_app.conditions import not_
from events.common_app.utils import add_model_prefix as add
from events.search.utils import ORMFullTextSearchConstructor


class SurveyConditions(object):
    @classmethod
    def is_profile_can_answer(cls, user=None, model_instance=None, ignore=None, prefix=None, preview=False):
        """
        if profile is None - then anonymous
        """
        # todo: test ignore
        if is_anonymous(user):
            user = None
        kwargs = {
            'user': user,
            'model_instance': model_instance,
            'prefix': prefix,
        }
        kwargs_without_profile = {
            'model_instance': model_instance,
            'prefix': prefix,
            'preview': preview,
        }
        if not ignore:
            ignore = []

        if preview and not user:
            return False

        condition = (
            not_(cls.is_already_answered_by_profile(**kwargs)) |
            cls.is_answer_could_be_edited(**kwargs_without_profile) |
            cls.is_answer_could_be_made_multiple_times(**kwargs_without_profile)
        )
        if 'is_ready_for_answer' not in ignore:
            condition = cls.is_ready_for_answer(**kwargs_without_profile) & condition
        if not user:
            condition = condition & not_(cls.is_need_auth_to_answer(**kwargs_without_profile))
        return condition

    @classmethod
    def is_ready_for_answer(cls, model_instance=None, prefix=None, preview=False):
        return cls.is_published_external(model_instance=model_instance, prefix=prefix, preview=preview)

    @classmethod
    def is_published_external(cls, model_instance=None, prefix=None, preview=False):
        if preview:
            return True
        elif settings.SURVEYME_SHOW_UNPUBLISHED_FORMS_ON_INTERNAL_SITE:  # TECH-3079
            return cls._is_published_for_current_site_type(model_instance=model_instance, prefix=prefix)
        else:
            if model_instance:
                return model_instance.is_published_external
            else:
                return Q(**{add(prefix, to='is_published_external'): True})

    @classmethod
    def _is_published_for_current_site_type(cls, model_instance=None, prefix=None):
        if model_instance:
            return (settings.IS_EXTERNAL_SITE and model_instance.is_published_external) or settings.IS_INTERNAL_SITE
        else:
            if settings.IS_INTERNAL_SITE:
                condition = Q()
            else:
                condition = Q(**{add(prefix, to='is_published_external'): True})
            return condition

    @classmethod
    def is_already_answered_by_profile(cls, user, model_instance=None, prefix=None):
        if model_instance:
            if getattr(user, 'pk', None) and not is_anonymous(user) and (model_instance.is_allow_answer_editing or not model_instance.is_allow_multiple_answers):
                # нам не нужен предыдущий пользовательский ответ,
                # если в настройках нет флага разрешать редактировать или
                # стоит запрет на создание новых ответов на форму
                return model_instance.profilesurveyanswer_set.filter(user=user).exists()
            else:
                return False
        else:
            return Q(**{add(prefix, to='profilesurveyanswer__user'): user})

    @classmethod
    def is_answer_could_be_edited(cls, model_instance=None, prefix=None, **kwargs):
        if model_instance:
            return model_instance.is_allow_answer_editing
        else:
            return Q(**{add(prefix, to='is_allow_answer_editing'): True})

    @classmethod
    def is_answer_could_be_made_multiple_times(cla, model_instance=None, prefix=None, **kwargs):
        if model_instance:
            return model_instance.is_allow_multiple_answers
        else:
            return Q(**{add(prefix, to='is_allow_multiple_answers'): True})

    @classmethod
    def is_has_questions(cls, model_instance=None, prefix=None):
        """Необходимо сделать .distinct() перед фильтрацией по условию"""
        if model_instance:
            return bool(model_instance.surveyquestion_set.count())
        else:
            return Q(**{add(prefix, to='surveyquestion__pk__isnull'): False})

    @classmethod
    def is_need_auth_to_answer(cls, model_instance=None, prefix=None, **kwargs):
        # todo: tesme
        if model_instance:
            return model_instance.need_auth_to_answer
        else:
            return Q(**{add(prefix, to='need_auth_to_answer'): True})

    @classmethod
    def is_fit_for_search_text(cls, search_text, model_instance=None, prefix=None):
        """

        @type model_instance: events.surveyme.models.Survey

        """
        if model_instance:
            raise NotImplementedError()
        else:
            condition = ORMFullTextSearchConstructor(
                prefix=prefix,
                not_localizable_params=[
                    add(prefix, to='name'),
                    add(prefix, to='slug'),
                    add(prefix, to='pk'),
                ],
                is_full_text_search=True
            ).get_q_objects_for_text(search_text=search_text)

        return condition
