# -*- coding: utf-8 -*-
from bson.objectid import ObjectId
from django.conf import settings
from django.db.models import Max, Q
from events.common_app.utils import disable_signals, get_language_or_default
from django.utils import timezone
from django.utils.translation import ugettext as _
from events.countme.models import AnswerCount
from events.surveyme.models import (
    Survey,
    SurveySubmitConditionNode,
    SurveySubmitConditionNodeItem,
    SurveyQuestion,
    SurveyQuestionChoice,
    SurveyQuestionMatrixTitle,
    SurveyQuestionShowConditionNode,
    SurveyQuestionShowConditionNodeItem,
    SurveyStyleTemplate,
    SurveyText,
)
from events.surveyme_integration.models import (
    IntegrationFileTemplate,
    JSONRPCSubscriptionData,
    JSONRPCSubscriptionParam,
    ServiceSurveyHookSubscription,
    StartrekSubscriptionData,
    SubscriptionAttachment,
    SubscriptionHeader,
    SurveyHook,
    SurveyHookCondition,
    SurveyHookConditionNode,
    SurveyVariable,
    WikiSubscriptionData,
)
from events.surveyme.sequences import get_sequence
from events.surveyme_integration.utils import macros_re

ANSWER_TYPE_ID_GROUP = (40, 1040)
ANSWER_TYPE_ID_CHOICE = 3


class SurveyQuestionCopier(object):
    def __init__(self, survey_id=None, label=None):
        self.survey_id = survey_id
        self.label = label
        self.sequence = get_sequence(
            SurveyQuestion,
            SurveyQuestionChoice,
            SurveyQuestionMatrixTitle,
            SurveyQuestionShowConditionNode,
            SurveyQuestionShowConditionNodeItem,
        )

    def copy(self, *questions):
        if questions:
            first_question = questions[0]
            is_group = False
            if len(questions) == 1 and first_question.answer_type_id in ANSWER_TYPE_ID_GROUP:
                is_group = True
                questions = list(
                    SurveyQuestion.objects.filter(
                        Q(pk=first_question.pk)
                        | Q(group_id=first_question.pk)
                    )
                    .select_related('answer_type')
                )
            return self._bulk_copy_questions(questions, is_group)

    def _bulk_copy_questions(self, questions, is_group):
        self.questions_map = {}
        self.choices_map = {}
        max_positions = {}
        lang = get_language_or_default()
        questions_seq = self.sequence(SurveyQuestion, len(questions))
        for question in questions:
            old_question_pk = question.pk
            question.pk = questions_seq.popleft()
            self.questions_map[old_question_pk] = question.pk
            if question.answer_type.kind == 'generic':
                if not self.survey_id:
                    # при копировании вопросов, код формы не передается и нужно менять слаги,
                    # чтобы новые не совпадали со старыми
                    question.param_slug = '%s_%s' % (question.answer_type.slug, question.pk)
            self._patch_question(old_question_pk, question)

            if self.survey_id:
                question.survey_id = self.survey_id
            elif question.group_id and is_group:
                pass
            else:
                params = tuple([
                    question.survey_id,
                    question.page,
                    question.group_id,
                ])
                if params in max_positions:
                    max_position = max_positions[params]
                else:
                    filter_params = dict(zip(['survey_id', 'page', 'group_id'], params))
                    max_position = SurveyQuestion.objects.filter(**filter_params).aggregate(Max('position')).get('position__max')
                    max_positions[params] = max_position
                if max_position:
                    question.position = max_position + 1
                    max_positions[params] += 1

                new_label = self.label
                if not new_label:
                    new_label = _('Копия "%(label)s"') % {'label': question.get_label()}
                question.set_translated_field('label', new_label)
                if question.survey.language == lang:
                    question.label = new_label

        for question in questions:
            if question.group_id:
                if self.questions_map.get(question.group_id):
                    question.group_id = self.questions_map[question.group_id]

        SurveyQuestion.objects.bulk_create(questions)

        self._create_choices()
        self._create_titles()
        self._create_nodes()
        self._create_items()

        return [
            question
            for question in questions
            if not is_group or question.group_id is None
        ]

    def _patch_question(self, old_question_pk, question):
        # патчим код вопроса в фильтре источника данных
        param_data_source_params = getattr(question, 'param_data_source_params', None)
        if (
            param_data_source_params and
            isinstance(param_data_source_params.get('filters'), list)
        ):
            tmp_filters = []
            for param in param_data_source_params['filters']:
                if (
                    param.get('filter') and
                    param['filter'].get('name') == 'question' and
                    param.get('value') == old_question_pk
                ):
                    param['value'] = question.pk
                tmp_filters.append(param)
            param_data_source_params['filters'] = tmp_filters
            question.param_data_source_params = param_data_source_params
        if question.answer_type.slug == 'answer_group':
            question.param_is_required = False

    def _create_choices(self):
        choices = SurveyQuestionChoice.objects.filter(survey_question_id__in=self.questions_map.keys())
        if choices:
            choices_seq = self.sequence(SurveyQuestionChoice, len(choices))
            for choice in choices:
                old_choice_pk = choice.pk
                choice.pk = choices_seq.pop()
                choice.survey_question_id = self.questions_map[choice.survey_question_id]
                self.choices_map[old_choice_pk] = choice.pk
        SurveyQuestionChoice.objects.bulk_create(choices)

    def _create_titles(self):
        titles = SurveyQuestionMatrixTitle.objects.filter(survey_question_id__in=self.questions_map.keys())
        if titles:
            titles_seq = self.sequence(SurveyQuestionMatrixTitle, len(titles))
            for title in titles:
                title.pk = titles_seq.pop()
                title.survey_question_id = self.questions_map[title.survey_question_id]
        SurveyQuestionMatrixTitle.objects.bulk_create(titles)

    def _create_nodes(self):
        nodes = SurveyQuestionShowConditionNode.objects.filter(survey_question_id__in=self.questions_map.keys())
        self.nodes_map = {}
        if nodes:
            nodes_seq = self.sequence(SurveyQuestionShowConditionNode, len(nodes))
            for node in nodes:
                old_node_pk = node.pk
                node.pk = nodes_seq.popleft()
                node.survey_question_id = self.questions_map[node.survey_question_id]
                self.nodes_map[old_node_pk] = node.pk
        SurveyQuestionShowConditionNode.objects.bulk_create(nodes)

    def _create_items(self):
        items = SurveyQuestionShowConditionNodeItem.objects.filter(survey_question_show_condition_node__survey_question_id__in=self.questions_map.keys())
        if items:
            items_seq = self.sequence(SurveyQuestionShowConditionNodeItem, len(items))
            for item in items:
                item.pk = items_seq.pop()
                item.survey_question_show_condition_node_id = self.nodes_map[item.survey_question_show_condition_node_id]
                if self.questions_map.get(item.survey_question_id):
                    item.survey_question_id = self.questions_map[item.survey_question_id]
                if item.value and item.value.isdigit() and self.choices_map.get(int(item.value)):
                    item.value = self.choices_map[int(item.value)]
        SurveyQuestionShowConditionNodeItem.objects.bulk_create(items)


class SurveyCopier(object):
    def __init__(self, survey):
        self.survey = survey
        self.sequence = get_sequence(
            IntegrationFileTemplate,
            JSONRPCSubscriptionData,
            JSONRPCSubscriptionParam,
            ServiceSurveyHookSubscription,
            StartrekSubscriptionData,
            SubscriptionAttachment,
            SubscriptionHeader,
            Survey,
            SurveyHook,
            SurveyHookConditionNode,
            SurveyHookCondition,
            SurveySubmitConditionNode,
            SurveySubmitConditionNodeItem,
            SurveyText,
            WikiSubscriptionData,
        )

    def copy(self, user_id):
        old_survey_pk = self.survey.pk
        with disable_signals():
            survey = self._copy_survey(self.survey, user_id)
        self._create_texts(old_survey_pk, survey)
        self._create_questions(old_survey_pk, survey)
        self._create_attachment_templates(old_survey_pk, survey)
        self._create_hooks(old_survey_pk, survey)
        self._create_counters(survey)
        nodes_map = self._create_submit_condition_nodes(old_survey_pk, survey)
        self._create_submit_condition_node_items(old_survey_pk, nodes_map)
        return survey

    def _copy_survey(self, survey, user_id):
        survey.pk = self._get_survey_pk()
        survey.user_id = user_id
        new_name = u'Копия "%s"' % survey.name
        survey.name = new_name
        if settings.IS_BUSINESS_SITE:
            survey.is_published_external = True
            survey.date_published = timezone.now()
        else:
            survey.is_published_external = False
            survey.date_published = None
        survey.date_unpublished = None
        survey.date_archived = None
        survey.date_exported = None
        survey.slug = None
        survey.styles_template = self._get_or_create_styles_template(survey)
        survey.save()
        return survey

    def _get_survey_pk(self):
        if settings.USE_HEX_IDENTIFIER_FIELD:
            return str(ObjectId())
        else:
            return self.sequence(Survey, 1).pop()

    def _get_or_create_styles_template(self, survey):
        styles_template = survey.styles_template
        if styles_template and styles_template.type == 'custom':
            styles_template = SurveyStyleTemplate.objects.create(
                name=styles_template.name,
                type=styles_template.type,
                styles=styles_template.styles,
            )
        return styles_template

    def _create_texts(self, old_survey_pk, survey):
        texts = SurveyText.objects.filter(survey_id=old_survey_pk)
        if texts:
            texts_seq = self.sequence(SurveyText, texts.count())
            for text in texts:
                text.pk = texts_seq.pop()
                text.survey_id = survey.pk
            SurveyText.objects.bulk_create(texts)

    def _create_questions(self, old_survey_pk, survey):
        questions = (
            SurveyQuestion.objects.all()
            .filter(survey_id=old_survey_pk)
            .select_related('answer_type')
            .order_by('pk')
        )
        if questions:
            copier = SurveyQuestionCopier(survey_id=survey.pk)
            copier.copy(*questions)
            self.questions_map = copier.questions_map
            self.choices_map = copier.choices_map

    def _create_attachment_templates(self, old_survey_pk, survey):
        templates = IntegrationFileTemplate.objects.filter(survey_id=old_survey_pk)
        if templates:
            templates_seq = self.sequence(IntegrationFileTemplate, templates.count())
            for template in templates:
                template.pk = templates_seq.pop()
                template.survey_id = survey.pk
            IntegrationFileTemplate.objects.bulk_create(templates)

    def _create_hooks(self, old_survey_pk, survey):
        hooks = SurveyHook.objects.filter(survey_id=old_survey_pk).prefetch_related('triggers')
        self.hooks_map = {}
        if hooks:
            triggers_map = {}
            hooks_seq = self.sequence(SurveyHook, hooks.count())
            for hook in hooks:
                old_hook_pk = hook.pk
                triggers = list(trigger.pk for trigger in hook.triggers.all())
                hook.pk = hooks_seq.popleft()
                triggers_map[hook.pk] = triggers
                self.hooks_map[old_hook_pk] = hook.pk
                hook.survey_id = survey.pk
            SurveyHook.objects.bulk_create(hooks)
            SurveyHookTriggers = SurveyHook.triggers.through
            relations_to_create = [
                SurveyHookTriggers(surveyhook_id=hook_id, surveyhooktrigger_id=trigger_id)
                for hook_id, triggers in triggers_map.items()
                for trigger_id in triggers
            ]
            SurveyHookTriggers.objects.bulk_create(relations_to_create)

            self._create_subscriptions(old_survey_pk, survey)
            nodes_map = self._create_hook_conditions_nodes(old_survey_pk)
            self._create_hook_conditions_node_items(old_survey_pk, nodes_map)

    def _create_subscriptions(self, old_survey_pk, survey):
        subscriptions = ServiceSurveyHookSubscription.objects.filter(survey_hook__survey_id=old_survey_pk).prefetch_related('questions')
        if subscriptions:
            subscriptions_map = {}
            subscription_questions = {}
            subscriptions_seq = self.sequence(ServiceSurveyHookSubscription, subscriptions.count())
            for subscription in subscriptions:
                old_service_type_action_id = subscription.service_type_action_id
                old_subscription_id = subscription.pk
                questions_list = subscription.questions.all()
                subscription.pk = subscriptions_seq.popleft()
                subscriptions_map[old_subscription_id] = subscription.pk
                subscription_questions[subscription.pk] = questions_list
                subscription.service_type_action_id = old_service_type_action_id
                subscription.survey_hook_id = self.hooks_map[subscription.survey_hook_id]
                if settings.APP_TYPE != 'forms_int':
                    # при копировании включаем проверку на спам
                    subscription.email_spam_check = True
                if settings.IS_BUSINESS_SITE:
                    subscription.email_from_address = f'{survey.pk}@forms-mailer.yaconnect.com'

            variables = self._create_variables(old_survey_pk, subscriptions_map)

            self._patch_subscriptions_pre_save(subscriptions)
            ServiceSurveyHookSubscription.objects.bulk_create(subscriptions)
            self._patch_subscriptions_post_save(subscriptions, subscription_questions)

            SurveyVariable.objects.bulk_create(variables)
            self._create_attachments(old_survey_pk, subscriptions_map)
            self._create_integrations(old_survey_pk, subscriptions_map)
            self._create_headers(old_survey_pk, subscriptions_map)

    def _create_variables(self, old_survey_pk, subscriptions_map):
        variables = SurveyVariable.objects.filter(hook_subscription__survey_hook__survey_id=old_survey_pk)
        self.variables_map = {}
        if variables:
            for variable in variables:
                old_variable_id = variable.variable_id
                variable.variable_id = str(ObjectId())
                self.variables_map[old_variable_id] = variable.variable_id
                if variable.hook_subscription_id:
                    variable.hook_subscription_id = subscriptions_map[variable.hook_subscription_id]
                variable.arguments = variable.arguments or {}
                if variable.arguments.get('question'):
                    variable.arguments['question'] = self.questions_map.get(variable.arguments.get('question'))
                elif variable.arguments.get('questions'):
                    if variable.arguments.get('is_all_questions') is True:
                        variable.arguments['questions'] = []
                    else:
                        variable.arguments['questions'] = [
                            self.questions_map[question]
                            for question in variable.arguments['questions']
                            if question in self.questions_map
                        ]
        return variables

    def _get_new_value(self, field_value):
        if not field_value:
            return field_value
        variable_ids = set(macros_re.findall(field_value))
        for variable_id in variable_ids:
            new_variable_id = self.variables_map.get(variable_id)
            if new_variable_id:
                field_value = field_value.replace(variable_id, new_variable_id)
        return field_value

    def _update_variables(self, obj, field):
        field_value = getattr(obj, field, None)
        new_field_value = self._get_new_value(field_value)
        if not new_field_value:
            return
        setattr(obj, field, new_field_value)

    def _patch_subscriptions_pre_save(self, subscriptions):
        fields = (
            'title',
            'body',
            'email_to_address',
            'email_from_address',
            'email_from_title',
            'http_url',
            'method',
        )
        for subscription in subscriptions:
            for value in fields:
                self._update_variables(subscription, value)

    def _patch_subscriptions_post_save(self, subscriptions, subscription_questions):
        ServiceSurveyHookSubscriptionQuestions = ServiceSurveyHookSubscription.questions.through
        relations_to_create = [
            ServiceSurveyHookSubscriptionQuestions(
                servicesurveyhooksubscription_id=subscription.pk,
                surveyquestion_id=self.questions_map.get(question.pk),
            )
            for subscription in subscriptions
            for question in subscription_questions[subscription.pk]
            if question.pk in self.questions_map
        ]
        ServiceSurveyHookSubscriptionQuestions.objects.bulk_create(relations_to_create)

    def _create_attachments(self, old_survey_pk, subscriptions_map):
        attachments = SubscriptionAttachment.objects.filter(subscription__survey_hook__survey_id=old_survey_pk)
        if attachments:
            attachments_seq = self.sequence(SubscriptionAttachment, attachments.count())
            for attachment in attachments:
                attachment.pk = attachments_seq.pop()
                attachment.subscription_id = subscriptions_map[attachment.subscription_id]
            SubscriptionAttachment.objects.bulk_create(attachments)

    def _create_integrations(self, old_survey_pk, subscriptions_map):
        rpc_map = self._create_json_rpc_subscription(old_survey_pk, subscriptions_map)
        self._create_json_rpc_params(old_survey_pk, rpc_map)
        self._create_startrek_subscription(old_survey_pk, subscriptions_map)
        self._create_wiki_subscription(old_survey_pk, subscriptions_map)

    def _create_json_rpc_subscription(self, old_survey_pk, subscriptions_map):
        rpc_subscriptions = JSONRPCSubscriptionData.objects.filter(subscription__survey_hook__survey_id=old_survey_pk)
        rpc_map = {}
        if rpc_subscriptions:
            rpc_subscriptions_seq = self.sequence(JSONRPCSubscriptionData, rpc_subscriptions.count())
            for subscription in rpc_subscriptions:
                old_subscription_pk = subscription.pk
                subscription.pk = rpc_subscriptions_seq.pop()
                rpc_map[old_subscription_pk] = subscription.pk
                subscription.subscription_id = subscriptions_map[subscription.subscription_id]
            self._patch_subscriptions_pre_save(rpc_subscriptions)
            JSONRPCSubscriptionData.objects.bulk_create(rpc_subscriptions)
        return rpc_map

    def _create_json_rpc_params(self, old_survey_pk, rpc_map):
        rpc_params = JSONRPCSubscriptionParam.objects.filter(subscription__subscription__survey_hook__survey_id=old_survey_pk)
        if rpc_params:
            rpc_params_seq = self.sequence(JSONRPCSubscriptionParam, rpc_params.count())
            for param in rpc_params:
                param.pk = rpc_params_seq.pop()
                param.subscription_id = rpc_map[param.subscription_id]
                for value in ('name', 'value'):
                    self._update_variables(param, value)
            JSONRPCSubscriptionParam.objects.bulk_create(rpc_params)

    def _create_startrek_subscription(self, old_survey_pk, subscriptions_map):
        startrek_subscriptions = StartrekSubscriptionData.objects.filter(subscription__survey_hook__survey_id=old_survey_pk)
        startrek_map = {}
        if startrek_subscriptions:
            startrek_subscriptions_seq = self.sequence(StartrekSubscriptionData, startrek_subscriptions.count())
            for subscription in startrek_subscriptions:
                old_subscription_pk = subscription.pk
                subscription.pk = startrek_subscriptions_seq.pop()
                startrek_map[old_subscription_pk] = subscription.pk
                subscription.subscription_id = subscriptions_map[subscription.subscription_id]
                for value in ('author', 'assignee'):
                    self._update_variables(subscription, value)
                if subscription.fields:
                    for field in subscription.fields:
                        value = field.get('value') or ''
                        field['value'] = self._get_new_value(value)
            StartrekSubscriptionData.objects.bulk_create(startrek_subscriptions)
        return startrek_map

    def _create_wiki_subscription(self, old_survey_pk, subscriptions_map):
        wiki_subscriptions = WikiSubscriptionData.objects.filter(subscription__survey_hook__survey_id=old_survey_pk)
        if wiki_subscriptions:
            wiki_subscriptions_seq = self.sequence(WikiSubscriptionData, wiki_subscriptions.count())
            for subscription in wiki_subscriptions:
                subscription.pk = wiki_subscriptions_seq.pop()
                subscription.subscription_id = subscriptions_map[subscription.subscription_id]
                for value in ('supertag', 'text'):
                    self._update_variables(subscription, value)
            WikiSubscriptionData.objects.bulk_create(wiki_subscriptions)

    def _create_headers(self, old_survey_pk, subscriptions_map):
        headers = SubscriptionHeader.objects.filter(subscription__survey_hook__survey_id=old_survey_pk)
        if headers:
            wiki_subscriptions_seq = self.sequence(SubscriptionHeader, headers.count())
            for header in headers:
                header.pk = wiki_subscriptions_seq.pop()
                header.subscription_id = subscriptions_map[header.subscription_id]
                for value in ('value', 'name'):
                    self._update_variables(header, value)
            SubscriptionHeader.objects.bulk_create(headers)

    def _create_hook_conditions_nodes(self, old_survey_pk):
        nodes = SurveyHookConditionNode.objects.filter(hook__survey_id=old_survey_pk)
        nodes_map = {}
        if nodes:
            nodes_seq = self.sequence(SurveyHookConditionNode, nodes.count())
            for node in nodes:
                old_node_pk = node.pk
                node.pk = nodes_seq.pop()
                nodes_map[old_node_pk] = node.pk
                node.hook_id = self.hooks_map[node.hook_id]
            SurveyHookConditionNode.objects.bulk_create(nodes)
        return nodes_map

    def _create_hook_conditions_node_items(self, old_survey_pk, nodes_map):
        items = SurveyHookCondition.objects.filter(condition_node__hook__survey_id=old_survey_pk)
        if items:
            items_seq = self.sequence(SurveyHookCondition, items.count())
            for item in items:
                item.pk = items_seq.pop()
                item.condition_node_id = nodes_map[item.condition_node_id]
                item.survey_question_id = self.questions_map.get(item.survey_question_id)
                if item.value and item.value.isdigit() and self.choices_map.get(int(item.value)):
                    item.value = self.choices_map[int(item.value)]
            SurveyHookCondition.objects.bulk_create(items)

    def _create_submit_condition_nodes(self, old_survey_pk, survey):
        nodes = SurveySubmitConditionNode.objects.filter(survey_id=old_survey_pk)
        nodes_map = {}
        if nodes:
            nodes_seq = self.sequence(SurveySubmitConditionNode, nodes.count())
            for node in nodes:
                old_node_pk = node.pk
                node.pk = nodes_seq.pop()
                nodes_map[old_node_pk] = node.pk
                node.survey_id = survey.pk
            SurveySubmitConditionNode.objects.bulk_create(nodes)
        return nodes_map

    def _create_submit_condition_node_items(self, old_survey_pk, nodes_map):
        items = SurveySubmitConditionNodeItem.objects.filter(survey_submit_condition_node__survey_id=old_survey_pk)
        if items:
            items_seq = self.sequence(SurveySubmitConditionNodeItem, items.count())
            for item in items:
                item.pk = items_seq.pop()
                item.survey_submit_condition_node_id = nodes_map[item.survey_submit_condition_node_id]
                if self.questions_map.get(item.survey_question_id):
                    item.survey_question_id = self.questions_map[item.survey_question_id]
                if item.value and item.value.isdigit() and self.choices_map.get(int(item.value)):
                    item.value = self.choices_map[int(item.value)]
            SurveySubmitConditionNodeItem.objects.bulk_create(items)

    def _create_counters(self, survey):
        AnswerCount.objects.create(survey=survey)
