# -*- coding: utf-8 -*-
import csv
import os

from django.db import connection
from io import StringIO
from uuid import uuid1

from events.common_app.utils import slugify_filename
from events.common_storages.storage import MdsClient


class SqlFetcher(object):
    fetch_num = 1000
    sql = ''

    def pre_execute(self, cursor, *args, **kwargs):
        pass

    def execute(self, *args, **kwargs):
        pks = kwargs.pop('pks')
        date_started = kwargs.get('date_started')
        date_finished = kwargs.get('date_finished')
        kwargs['pks'] = ','.join(str(pk) for pk in pks or [0])
        kwargs['all_answers'] = self.bool_to_dbbool(not pks)
        kwargs['ignore_date_started'] = self.bool_to_dbbool(not date_started)
        kwargs['ignore_date_finished'] = self.bool_to_dbbool(not date_finished)
        c = connection.cursor()
        self.pre_execute(c, *args, **kwargs)
        sql = self.sql.format(*args, **kwargs)
        try:
            sql_args = list(args) + [date_started, date_finished]
            c.execute(sql, sql_args)
            while True:
                rs = c.fetchmany(self.fetch_num)
                if not rs:
                    break
                yield rs
        finally:
            c.close()

    def bool_to_dbbool(self, value):
        if connection.vendor == 'sqlite':
            return '1' if value else '0'
        return 'true' if value else 'false'


class XlsxExporter(object):
    def __init__(self, stream):
        self.stream = stream
        self.wb, self.ws = None, None

    def __enter__(self):
        from openpyxl import Workbook
        self.wb = Workbook(write_only=True)
        self.ws = self.wb.create_sheet()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        from zipfile import ZipFile, ZIP_DEFLATED
        from openpyxl.writer.excel import ExcelWriter

        with ZipFile(self.stream, 'w', ZIP_DEFLATED, allowZip64=True) as archive:
            writer = ExcelWriter(self.wb, archive)
            writer.write_data()

    def writeheader(self, values):
        from openpyxl.styles import Font
        row = self.ws.row_dimensions[1]
        row.font = Font(name=row.font.name, bold=True)
        self.ws.append(values)

    def writerow(self, values):
        self.ws.append(values)


class UnicodeWriter(object):
    """
    A CSV writer which will write rows to CSV file "f",
    which is encoded in the given encoding.
    """

    def __init__(self, f, dialect=csv.excel, **kwds):
        # Redirect output to a queue
        self.queue = StringIO()
        self.writer = csv.writer(self.queue, dialect=dialect, **kwds)
        self.stream = f

    def writerow(self, row):
        self.writer.writerow(row)
        # Fetch UTF-8 output from the queue ...
        data = self.queue.getvalue()
        # write to the target stream
        self.stream.write(data.encode('utf-8'))
        # empty queue
        self.queue.seek(0)
        self.queue.truncate(0)

    def writerows(self, rows):
        for row in rows:
            self.writerow(row)


class CsvExporter(UnicodeWriter):
    def __init__(self, stream):
        super(CsvExporter, self).__init__(stream)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        pass

    def writeheader(self, row):
        self.writerow(row)


class MDSUploader(object):
    def __init__(self):
        self.client = MdsClient()

    def upload(self, file_name, data):
        name = slugify_filename(file_name)
        path = self.client.upload(data, '%s_%s' % (uuid1().hex, name))
        return path


class MDSDownloader(object):
    def __init__(self):
        self.client = MdsClient()

    def download(self, path):
        return self.client.get(path)


class FileUploader(object):
    def upload(self, file_name, data):
        with open(file_name, 'wb') as f:
            f.write(data)
        return os.path.realpath(file_name)
