# -*- coding: utf-8 -*-
import json
import re
import logging
import itertools
from copy import deepcopy
from collections import defaultdict

from django.conf import settings
from django.core import validators
from django.core.exceptions import ValidationError
from django.db import transaction
from django.forms import CheckboxInput, SelectDateWidget
from django import forms
from django.utils import timezone
from django.utils.datastructures import MultiValueDict
from django.utils.encoding import force_str
from django.utils.translation import ugettext as _
from django.db.utils import DatabaseError

from events.accounts.models import User
from events.accounts.utils import is_anonymous
from events.common_app.forms import (
    SelectWidget,
    MultiFileField,
    MultiFileInput,
)
from events.captcha.fields import (
    CaptchaField,
    CaptchaTextInput
)
from events.common_app.zora import get_session
from events.data_sources.forms import DataSourceField, AnswerMatrixDataSourceField
from events.forme.forms.forms import FormAsDictMixin
from events.surveyme.models import (
    ProfileSurveyAnswer,
    AnswerExportYtStatus,
)

from events.common_app.utils import (
    get_cleaned_source_request,
    get_user_ip_address,
    retry,
    timeit,
)
from events.common_storages.models import ProxyStorageModel
from events.rest_framework_contrib.fields import ImageField
from events.surveyme.utils import FormConditionEvaluator
from events.surveyme.tasks import (
    send_information_about_robot_actions,
)
from events.history.tasks import save_auto_close_history_entry
from events.balance import forms as balance_forms
from events.common_app import forms as common_app_forms
from events.common_app import widgets as common_app_widgets
from events.data_sources.models import DataSourceItem
from events.surveyme.exceptions import MaxSurveyAnswersException
from events.surveyme.fields.base import fields as base_fields
from events.surveyme.fields.base import widgets as base_widgets
from events.surveyme.fields.passport import fields as passport_fields
from events.yauth_contrib.auth import OAuth
from events.surveyme.fields.url.fields import CustomURLField


logger = logging.getLogger(__name__)


class RadioSelectMultiple(forms.RadioSelect):
    def value_from_datadict(self, data, files, name):
        # todo: test me
        if isinstance(data, MultiValueDict):
            # оставляем 1 элемент, т.к. это в конце концов селект, и особо умные
            # пользователи могут попробовать отправить несколько значений
            return data.getlist(name)[0:1]
        return data.get(name, None)


class AgreementCheckboxInput(CheckboxInput):
    pass


class ProfileSurveyQuestionAnswerForm(forms.Form):
    answer_short_text = forms.CharField(max_length=255)
    answer_long_text = forms.CharField(widget=forms.Textarea)
    answer_url = CustomURLField(max_length=1024)
    answer_non_profile_email = forms.EmailField(max_length=254)
    answer_count_of_tickets = balance_forms.TicketsQuantityField()
    answer_number = common_app_forms.IntegerField()
    answer_boolean = forms.BooleanField()
    answer_choices = DataSourceField()
    answer_files = MultiFileField()
    answer_name = passport_fields.PassportNameField()
    answer_surname = passport_fields.PassportSurnameField()
    answer_phone = base_fields.PhoneField()
    answer_payment = base_fields.PaymentField()

    def __init__(self, *args, **kwargs):
        # todo: test me date
        self.question = kwargs.pop('question', None)
        super(ProfileSurveyQuestionAnswerForm, self).__init__(*args, **kwargs)
        self._set_fields()

    def _set_fields(self):
        self._set_answer_date_field_if_needed()
        self._set_answer_choices_if_needed()

    def _set_answer_date_field_if_needed(self):
        # todo: test me date
        if self.question and self.question.answer_type.slug == 'answer_date':
            if self.question.param_date_field_type == 'date':
                self.fields['answer_date'] = base_fields.SimpleDateField()
            elif self.question.param_date_field_type == 'daterange':
                self.fields['answer_date'] = base_fields.DateRangeField()

    def _set_answer_choices_if_needed(self):
        if self.question and self.question.param_data_source == 'survey_question_matrix_choice':
            self.fields['answer_choices'] = AnswerMatrixDataSourceField()


class EmptyWidget(forms.Widget):
    def render(self, *args, **kwargs):
        return '<p></p>'


class GroupWidget(EmptyWidget):
    pass


class ParamsApplier(object):
    def __init__(self, form, field_name, field, question, group_question, user, questions_dict, data_by_question_id):
        self.field = field
        self.question = question
        self.group_question = group_question
        self.form = form
        self.field_name = field_name
        self.user = user
        self.questions_dict = questions_dict
        self.data_by_question_id = data_by_question_id
        if not hasattr(self.field.widget, 'input_type'):
            setattr(self.field.widget, 'input_type', None)
        self.input_type = self.field.widget.input_type
        self.survey_language = form.survey.language

    def apply(self):
        self.apply_essential_params()
        self.apply_overall_params()
        if self.question.answer_type.kind == 'generic':
            if self.question.answer_type.slug == 'answer_choices':
                self.apply_choices_params()
            elif self.question.answer_type.slug == 'answer_count_of_tickets':
                self.apply_answer_count_of_tickets_params()
            elif self.question.answer_type.slug == 'answer_payment':
                self.apply_answer_payment_params()
            elif self.question.answer_type.slug == 'answer_non_profile_email':
                self.apply_non_profile_answer_email_params()
            elif self.question.answer_type.slug in ('answer_short_text', 'answer_long_text'):
                self.apply_text_answer_params()
        self.apply_field_type_specific_params()
        self.remove_bad_help_text()
        self.apply_hidden()

    def apply_essential_params(self):
        self.field.question = self.question
        self.field.page = self.question.page
        self.field.position = self.question.position
        self.field.group_id = self.group_question and self.group_question.pk
        self.field.group_slug = self.group_question and self.group_question.param_slug
        if self.field.initial is None and self.question.initial is not None:
            self.field.initial = self.form.get_initial(self.question)

    def apply_overall_params(self):
        self.apply_label()
        self.apply_required()
        self.apply_custom_clean_method()
        self.apply_help_text()
        self.apply_hint()

    def apply_label(self):
        self.field.label = self.question.get_label()

    def apply_required(self):
        self.field.required = bool(
            self.question.param_is_required and
            not self.question.answer_type.is_read_only and
            not self.form.make_all_fields_not_required
        )

    def apply_hidden(self):
        self.field.widget.input_type = 'hidden' if self.question.param_is_hidden else self.input_type

    def apply_custom_clean_method(self):
        old_clean_method = self.field.clean

        def new_clean_method(value, *args, **kwargs):
            # если поле не должно быть показано по условию
            if getattr(self.field, '_is_should_be_shown_by_condition', None) is False:
                return None
            else:
                data = old_clean_method(value, *args, **kwargs)
                # validate file size
                if isinstance(self.field, MultiFileField):
                    self._multifile_field_validate(data)
                elif isinstance(self.field, forms.FileField):
                    self._file_field_validate(data)
                elif isinstance(self.field, base_fields.DateRangeField):
                    self._date_range_field_validate(data)
                elif isinstance(self.field, base_fields.SimpleDateField):
                    self._simple_date_field_validate(data)
                self._apply_custom_validator(data)
                return data

        self.field.clean = new_clean_method

    def _apply_custom_validator(self, data):
        if data not in validators.EMPTY_VALUES:
            validator_type = self.question.validator_type
            options = self.question.validator_options or {}
            if validator_type and not validator_type.is_external:
                try:
                    validator_class = settings.SURVEYME_VALIDATORS.get(validator_type.slug)
                    if validator_class:
                        validator = validator_class(**options)
                        validator(data)
                except ValidationError:
                    raise
                except Exception:
                    logger.exception('Got unexpected error while processing through custom validator')

    def _file_field_validate(self, data):
        if data not in validators.EMPTY_VALUES and hasattr(data, 'size'):
            max_file_size = self.field.question.param_max_file_size * 1024 * 1024
            if data.size > max_file_size:
                # u'Размер загружаемого файла не должен превышать {0} мегабайт'
                raise ValidationError(_('max_file_size_exceeded') % {'max_file_size': self.field.question.param_max_file_size})

    def _multifile_field_validate(self, data):
        if len(data) > self.field.question.param_max_files_count:
            # u'Превышено максимально допустимое число файлов ({0})'
            self.field.initial = json.dumps(data)
            self.form.add_other_data_info_for_multifile(self.field)
            raise ValidationError(_('max_files_count_exceeded') % {'max_files_count': self.field.question.param_max_files_count})
        size = 0

        max_files_size = self.field.question.param_max_file_size * 1024 * 1024
        for file_path in data:
            try:
                file_meta_info = ProxyStorageModel.objects.get(path=file_path)
                size += file_meta_info.file_size
            except ProxyStorageModel.DoesNotExist:
                pass

        if size > max_files_size:
            # u'Суммарный размер загружаемых файлов не должен превышать {0} мегабайт'
            self.field.initial = json.dumps(data)
            self.form.add_other_data_info_for_multifile(self.field)
            raise ValidationError(_('max_files_size_exceeded') % {'max_files_size': self.field.question.param_max_file_size})

    def _date_range_field_validate(self, data):
        # todo: test me
        if data and data.get('date_start'):
            msg = _('answer_date_less_date_field_min') % {'date': self.field.question.param_date_field_min}
            self._validate_date_gt_min_date(data.get('date_start'), msg=msg)
            self._validate_date_lt_max_date(data.get('date_start'), msg=msg)
        if data and data.get('date_end'):
            msg = _('answer_date_gt_date_field_max') % {'date': self.field.question.param_date_field_max}
            self._validate_date_gt_min_date(data.get('date_end'), msg=msg)
            self._validate_date_lt_max_date(data.get('date_end'), msg=msg)

    def _simple_date_field_validate(self, data):
        # todo: test me
        if data and data.get('date_start') and self.field.question.param_date_field_min and self.field.question.param_date_field_max:
            msg = _('date_not_in_interval') % {
                'date_start': self.field.question.param_date_field_min,
                'date_end': self.field.question.param_date_field_max,
            }
            self._validate_date_gt_min_date(data.get('date_start'), msg=msg)
            self._validate_date_lt_max_date(data.get('date_start'), msg=msg)
        elif data and data.get('date_start') and self.field.question.param_date_field_min:
            msg = _('date_must_be_later') % {'date': self.field.question.param_date_field_min}
            self._validate_date_gt_min_date(data.get('date_start'), msg=msg)
        elif data and data.get('date_start') and self.field.question.param_date_field_max:
            msg = _('date_must_be_earlier') % {'date': self.field.question.param_date_field_max}
            self._validate_date_lt_max_date(data.get('date_start'), msg=msg)

    def _validate_date_gt_min_date(self, answer_date, msg=None):
        # todo: test me
        if self.field.question.param_date_field_min and answer_date < self.field.question.param_date_field_min:
            raise ValidationError(msg)

    def _validate_date_lt_max_date(self, answer_date, msg=None):
        # todo: test me
        if self.field.question.param_date_field_max and answer_date > self.field.question.param_date_field_max:
            raise ValidationError(msg)

    def apply_help_text(self):
        self.field.help_text = self.question.get_param_help_text()

    def apply_hint(self):
        # todo: test me
        self.field.hints = []
        if self.question.param_hint_data_source:
            self.field.hints = self.question.get_hint_data(
                user=self.user,
                questions_dict=self.questions_dict,
            )

    def apply_choices_params(self):
        # todo: test me
        self.field.data_source = self.question.param_data_source
        self.field.allow_multiple_choice = self.question.param_is_allow_multiple_choice
        self.field.random_choices_position = self.question.param_is_random_choices_position
        self.field.modify_choices = self.question.param_modify_choices
        self.field.suggest_choices = self.question.param_suggest_choices
        self.field.form = self.form
        self.field.questions_dict = self.questions_dict
        self.field.user = self.user
        self.field.filters = self._build_choices_filter_data(
            filters=self.question.param_data_source_params and self.question.param_data_source_params.get('filters') or []
        )
        # self.field.queryset = self.question.surveyquestionchoice_set  # не использовать all, иначе при следующем all
        #                                                               # не будет использоваться кэш из prefetch_related

    def _build_choices_filter_data(self, filters):
        # todo: test me by unit tests (already tested functionally)
        filter_data = {}
        for filter in filters:
            if filter['type'] == 'specified_value':
                multiple_value = filter_data.get(filter['filter']['name'])
                if multiple_value is not None:
                    if isinstance(multiple_value, list):
                        filter_data[filter['filter']['name']].append(filter['value'])
                    else:
                        filter_data[filter['filter']['name']] = [multiple_value, filter['value']]
                else:
                    filter_data[filter['filter']['name']] = filter['value']

            elif filter['type'] == 'field_value':
                field_id = int(filter['field'])
                if field_id in self.data_by_question_id:
                    filter_data[filter['filter']['name']] = self.data_by_question_id[field_id]
        return filter_data

    def apply_answer_count_of_tickets_params(self):
        price = self.question.param_price or self.question.survey.tickets_info.price
        attrs = {'price': price, 'currency': self.question.survey.tickets_info.currency}
        self.field.widget.attrs.update(attrs)

    def apply_answer_payment_params(self):
        param_payment = self.question.param_payment or {}
        other_data = {
            'is_fixed': bool(param_payment.get('is_fixed')),
            'min': self.field.min_amount,
            'max': self.field.max_amount,
        }
        self.field.other_data = other_data

    def apply_non_profile_answer_email_params(self):
        pass

    def apply_field_type_specific_params(self):
        # проверка виджета из-за города, иначе была попытка сменить виджет + заполучить все choices
        if isinstance(self.field, forms.ChoiceField) and isinstance(self.field.widget, forms.Select):
            # возможно стоить проверять не на kind == 'generic',
            # а на то, что результирующее поле ManyToMany
            if self.question.answer_type.kind == 'generic':
                self.apply_choice_field_widget()
        if isinstance(self.field, common_app_forms.IntegerField):
            self._apply_integer_field_params()

    def apply_text_answer_params(self):
        param_max, param_min = self.question.param_max, self.question.param_min
        self.field.widget.attrs.update({'min': param_min, 'max': param_max})
        if param_min is not None:
            self.field.validators.append(validators.MinLengthValidator(param_min))
        if param_max is not None:
            self.field.validators.append(validators.MaxLengthValidator(param_max))

    def _apply_integer_field_params(self):
        param_max, param_min = self._get_min_and_max_integer_field_params()
        if param_max is None:
            param_max = settings.MAX_POSTGRESQL_INT_VALUE  # FORMS-3035
        self.field.widget.attrs.update({'min': param_min, 'max': param_max})
        if param_min is not None:
            self.field.validators.append(validators.MinValueValidator(param_min))
        if param_max is not None:
            self.field.validators.append(validators.MaxValueValidator(param_max))

    def _get_min_and_max_integer_field_params(self):
        if isinstance(self.field, balance_forms.TicketsQuantityField):
            available_quantity = self.question.survey.get_tickets_status().get('available_quantity')
            if available_quantity is None:
                param_max = self.question.param_max
            else:
                if self.question.param_max is None:
                    param_max = int(available_quantity)
                else:
                    param_max = min(int(available_quantity), self.question.param_max)

            if self.question.param_min is None:
                if param_max is not None:
                    param_min = min(param_max, 1)
                else:
                    param_min = 1
            else:
                param_min = self.question.param_min
        else:
            param_max, param_min = self.question.param_max, self.question.param_min
        return param_max, param_min

    def apply_choice_field_widget(self):
        def new_bound_data(data, initial):
            # длял рендеринга радиобаттонов необходимо только одно значение
            if data:
                return data[0]

        attrs = self.field.widget.attrs
        choices = self.field.widget.choices
        if self.question.param_is_allow_multiple_choice:
            if self.question.param_widget == 'list':
                self.field.widget = forms.CheckboxSelectMultiple(
                    attrs=attrs,
                    choices=choices,
                    is_show_labels_as_links=False,
                )
            else:
                self.field.widget = forms.SelectMultiple(attrs=attrs, choices=choices)
        else:
            initial = self.field.initial
            if isinstance(initial, (list, tuple)) and len(initial):
                self.field.initial = initial[0]
            if self.question.param_widget == 'list':
                self.field.widget = RadioSelectMultiple(
                    attrs=attrs,
                    choices=choices,
                    is_show_labels_as_links=False,
                )
                self.field.bound_data = new_bound_data
            else:
                self.field.widget = SelectWidget(attrs=attrs, choices=choices)

    def remove_bad_help_text(self):
        bad_help_text = force_str(_('Hold down "Control", or "Command" on a Mac, to select more than one.')).strip()
        if force_str(self.field.help_text).strip() == bad_help_text:
            self.field.help_text = None


class SurveyForm(FormAsDictMixin, forms.ModelForm):
    profile_prefix = 'user'
    question_answer_prefix = 'question_answer'
    param_phone = base_fields.PhoneField()

    class Meta:
        model = User
        widgets = {
            'param_gender': forms.RadioSelect,
        }
        fields = '__all__'
        exclude = ['uid', 'cloud_uid', 'username', 'email', 'is_superuser', ]

    def clean_param_course(self):
        param_course = self.cleaned_data.get('param_course')
        if isinstance(param_course, int) and param_course <= 0:
            raise ValidationError(_('Курс должен быть больше 0'))
        return param_course

    def _need_to_skip(self, value):
        VALUES_TO_SKIP = (None, '', )

        skipping = False
        if isinstance(value, list) and not len(value):
            skipping = True
        elif value in VALUES_TO_SKIP:
            skipping = True
        return skipping

    @timeit
    def clean(self):
        cleaned_data = super(SurveyForm, self).clean()
        for name, value in list(cleaned_data.items()):
            if isinstance(self.fields[name].widget, RadioSelectMultiple):
                if value:
                    cleaned_data[name] = value[0]
            # Убираем из данных поля, которые не нужно показывать по условиям.
            # Благодаря этому они не обновят существующие значения этих полей
            # дополнительное условие: удаляем пустые значения
            if self.evaluated_conditions.get(name) is False or self._need_to_skip(value):
                try:
                    del cleaned_data[name]
                except KeyError:
                    pass
        if self.survey.validator_url:
            self.invoke_external_validator(cleaned_data)
        return cleaned_data

    def __init__(self,
                 survey,
                 is_with_additional_fields=True,
                 make_all_fields_not_required=False,
                 is_with_captcha=True,
                 **kwargs):
        self.survey = survey
        self.is_with_additional_fields = is_with_additional_fields
        self.make_all_fields_not_required = make_all_fields_not_required
        self.is_with_captcha = is_with_captcha
        self.request = kwargs.pop('request', None)
        super(SurveyForm, self).__init__(**kwargs)
        self.init_fields()

    def get_last_non_group_field(self, fields):
        for (it, field) in fields[::-1]:  # noqa
            if not field.group_id:
                return field

    def init_fields(self):
        # fields for every question in survey
        expected_fields = self.get_survey_question_fields()
        page, position = 1, 1
        if expected_fields:
            last_field = self.get_last_non_group_field(expected_fields)
            if last_field:
                page = last_field.page
                position = last_field.position + 1

        if self.is_with_additional_fields:
            # agreement checkboxes
            for agreement in self.survey.agreements.all():
                expected_fields.append(self.get_agreement_name_and_field(agreement, page, position))
                position += 1

            # captcha
            if self.is_with_captcha:
                expected_fields.append(('captcha', self.get_captcha_field(page, position)))

        field_names, fields_by_name, show_conditions_for_fields = self.get_show_conditions_fields(
            expected_fields,
            with_additional_info=True,
        )
        self.evaluated_conditions = self.get_evaluated_conditions(
            self.data,
            field_names,
            fields_by_name,
            show_conditions_for_fields,
        )

        # we need to remove additional_info from conditions
        for field_name, condition_nodes in show_conditions_for_fields.items():
            for node in condition_nodes:
                for condition in node:
                    condition.pop('additional_info', None)

        # add some utils widget info to fields
        self.patch_fields(expected_fields, show_conditions_for_fields, self.evaluated_conditions)

        # clear and use only expected fields
        self.use_only_fields(expected_fields)

    def get_or_create_datasourceitem(self, data_source, identity):
        datasourceitem, _ = DataSourceItem.objects.get_or_create(
            data_source=data_source,
            identity=identity,
        )
        return datasourceitem.pk

    def get_initial(self, question):
        if question.answer_type.slug == 'answer_choices':
            return [
                self.get_or_create_datasourceitem(question.param_data_source, identity)
                for identity in question.initial
                if identity is not None
            ]
        return question.initial

    def get_survey_question_fields(self):
        is_editable = bool(self.data)
        groups_counter = {}
        if is_editable:
            groups_map = self.get_questions_group_map()
            groups_counter = defaultdict(set)
            group_question_re = re.compile(r'(?P<group_key>.+?)__(?P<group_counter>\d+)$')
            for field_form_name in self.data.keys():
                group_question = group_question_re.match(field_form_name)
                if group_question:
                    group_key = group_question.group('group_key')
                    if group_key in groups_map:
                        groups_counter[groups_map[group_key]].add(group_question.group('group_counter'))

        survey_question_fields = []
        for question in self.get_questions():
            field_name = question.param_slug
            if question.group_id is not None:
                continue  # пропускаем вопросы члены группы, будем их обрабатывать после самой группы
            elif question.answer_type.kind == 'generic':
                fields = self.create_generic_field(question)
            else:
                raise Exception('Fatal error')
            for field in fields:
                self.apply_params_to_field(form=self, field_name=field_name, field=field, question=question)

            survey_question_fields.extend(itertools.product([field_name], fields))

            for child_question in getattr(question, 'children', []):
                fields = self.create_generic_field(child_question)
                if is_editable:
                    for i in groups_counter.get(question.pk, [0]):
                        field = deepcopy(fields[0])
                        self.apply_params_to_field(
                            form=self, field_name=field_name, field=field,
                            question=child_question, group_question=question
                        )
                        survey_question_fields.append((child_question.get_form_field_name(i), field))
                else:
                    for i, field in enumerate(fields):
                        self.apply_params_to_field(
                            form=self, field_name=field_name, field=field,
                            question=child_question, group_question=question
                        )
                        survey_question_fields.append((child_question.get_form_field_name(i), field))

        return survey_question_fields

    def get_questions_as_dict(self):
        if not hasattr(self, '_questions_as_dict'):
            self._questions_as_dict = dict([(i.id, i) for i in self.get_questions()])
        return self._questions_as_dict

    def get_questions_group_map(self):
        return {
            question.get_form_field_name(): question.group_id
            for question in self.get_questions()
        }

    def link_group_with_children(self, questions):
        groups = {
            question.pk: question
            for question in questions
            if question.answer_type.slug == 'answer_group'
        }
        if groups:
            for question in questions:
                group_id = question.group_id
                if group_id and group_id in groups:
                    group = groups[group_id]
                    if not hasattr(group, 'children'):
                        setattr(group, 'children', [])
                    group.children.append(question)
        return questions

    def get_questions(self):
        if not hasattr(self, '_questions'):
            questions = self.survey.surveyquestion_set.all()
            self._questions = self.link_group_with_children(questions)
        return self._questions

    def get_data_by_question_id(self):
        # todo: test me
        if not hasattr(self, '_data_by_question_id'):
            self._data_by_question_id = dict([
                (i.id, self.data.get(i.get_form_field_name()))
                for i in self.get_questions()
            ])
        return self._data_by_question_id

    def get_agreement_name_and_field(self, agreement, page, position):
        agreement_field_name, agreement_field_params = self.get_agreement_field_params(agreement)
        field = forms.BooleanField(**agreement_field_params)
        field.page = page
        field.position = position
        field.group_id = None
        field.group_slug = None
        return agreement_field_name, field

    def get_agreement_field_params(self, agreement):
        initial = False
        required_error = _('This field is required.')
        field_name = 'is_agree_with_%s' % agreement.slug

        if agreement.slug in ('hr', 'events'):
            required_error = _('registration_agreement_error_message')

        params = {
            'initial': initial,
            'label': force_str(agreement.get_text()),
            'required': agreement.is_required,
            'error_messages': {'required': required_error},
            'widget': AgreementCheckboxInput
        }
        return field_name, params

    def get_captcha_field(self, page, position):
        captcha_type = self.survey.captcha_type
        if captcha_type == 'ocr':
            label = _('enter_ocr_captcha')
        elif captcha_type == settings.CAPTCHA_TYPE:
            label = _('enter_number_from_image')
        field = CaptchaField(
            label=label,
            captcha_type=captcha_type,
        )
        field.page = page
        field.position = position
        field.group_id = None
        field.group_slug = None
        return field

    def patch_fields(self, fields, show_conditions, evaluated_conditions):
        for field_name, field in fields:
            self.add_other_data(
                field=field, field_name=field_name,
                show_conditions=show_conditions,
                evaluated_conditions=evaluated_conditions,
            )
            self.remove_choices_empty_label(field)

    def add_other_data(self, field, field_name, show_conditions, evaluated_conditions):
        if not hasattr(field, 'other_data'):
            field.other_data = {}
        self.add_widget_info(field)
        self.add_widget_settings_info(field)
        self.add_label_image(field)
        self.add_show_conditions_info(
            field=field, field_name=field_name,
            show_conditions=show_conditions,
            evaluated_conditions=evaluated_conditions,
        )
        self.add_date_field_info(field)

    def add_widget_info(self, field):
        field.other_data['widget'] = self.get_widget_info(field=field)
        if isinstance(field, CaptchaField):
            field.other_data['captcha_type'] = field.captcha_type

    def add_widget_settings_info(self, field):
        if field.other_data['widget'] == 'statement':
            field.other_data['is_section_header'] = field.question.param_is_section_header
        if field.other_data['widget'] == 'file':
            field.other_data['max_file_size'] = field.question.param_max_file_size
            if field.initial:
                field.other_data['already_uploaded_filename'] = force_str(field.initial).split('/')[-1]
        if field.other_data['widget'] == 'multifile':
            self.add_other_data_info_for_multifile(field)

    def add_label_image(self, field):
        data = None
        if hasattr(field, 'question') and field.question.label_image:
            links = ImageField(sizes=settings.IMAGE_SIZES)
            data = {
                'links': links.to_representation(field.question.label_image.image),
            }
        field.label_image = data

    def add_other_data_info_for_multifile(self, field):
        field.other_data['max_file_size'] = field.question.param_max_file_size
        field.other_data['max_files_count'] = field.question.param_max_files_count

        # todo: test me
        if hasattr(self.data, 'getlist'):
            sended_files_ids = self.data.getlist(field.question.get_form_field_name(), [])
        else:
            sended_files_ids = self.data.get(field.question.get_form_field_name(), [])
        if sended_files_ids:
            if not isinstance(sended_files_ids, list):
                sended_files_ids = [sended_files_ids]
            sended_files_ids = [_id for _id in sended_files_ids if _id]
            field.other_data['already_uploaded_files'] = self._get_already_uploaded_files_info(
                sended_files_ids,
                get_by_sha256=True,
            )
        elif field.initial:
            field.other_data['already_uploaded_files'] = self._get_already_uploaded_files_info(json.loads(field.initial))

    def _get_already_uploaded_files_info(self, initial_data, get_by_sha256=False):
        already_uploaded_files = []
        for file_path in initial_data:
            try:
                if not get_by_sha256:
                    file_meta_info = ProxyStorageModel.objects.get(path=file_path)
                else:
                    file_meta_info = ProxyStorageModel.objects.get_by_sha256(file_path, self.survey.pk)
            except ProxyStorageModel.DoesNotExist:
                raise ValidationError(_('files_uploading_error'))
            already_uploaded_files.append({
                'name': file_meta_info.original_name,
                'id': file_meta_info.sha256,
                'size': file_meta_info.file_size,
                'namespace': file_meta_info.namespace or settings.MDS_OLD_NAMESPACE,
            })
        return already_uploaded_files

    def add_show_conditions_info(self, field, field_name, show_conditions, evaluated_conditions):
        if hasattr(field, 'question'):
            show_conditions = show_conditions.get(field_name)
            group_slug = getattr(field, 'group_slug', None)

            if show_conditions:
                field.other_data['show_conditions'] = show_conditions
                self._set_is_should_be_shown_by_condition(field, field_name, evaluated_conditions)

            if group_slug and getattr(field, '_is_should_be_shown_by_condition', None) is not False:
                # Проверим что сама группа показывается - если нет, то и сам вопрос не должен быть показан
                self._set_is_should_be_shown_by_condition(field, group_slug, evaluated_conditions)

    def _set_is_should_be_shown_by_condition(self, field, slug, evaluated_conditions):
        evaluated_condition = evaluated_conditions.get(slug)
        if evaluated_condition is False:
            field._is_should_be_shown_by_condition = False

    def add_date_field_info(self, field):
        if isinstance(field, base_fields.DateRangeField) or isinstance(field, base_fields.SimpleDateField):
            for param, name in {'param_date_field_min': 'from', 'param_date_field_max': 'to'}.items():
                param_value = getattr(field.question, param)
                if param_value:
                    if 'allowed_range' not in field.other_data:
                        field.other_data['allowed_range'] = {}
                    field.other_data['allowed_range'][name] = param_value.isoformat()

    def get_widget_info(self, field):
        # todo: test me
        widget = None
        widget_type = type(field.widget)
        if widget_type == AgreementCheckboxInput:
            widget = 'agreement'
        elif widget_type == forms.CheckboxInput:
            widget = 'checkbox'
        elif widget_type in [
            RadioSelectMultiple,
            forms.CheckboxSelectMultiple,
            forms.RadioSelect
        ]:
            widget = 'radio_or_checkbox_list'
        elif widget_type in [SelectDateWidget]:
            widget = 'date'
        elif widget_type in [EmptyWidget]:
            widget = 'statement'
        elif widget_type in [CaptchaTextInput]:
            widget = 'captcha'
        elif issubclass(widget_type, MultiFileInput):
            widget = 'multifile'
        elif issubclass(widget_type, base_widgets.DateRangeFieldInput):
            widget = 'date_range'
        elif issubclass(widget_type, forms.DateInput):
            widget = 'date'
        elif issubclass(widget_type, forms.widgets.FileInput):
            widget = 'file'
        elif issubclass(widget_type, balance_forms.TicketsQuantityInput):
            widget = 'tickets'
        elif issubclass(widget_type, common_app_widgets.NumberInput):
            widget = 'number'
        return widget

    def patch_widget_input_type(self, field):
        pass

    def remove_choices_empty_label(self, field):
        if hasattr(field, 'choices') and isinstance(field.choices, list) and field.choices[0][0] == '':
            choices = field.choices
            del choices[0]
            field.choices = choices

    def use_only_fields(self, use_fields):
        self.fields.clear()
        for i, field in enumerate(use_fields):
            field_name, field_instance = field
            self.fields[field_name] = field_instance

    def create_generic_field(self, question):
        field_name = question.answer_type.slug
        if question.answer_type.is_read_only:
            generic_fields = self.create_generic_field_for_question_with_read_only_answer(field_name)
        else:
            generic_fields = self.create_generic_field_for_question_with_writable_answer(question, field_name)
        for generic_field in generic_fields:
            generic_field.is_generic_answer = True
        return generic_fields

    def create_generic_field_for_question_with_read_only_answer(self, field_name):
        widget = GroupWidget if field_name == 'answer_group' else EmptyWidget
        return [forms.Field(widget=widget)]

    def create_generic_field_for_question_with_writable_answer(self, question, field_name):
        existing_question_answers = None
        if self.instance and self.survey.is_allow_answer_editing:
            existing_question_answers = self.get_profile_existing_answer_for_question(
                question=question,
            )
        existing_question_answers = existing_question_answers or [None]

        generic_fields = []
        for existing_question_answer in existing_question_answers:
            answer_form = ProfileSurveyQuestionAnswerForm(question=question)
            generic_field = answer_form.fields[field_name]
            generic_field.initial = answer_form.initial.get(field_name)
            generic_field.existing_question_answer = existing_question_answer
            generic_fields.append(generic_field)
        return generic_fields

    def get_profile_existing_answer_for_question(self, question):
        # todo: поправить получение данных о ответе на вопрос
        return self.get_profile_existing_answers_for_survey().get(question.pk)

    def get_profile_existing_answers_for_survey(self):
        if not hasattr(self, '_profile_existing_answers_for_survey'):
            self._profile_existing_answers_for_survey = self.get_existing_answers() or {}
        return self._profile_existing_answers_for_survey

    @timeit
    def get_existing_answers(self):
        if is_anonymous(self.instance):
            return None

        profile_answer = (
            ProfileSurveyAnswer.objects.using(settings.DATABASE_ROLOCAL)
            .filter(
                survey=self.survey,
                user=self.instance,
            )
            .order_by('-date_updated')
            .first()
        )

        if profile_answer:
            return profile_answer.as_dict()

    def apply_params_to_field(self, field_name, form, field, question, group_question=None):
        applier = ParamsApplier(
            form=form,
            field_name=field_name,
            field=field,
            question=question,
            group_question=group_question,
            user=self.instance,
            questions_dict=self.get_questions_as_dict(),
            data_by_question_id=self.get_data_by_question_id()
        )
        applier.apply()

    @timeit
    def get_show_conditions_fields(self, fields, with_additional_info=False):
        field_names = []
        fields_by_name = {}
        show_conditions_dict = {}
        for field_name, field in fields:
            fields_by_name[field_name] = field
            if hasattr(field, 'question'):
                show_conditions = field.question.get_show_conditions(
                    self.get_questions_as_dict(),
                    with_additional_info=with_additional_info,
                    field_name=field_name,
                    field=field,
                )
                if show_conditions:
                    field_names.append(field_name)
                    show_conditions_dict[field_name] = show_conditions
        return field_names, fields_by_name, show_conditions_dict

    @timeit
    def get_evaluated_conditions(self, data, field_names, fields_by_name, conditions):
        return FormConditionEvaluator(conditions=conditions).evaluate(
            source_data=data,
            field_names=field_names,
            fields_by_name=fields_by_name,
        )

    def get_fields_by_name(self):
        return {
            field_name: field
            for field_name, field in self.fields.items()
        }

    @timeit
    def save(self, commit=True):
        profile_survey_answer, is_profile_survey_answer_created = self.get_profile_survey_answer()
        self.update_answer_data(profile_survey_answer)
        self.post_save(is_profile_survey_answer_created)

        return {
            'profile_survey_answer': profile_survey_answer,
            'is_profile_survey_answer_created': is_profile_survey_answer_created,
        }

    def get_profile_survey_answer(self):
        if self.survey.is_allow_multiple_answers or is_anonymous(self.instance):
            answer, is_created = ProfileSurveyAnswer(user=self.instance, survey=self.survey), True
            answer.save()
        else:
            answer, is_created = ProfileSurveyAnswer.objects.filter(user=self.instance, survey=self.survey).first(), False
            if not answer:
                answer, is_created = ProfileSurveyAnswer(user=self.instance, survey=self.survey), True
                answer.save()
        return answer, is_created

    @timeit
    def post_save(self, is_profile_survey_answer_created):
        # инкремент количества ответов
        if is_profile_survey_answer_created:
            self._increment_answers_count()

    @timeit
    def _increment_answers_count(self):
        from events.surveyme.models import SurveyTicket

        if self.survey.maximum_answers_count is not None:
            locked_ticket = self._get_fresh_locked_survey_ticket()
            free_tickets = SurveyTicket.objects.filter(survey=self.survey, acquired=False).exists()
            if not free_tickets or not locked_ticket:
                self.survey.is_published_external = False
                self.survey.date_unpublished = timezone.now()
                self.survey.save(update_fields=['is_published_external', 'date_unpublished'])
                transaction.on_commit(lambda: save_auto_close_history_entry.delay(self.survey.pk))
                send_information_about_robot_actions.delay(
                    self.survey.pk,
                    'закрыта (достигнуто максимальное количество ответов)'
                )
            if not locked_ticket:
                raise MaxSurveyAnswersException()
        else:
            SurveyTicket.objects.create(survey=self.survey, acquired=True)

    @retry(exceptions=(DatabaseError, ), attempts=5)
    def _get_fresh_locked_survey_ticket(self):
        from events.surveyme.models import SurveyTicket

        try:
            free_ticket_id = (
                SurveyTicket.objects
                            .filter(survey=self.survey, acquired=False)
                            .order_by('?')
                            .values_list('pk', flat=True)[0]
            )
        except IndexError:
            pass
        else:
            with transaction.atomic():
                ticket = SurveyTicket.objects.select_for_update().get(pk=free_ticket_id)
                if ticket.acquired:
                    return self._get_fresh_locked_survey_ticket()
                ticket.acquired = True
                ticket.save(update_fields=['acquired'])
            return ticket

    def invoke_external_validator(self, cleaned_data):
        try:
            self._invoke_external_validator(cleaned_data)
        except ValidationError:
            raise
        except Exception:
            logger.exception('Got error while invoke external validator')

    def _invoke_external_validator(self, cleaned_data):
        data = self.get_validated_data(cleaned_data)
        if data['questions']:
            self.validation_status = 'error'
            response = self.post_data(self.survey.validator_url, data)
            if response.status_code == 200:
                try:
                    data = response.json()
                except Exception as e:
                    raise ValidationError(str(e))
                else:
                    if data.get('status') != 'OK':
                        errors = {
                            field: error
                            for field, error in data.get('errors', {}).items()
                            if field in self.fields
                        }
                        raise ValidationError(errors or 'Unknown error')
                    self.validation_status = 'success'
            else:
                logger.warning('external validator response (%s) %s', response.status_code, force_str(response.content))

    def post_data(self, url, data):
        request_id = self.data.get('request_id')
        session = get_session(url, request_id)
        headers = {
            'X-Uid': str(self.instance.uid or ''),
            'X-Login': self.instance.email,
            'X-User-Ip': get_user_ip_address(),
        }
        return session.post(
            url=url,
            json=data,
            headers=headers,
            auth=OAuth(settings.HTTP_INTEGRATION_OAUTH_TOKEN),
        )

    def get_value_text(self, question, value):
        from django.db.models.query import QuerySet
        if isinstance(value, (QuerySet, list)):
            return [self.get_value_text(question, it) for it in value]
        elif isinstance(value, DataSourceItem):
            return value.get_text()
        elif question.answer_type.slug == 'answer_date':
            date_start = value.get('date_start')
            if date_start:
                date_start = date_start.isoformat()
            date_end = value.get('date_end')
            if date_end:
                date_end = date_end.isoformat()
            return [date_start, date_end]
        elif value is None:
            return None
        return force_str(value)

    def get_validated_data(self, cleaned_data):
        question_slugs = set(cleaned_data.keys())
        questions = [
            {
                'id': question.pk,
                'label': question.label,
                'slug': question.param_slug,
                'value': self.get_value_text(question, cleaned_data[question.param_slug]),
                'answer_type': {
                    'slug': question.answer_type.slug,
                    'name': question.answer_type.name,
                },
            }
            for question in self.get_questions()
            if (
                question.validator_type and question.validator_type.is_external
                and question.param_slug in question_slugs
            )
        ]
        return {
            'id': self.survey.pk,
            'slug': self.survey.slug,
            'name': self.survey.name,
            'questions': questions,
        }

    def set_export_yt_status(self, answer, exported):
        try:
            answer_yt_status = AnswerExportYtStatus.objects.get(answer=answer)
            if answer_yt_status.exported != exported:
                answer_yt_status.exported = exported
                try:
                    answer_yt_status.save(update_fields=['exported'])
                except Exception as e:
                    logger.warn(e)
        except AnswerExportYtStatus.DoesNotExist:
            AnswerExportYtStatus.objects.create(answer=answer, exported=exported)

    @timeit
    def update_answer_data(self, answer):
        from events.surveyme.aggregate_answer import get_answer_data
        answer.data = get_answer_data(self.survey, answer, self.cleaned_data)
        answer.date_updated = timezone.now()
        answer.source_request = get_cleaned_source_request(self.request)
        answer.save(update_fields=['data', 'date_updated', 'source_request'])
        if self.survey.save_logs_for_statbox or settings.IS_BUSINESS_SITE:
            self.set_export_yt_status(answer, False)
