# coding: utf-8

from collections import defaultdict

from rest_framework.exceptions import ValidationError

from events.surveyme.models import SurveyQuestion, get_show_conditions, get_move_error_message


def move_page_questions(survey, page_from, page_to):
    """
    Перенести группу вопросов со страницы page_from на страницы page_to.
    """
    questions = SurveyQuestion.objects.filter(survey=survey)

    page_to_questions = _get_page_to_questions(questions)

    if page_from not in page_to_questions:
        raise ValidationError('No such page: %d' % page_from)
    if page_to not in page_to_questions:
        raise ValidationError('No such page: %d' % page_to)

    if page_from > page_to:
        shift_from = page_to
        shift_to = page_from - 1
        shift = 1
    else:
        shift_from = page_from + 1
        shift_to = page_to
        shift = -1
    shift_range = range(shift_from, shift_to + 1)

    changed_questions = []
    for page, page_questions in page_to_questions.items():
        if page == page_from:
            target_page = page_to
        elif page in shift_range:
            target_page = page + shift
        else:
            continue
        for question in page_questions:
            question.page = target_page
            changed_questions.append(question)

    if page_to > page_from:
        # Если страница с вопросами переместилась ниже, то надо проверить, что
        # отображение ни одного из вопросов со сдвигаемых выше страниц не зависело
        # от какого-либо вопроса перемещаемой страницы.

        pk_to_changed_question = {question.pk: question for question in changed_questions}
        conditions_queryset = get_show_conditions(survey)

        for question_with_condition_pk, question_in_condition_pk in conditions_queryset:
            if all(
                    x in pk_to_changed_question
                    for x in (question_with_condition_pk, question_in_condition_pk)
            ):
                question_with_condition = pk_to_changed_question[question_with_condition_pk]
                question_in_condition = pk_to_changed_question[question_in_condition_pk]

                if question_in_condition.page == page_to:
                    msg = get_move_error_message(question_with_condition, question_in_condition)
                    raise ValidationError(msg)

    for question in changed_questions:
        question.save()


def delete_page_questions(survey, page):
    """
    Перенести группу вопросов стрраницы page.
    """

    questions = SurveyQuestion.objects.filter(survey=survey)

    page_to_questions = _get_page_to_questions(questions)
    if page not in page_to_questions:
        raise ValidationError('No such page: %d' % page)

    changed_questions = []
    for question in questions:
        if question.page > page:
            question.page -= 1
            changed_questions.append(question)

    SurveyQuestion.objects.filter(survey=survey, page=page).update(is_deleted=True)

    for question in changed_questions:
        question.save()


def _get_page_to_questions(questions):
    page_to_questions = defaultdict(list)
    for question in questions:
        page_to_questions[question.page].append(question)
    return page_to_questions
