# -*- coding: utf-8 -*-
from datetime import timedelta
from django.db import models
from django.db.models.query import QuerySet
from django.utils import timezone

from events.common_app.managers import ProxyToQuerySetMixin
from events.surveyme.conditions import SurveyConditions


class SurveyQuerySet(QuerySet):
    def filter_by_published_external(self):
        return self.filter(SurveyConditions.is_published_external())

    def filter_by_ready_for_answer(self):
        return self.filter(SurveyConditions.is_ready_for_answer())

    def filter_by_is_answer_could_be_edited(self):
        return self.filter(SurveyConditions.is_answer_could_be_edited())

    def filter_by_fit_for_search_text(self, search_text):
        return self.filter(SurveyConditions.is_fit_for_search_text(search_text=search_text))

    def filter_by_publication_status(self, status):
        return self.filter(SurveyConditions.publication_status(status))


class SurveyGroupManager(models.Manager):
    def get_queryset(self):
        return super(SurveyGroupManager, self).get_queryset()


class SurveyManager(ProxyToQuerySetMixin, models.Manager):
    def get_queryset(self):
        queryset = SurveyQuerySet(self.model, using=self._db)
        return queryset.filter(is_deleted=False)


class SurveyQuestionQuerySet(QuerySet):
    def filter_by_answer_type_with_files(self):
        # todo: test me
        return self.filter(answer_type__slug='answer_files')

    def exclude_with_file_answer_type(self):
        # todo: test me
        return self

    def exclude_by_answer_type_with_files(self):
        # todo: test me
        return self.exclude(answer_type__slug='answer_files')


class SurveyQuestionManager(ProxyToQuerySetMixin, models.Manager):
    def get_queryset(self):
        queryset = SurveyQuestionQuerySet(self.model, using=self._db)
        return queryset.filter(is_deleted=False)

    def check_answer_payment_exists(self, survey_pk):
        return self.model.objects.filter(
            survey=survey_pk,
            answer_type__slug='answer_payment',
            is_deleted=False,
        ).exists()


class SurveyQuestionChoiceManager(models.Manager):
    def get_queryset(self):
        queryset = super().get_queryset()
        return queryset.filter(is_deleted=False)


class SurveyQuestionMatrixTitleManager(models.Manager):
    def get_queryset(self):
        queryset = super().get_queryset()
        return queryset.filter(is_deleted=False)


class ProfileSurveyAnswerManager(ProxyToQuerySetMixin, models.Manager):
    def get_answers_count(self, delta_in_seconds):
        return self.model.objects.filter(
            date_created__gte=timezone.now() - timedelta(seconds=delta_in_seconds)
        ).count()


class ExportSurveyPageQuerySet(QuerySet):
    def filter_by_finished(self):
        return self.filter(date_finished__isnull=False)


class ExportSurveyPageManager(ProxyToQuerySetMixin, models.Manager):
    def get_queryset(self):
        return ExportSurveyPageQuerySet(self.model, using=self._db)
