# -*- coding: utf-8 -*-
import copy
import datetime
import hashlib
import hmac
import itertools
import logging
import operator
import random
import urllib.error
import urllib.parse
import urllib.request

from collections import defaultdict
from django.conf import settings
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ValidationError
from django.db import models, transaction
from django.db.utils import IntegrityError
from django.forms.models import ModelMultipleChoiceField
from django.template import Context, Template
from django.utils import timezone
from django.utils.cache import caches
from django.utils.encoding import force_str
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _
from rest_framework.exceptions import ValidationError as RestValidationError

from events.accounts.models import User, Organization
from events.accounts.utils import (
    PassportPersonalData,
    DirPersonalData,
    is_anonymous,
)

from events.common_app.fields import (
    AutoIdentifierField,
    JSONFieldForSimpleText,
    SimpleTextField,
)
from events.common_app.jsonfield.fields import JSONField as JSONFieldWithFallback
from events.common_app.utils import (
    is_duplicate_entry_error,
    get_duplicate_entry_error_model,
    timeit,
    get_lang_from_query_params,
    get_lang_from_accept_language,
    generate_code,
    class_localcache,
)
from events.conditions.models import ConditionNodeBase, ConditionItemBase
from events.data_sources.sources import data_sources_by_name
from events.followme.models import FollowersMixin
from events.media.api_admin.v2.serializers import ImageSerializer
from events.surveyme.conditions import SurveyConditions
from events.surveyme.dataclasses import ParamQuiz
from events.surveyme.managers import (
    ProfileSurveyAnswerManager,
    SurveyGroupManager,
    SurveyManager,
    SurveyQuestionManager,
    SurveyQuestionChoiceManager,
    SurveyQuestionMatrixTitleManager,
)
from events.surveyme.models_modules.states import (
    # этот импорт необходим
    SurveyStateConditionNode,
    SurveyStateConditionItem,
    ProfileSurveyAnswerStateConditionResult
)
from events.media.models import Image
from events.translation.models import TranslationModel

__all__ = [
    'SurveyStateConditionNode',
    'SurveyStateConditionItem',
    'ProfileSurveyAnswerStateConditionResult',
]
logger = logging.getLogger(__name__)

NO_GROUP_QUESTIONS_IN_LOGIC = _('You can not use group questions in logic')

ANOTHER_GROUP_QUESTION_IN_LOGIC = _('You can not use group questions from another group in logic')
STYLES_TEMPLATE_DOES_NOT_EXIST = _('Стиль шаблона формы не существует')
GROUP_QUESTION_CANT_USE_IN_SURVEY_HOOK_LOGIC = _('Вопрос с условием срабатывания интеграции не может быть внутри серии вопросов')
GROUP_QUESTION_CANT_USE_IN_SHOW_QUESTION_LOGIC = _('Вопрос с условием не может быть внутри серии вопросов')

ANSWER_TYPE_KIND_CHOICE = (
    ('generic', 'Обычный'),
    ('profile', 'Для профиля'),
)

ADMIN_PREVIEW_CHOICES = (
    ('input', 'Input'),
    ('textarea', 'Textarea'),
    ('list', 'List'),
    ('date', 'Date'),
    ('gender', 'Gender'),
    ('statement', 'Statement'),
)

EXPORT_SURVEY_CHOICES = (
    ('text/csv', 'csv'),
    ('application/excel', 'xlsx'),
)

CONDITION_CHOICES = (
    ('eq', 'равно'),
    ('neq', 'не равно'),
)

OPERATOR_CHOICES = (
    ('and', 'и'),
    ('or', 'или'),
)

INVITE_STATUS_CHOICES = (
    ('invited', 'пригласить'),
    ('refused', 'отказать'),
    ('pending', 'под вопросом'),
)

VISIT_STATUS_CHOICES = (
    ('visited',     'пришел'),
    ('not_visited', 'не пришел'),
)

SUBMISSION_SOURCE_CHOICES = (
    ('website', 'Сайт'),
    ('import', 'Импорт')
)

ANSWERS_IMPORT_STATUS_CHOICES = (
    ('validation', 'Валидация'),
    ('import', 'Импорт')
)

SURVEY_QUESTION_WIDGETS = (
    ('list', 'Список'),
    ('select', 'Селект'),
    ('matrix', 'Шкала'),
)

SURVEY_CAPTCHA_DISPLAY_MODES = (
    ('auto', 'автоматически'),
    ('always', 'всегда'),
)

SURVEY_CAPTCHA_TYPES = (
    (settings.CAPTCHA_TYPE, 'Стандартная'),
    ('ocr', 'OCR'),
)

MATRIX_TITLE_TYPES = (
    ('row', 'строка'),
    ('column', 'столбец'),
)

SURVEY_MODIFY_CHOICES = (
    ('natural', 'Не менять порядок'),
    ('shuffle', 'Перемешать'),
    ('sort', 'Отсортировать'),
)

FOLLOW_TYPE_CHOICES = (
    ('5m', '5 Minutes'),
    ('1h', '1 Hour'),
    ('1d', '1 Day'),
)


ANSWER_EXPORT_STATUS_SUCCESS = 0
ANSWER_EXPORT_STATUS_PENDING = 1
ANSWER_EXPORT_STATUS_CHOICES = (
    (ANSWER_EXPORT_STATUS_PENDING, 'Ожидается выгрузка данных'),
    (ANSWER_EXPORT_STATUS_SUCCESS, 'Данные выгружены успешно'),
)

SURVEY_STATUS_PUBLISHED = 'published'
SURVEY_STATUS_UNPUBLISHED = 'unpublished'

SURVEY_OWNERSHIP_MYFORMS = 'myforms'
SURVEY_OWNERSHIP_SHARED = 'shared'

STYLE_TEMPLATE_TYPES = (
    ('default', u'по-умолчанию'),
    ('custom', u'пользовательский'),
)


class AnswerType(models.Model):
    allow_settings = models.ManyToManyField('AnswerTypeSettings', related_name='allow_settings_answertype_set')
    required_settings = models.ManyToManyField('AnswerTypeSettings', related_name='required_settings_answertype_set')
    app_types = models.ManyToManyField('common_app.AppType', related_name='app_types')
    is_allow_choices = models.BooleanField(default=False)
    is_read_only = models.BooleanField(default=False)
    is_allow_widgets = models.BooleanField(default=False)
    is_could_be_used_in_conditions = models.BooleanField(default=False)
    slug = models.SlugField(unique=True)
    name = models.CharField(max_length=255)
    icon = models.CharField(max_length=100)
    admin_preview = models.CharField(choices=ADMIN_PREVIEW_CHOICES, max_length=100)
    kind = models.CharField(choices=ANSWER_TYPE_KIND_CHOICE, max_length=100)

    def __str__(self):
        return 'AnswerType(%s, %s)' % (self.slug, self.pk)


class SurveyAgreement(TranslationModel):
    FIELDS_FOR_TRANSLATION = ('text', )

    name = models.CharField(max_length=255)
    text = models.TextField()
    is_required = models.BooleanField(default=True)
    slug = models.SlugField(max_length=255, null=True)

    objects = SurveyGroupManager()

    class Meta:
        verbose_name = 'соглашение'
        verbose_name_plural = 'соглашения'

    def __str__(self):
        return self.name

    def get_default_language(self):
        return settings.MODELTRANSLATION_DEFAULT_LANGUAGE

    def get_text(self):
        return self.get_translated_field('text')


class SurveyStyleTemplate(models.Model):
    name = models.CharField(max_length=255, null=True)
    type = models.CharField(
        choices=STYLE_TEMPLATE_TYPES,
        max_length=32,
        default=STYLE_TEMPLATE_TYPES[0][0]
    )
    styles = JSONFieldWithFallback(null=True)
    image_page = models.ForeignKey(Image, blank=True, null=True, on_delete=models.SET_NULL, related_name='+')
    image_form = models.ForeignKey(Image, blank=True, null=True, on_delete=models.SET_NULL, related_name='+')

    def __str__(self):
        return 'SurveyStyleTemplate(%s, "%s", "%s")' % (self.pk, self.name, self.type)

    class Meta:
        verbose_name = u'стиль формы'
        verbose_name_plural = u'стили форм'


class SurveyGroup(models.Model):
    name = models.CharField(max_length=255)
    user = models.ForeignKey(User, blank=True, null=True, verbose_name='Автор (user)', on_delete=models.CASCADE)
    org = models.ForeignKey(Organization, blank=True, null=True, verbose_name='Организация Я.Коннект', on_delete=models.DO_NOTHING)
    metrika_counter_code = models.CharField('номер счетчика метрики', max_length=255, blank=True)
    users_count = models.PositiveIntegerField(default=0)  # deprecated
    groups_count = models.PositiveIntegerField(default=0)  # deprecated

    class Meta:
        verbose_name = 'Группа форм'
        verbose_name_plural = 'Группы форм'

    def __str__(self):
        return str(self.pk)

    @property
    def admin_url(self):
        return f'{settings.ADMIN_URL}/groups/{self.pk}/settings/'


class Survey(FollowersMixin, TranslationModel):
    FIELDS_FOR_TRANSLATION = ('name', )

    id = AutoIdentifierField(primary_key=True, editable=False)
    group = models.ForeignKey(SurveyGroup, blank=True, null=True, verbose_name='Группа', on_delete=models.SET_NULL)
    user = models.ForeignKey(User, blank=True, null=True, verbose_name='Автор (user)', on_delete=models.CASCADE)
    org = models.ForeignKey(Organization, blank=True, null=True, verbose_name='Организация Я.Коннект', on_delete=models.DO_NOTHING)
    type = models.CharField(max_length=100, default='simple_form')
    name = models.CharField('Название', max_length=255, blank=True)
    need_auth_to_answer = models.BooleanField(
        'Разрешить ответ на опрос только авторизированным пользователям',
        default=False
    )
    is_published_external = models.BooleanField(default=settings.IS_BUSINESS_SITE)  # опубликованность простых форм
    is_public = models.BooleanField(default=settings.IS_BUSINESS_SITE)  # признак публичных форм в Б2Б
    is_only_for_iframe = models.BooleanField('Показывать только в iframe', default=False)  # FORMS-466
    is_allow_answer_editing = models.BooleanField('Разрешить пользователю редактирование ответа', default=False)
    is_allow_multiple_answers = models.BooleanField('Разрешить делать несколько ответов на форму', default=True)
    is_allow_answer_versioning = models.BooleanField('Показывать пользователю предыдущий ответ в форме', default=False)
    # deprecated
    is_remove_answer_after_integration = models.BooleanField('Удалять ответ после срабатывания всех интеграций', default=False)
    is_send_user_notification_email_message = models.BooleanField(
        default=False,
        verbose_name='Отправить информационное сообщение на почтовый ящик в случае успешного заполнения формы'
    )
    user_notification_email_message_subject = models.CharField(
        max_length=75,
        blank=True,
        verbose_name='Заголовок сообщения об успешном заполнении формы'
    )
    user_notification_email_message_text = models.TextField(blank=True, verbose_name='Текст информационного сообщения')
    user_notification_email_frontend_name = models.CharField(null=True, blank=True, max_length=255)
    agreements = models.ManyToManyField(SurveyAgreement, blank=True, related_name='surveys')
    content_type = models.ForeignKey(ContentType, blank=True, null=True, on_delete=models.SET_NULL)
    object_id = models.PositiveIntegerField(blank=True, null=True)
    content_object = GenericForeignKey()
    cached_unicode_value = models.CharField(max_length=255, blank=True, null=True)
    is_recent_data_is_sended_to_localize = models.BooleanField(default=False)
    metrika_counter_code = models.CharField('номер счетчика метрики', max_length=255, blank=True)
    save_logs_for_statbox = models.BooleanField('Сохранять логи для статистики (statbox)', default=False)
    captcha_display_mode = models.CharField(
        'Режим показа капчи',
        choices=SURVEY_CAPTCHA_DISPLAY_MODES,
        max_length=100,
        default=SURVEY_CAPTCHA_DISPLAY_MODES[0][0]
    )
    is_spam_detected = models.NullBooleanField('Признак подозрения на спам')  # deprecated
    is_ban_detected = models.NullBooleanField('Признак бана формы')
    slug = models.SlugField(unique=True, blank=True, null=True)
    captcha_type = models.CharField(
        'Тип капчи',
        choices=SURVEY_CAPTCHA_TYPES,
        max_length=100,
        default=SURVEY_CAPTCHA_TYPES[0][0]
    )
    maximum_answers_count = models.PositiveIntegerField(blank=True, null=True)  # FORMS-493
    auto_control_publication_status = models.BooleanField(
        'Управлять статусом публикации автоматически ',
        default=False,
    )
    datetime_auto_open = models.DateTimeField(blank=True, null=True)
    datetime_auto_close = models.DateTimeField(blank=True, null=True)
    validator_url = models.CharField(max_length=255, blank=True, null=True)
    is_deleted = models.BooleanField(default=False, db_index=True)
    date_created = models.DateTimeField(auto_now_add=True, null=True)  # дата создания
    date_updated = models.DateTimeField(auto_now=True, null=True)  # дата изменения
    date_published = models.DateTimeField(blank=True, null=True)  # дата публикации формы
    date_unpublished = models.DateTimeField(blank=True, null=True)  # дата снятия с публикации
    date_archived = models.DateTimeField(blank=True, null=True)  # дата последнего переноса ответов в архив
    date_exported = models.DateTimeField(blank=True, null=True)  # дата последнего экспорта ответов в файл
    styles_template = models.ForeignKey(SurveyStyleTemplate, blank=True, null=True, on_delete=models.SET_NULL)
    users_count = models.PositiveIntegerField(default=0, null=True, blank=True)  # deprecated
    groups_count = models.PositiveIntegerField(default=0, null=True, blank=True)  # deprecated
    language = models.CharField(max_length=10, default='ru')
    redirect = JSONFieldWithFallback(blank=True, null=True)  # deprecated
    footer = JSONFieldWithFallback(blank=True, null=True)  # deprecated
    quiz = JSONFieldWithFallback(blank=True, null=True)  # deprecated
    stats = JSONFieldWithFallback(blank=True, null=True)  # deprecated
    extra = JSONFieldWithFallback(blank=True, null=True)
    follow_type = models.CharField(
        choices=FOLLOW_TYPE_CHOICES, blank=True, null=True, max_length=20,
    )

    objects = SurveyManager()
    with_deleted_objects = models.Manager()

    class Meta(TranslationModel.Meta):
        verbose_name = 'Форма'
        verbose_name_plural = 'Формы'
        index_together = (
            ('auto_control_publication_status', 'datetime_auto_open', 'is_published_external'),
            ('auto_control_publication_status', 'datetime_auto_close', 'is_published_external'),
        )

    def __str__(self):
        return str(self.pk)

    def save(self, *args, **kwargs):
        if self.is_published_external and self.date_published is None:
            self.date_published = timezone.now()
            if 'update_fields' in kwargs:
                kwargs['update_fields'].append('date_published')
        return super().save(*args, **kwargs)

    @staticmethod
    def get_spam_detected(survey_id):
        if settings.IS_BUSINESS_SITE:
            cache = caches['survey']
            return cache.get(f'spam:{survey_id}', False)
        return False

    @staticmethod
    def set_spam_detected(survey_id, status):
        if settings.IS_BUSINESS_SITE:
            cache = caches['survey']
            cache.set(f'spam:{survey_id}', status)

    @property
    def admin_url(self):
        return f'{settings.ADMIN_URL}/{self.pk}/edit/'

    def get_default_language(self):
        return self.language

    def get_name(self):
        return self.get_translated_field('name')

    def get_dir_id(self):
        return self.org and self.org.dir_id

    def get_index_value(self):
        return '%s %s' % (self.__str__(), self.pk)

    def validate_content_type(self, expected_models):
        pass

    def validate_unique_for_type_and_content_object(self):
        pass

    def clean_fields(self, *args, **kwargs):
        super(Survey, self).clean_fields(*args, **kwargs)
        self.validate_slug()

    def validate_slug(self):
        if self.slug and len(self.slug) and self.slug[0].isdigit():
            raise ValidationError({'slug': 'Слаг формы не должен начинаться с цифры.'})

    def get_frontend_url(self, is_with_site=False, external=False):
        return '{schema}://{domain}{path}'.format(
            schema='https',
            domain=settings.DEFAULT_FRONTEND_DOMAIN,
            path=f'/v1/surveys/{self.pk}/form/',
        )

    @timeit
    def get_form(self, **kwargs):
        """Возвращает форму"""
        from events.surveyme.forms import SurveyForm
        kwargs.update({'survey': self})
        return SurveyForm(**kwargs)

    def get_admin_header_emulator_url(self):
        # todo: test me
        base_url = '/admin/common_app/header/'
        params = {
            'object_type': 'survey',
            'object_id': self.pk
        }
        return base_url + '?' + urllib.parse.urlencode(params)

    def is_profile_can_answer(self, user, ignore=None, preview=False):
        return SurveyConditions.is_profile_can_answer(user=user, model_instance=self, ignore=ignore, preview=preview)

    def is_already_answered_by_profile(self, user):
        return SurveyConditions.is_already_answered_by_profile(user=user, model_instance=self)

    def why_profile_cant_answer(self, user, ignore=None, preview=False):
        if is_anonymous(user):
            user = None

        messages = {}
        if not ignore:
            ignore = []
        if 'is_ready_for_answer' not in ignore and not self.is_ready_for_answer:
            messages.update(self.why_is_not_ready_for_answer(preview))
        if not (self.is_answer_could_be_made_multiple_times
           or self.is_answer_could_be_edited
           or not self.is_already_answered_by_profile(user)):
            messages['already_answered'] = 'is already answered by this profile and answer is could not be edited'
        if (self.is_need_auth_to_answer or preview) and user is None:
            messages['need_auth'] = 'is need auth to answer'
        return messages

    def why_is_not_ready_for_answer(self, preview=False):
        # todo: test me
        messages = {}
        if not self.is_published_external:
            messages['survey_closed'] = 'survey closed'
        return messages

    def send_confirmation_email_to_profile_if_needed(self, profile, authorized_frontend_name):
        return None

    def _get_content_object_title(self):
        pass

    def create_trust_texts(self):
        texts = copy.deepcopy(settings.SURVEYME_SURVEY_DEFAULT_TEXTS_FOR_TRUST)
        return self._create_texts(texts.items())

    def create_texts(self):
        return self._create_texts(SurveyText.get_default_texts_for_survey_type(self.type).items())

    def _create_texts(self, texts):
        for slug, text in texts:
            source_value = text.get('value')
            if source_value:
                translations = {
                    'value': {
                        'ru': source_value.get('ru', ''),
                        'en': source_value.get('en', ''),
                        'tr': source_value.get('tr', ''),
                        'uk': source_value.get('uk', ''),
                    }
                }
                field_value = translations.get('value', {})
                value = field_value.get(self.get_default_language(), '')
            else:
                translations = {}
                value = ''
            null = text.get('null') or False
            max_length = text.get('max_length')

            try:
                SurveyText.objects.get_or_create(
                    slug=slug,
                    survey=self,
                    defaults={
                        'max_length': max_length,
                        'null': null,
                        'value': value,
                        'translations': translations,
                    }
                )
            except IntegrityError as e:
                if is_duplicate_entry_error(e) and get_duplicate_entry_error_model(e) == SurveyText:
                    logger.info('SurveyText race conditions protection worked', exc_info=1)
                else:
                    raise

    def get_tickets_status(self):
        from events.balance.models import Ticket

        try:
            return self.tickets_info.get_status()
        except Ticket.DoesNotExist:
            return {}

    def get_payment_question(self):
        for question in self.surveyquestion_set.all():
            if question.answer_type.slug == 'answer_payment':
                return question

    def get_submit_conditions(self, with_additional_info=False):
        from events.surveyme import utils as surveyme_utils
        condition_qs = list(self.submit_condition_nodes.all())
        if not condition_qs:
            return []

        # Оптимизация на случай если в prefetch-кеше уже содержится список вопросов
        # необходимо учесть, кеша может физически не быть
        if 'surveyquestion' not in getattr(self, '_prefetched_objects_cache', []):
            surveyquestion_qs = self.surveyquestion_set.select_related('answer_type')
        else:
            surveyquestion_qs = self.surveyquestion_set.all()

        questions_dict = {
            question.id: question
            for question in surveyquestion_qs
        }
        return surveyme_utils.get_condition_nodes(
            self,
            questions_dict=questions_dict,
            condition_queryset=condition_qs,
            with_additional_info=with_additional_info,
        )

    def sync_questions(self):
        from events.surveyme import utils as surveyme_utils
        with transaction.atomic():
            for q in surveyme_utils.sync_questions(self.surveyquestion_set.all()):
                SurveyQuestion.objects.filter(pk=q.pk).update(page=q.page, position=q.position)

    @property
    def publication_status(self):
        if self.is_published_external:
            return SURVEY_STATUS_PUBLISHED
        else:
            return SURVEY_STATUS_UNPUBLISHED

    @property
    def free_tickets_count(self):
        return self.surveyticket_set.filter(acquired=False).count()

    @property
    def is_need_auth_to_answer(self):
        return SurveyConditions.is_need_auth_to_answer(model_instance=self)

    @property
    def is_answer_could_be_edited(self):
        return SurveyConditions.is_answer_could_be_edited(model_instance=self)

    @property
    def is_answer_could_be_made_multiple_times(self):
        return SurveyConditions.is_answer_could_be_made_multiple_times(model_instance=self)

    @property
    def is_ready_for_answer(self):
        return SurveyConditions.is_ready_for_answer(model_instance=self)

    @property
    def is_has_questions(self):
        return SurveyConditions.is_has_questions(model_instance=self)

    @property
    def is_action_is_ready_for_registration(self):
        return SurveyConditions.is_action_is_ready_for_registration(model_instance=self)

    @property
    def questions_count(self):
        return self.surveyquestion_set.count()

    @property
    def hashed_id(self):
        if (
            settings.APP_TYPE in ('forms_ext', 'forms_ext_admin')
            and self.id >= settings.SURVEY_HASH_START_ID
        ):
            encoded_id = str(self.id).encode()
            encoded_secret = settings.SURVEY_HASH_SECRET.encode()
            h = hmac.new(encoded_secret, encoded_id, hashlib.sha1)
            return '{}.{}'.format(self.id, h.hexdigest())

    @property
    @class_localcache
    def total_scores(self):
        total_scores = sum(
            question.get_scores()
            for question in self.surveyquestion_set.all()
        )
        return round(total_scores, 2)

    @property
    @class_localcache
    def quiz_question_count(self):
        return sum(
            1
            for question in self.surveyquestion_set.all()
            if question.has_quiz()
        )

    def prepare_answers_to_export(self):
        from events.surveyme import tasks
        tasks.prepare_answers_to_export.delay(self.pk)

    def remove_answers_from_export(self):
        from events.surveyme import tasks
        params = {
            'args': (self.pk,),
            'countdown': settings.YT_ANSWERS_REMOVE_COUNTDOWN,  # задержка перед удалением данных
        }
        tasks.remove_answers_from_export.apply_async(**params)
        tasks.remove_table_from_yt.apply_async(**params)

    @property
    def languages(self):
        from django.contrib.contenttypes.models import ContentType
        from events.tanker.models import TankerKeyset

        if settings.IS_BUSINESS_SITE:
            return None

        ct = ContentType.objects.get_for_model(Survey)
        qs = TankerKeyset.objects.filter(object_id=self.id, content_type=ct).first()
        if qs is not None:
            return qs.languages


class SurveyTicket(models.Model):
    """
        # FORMS-493
        Нужен для ограничения максимального количества ответов на Survey.
        При новом ответе на форму (не редактировании ответа) мы выбираем свободный билет и лочим его на запись.
        Если билетов нет - закрываем регистрацию.
        Если билет есть - ставим ему acquired = True.
        Это лучше, чем хранить счетчик внутри Survey - потому что тогда придется блокировать его на запись
        чтобы избежать эффекта гонки и одновременно сможет отвечать лишь один пользователь.
    """
    acquired = models.BooleanField(default=False)
    survey = models.ForeignKey(Survey, on_delete=models.CASCADE)

    class Meta:
        index_together = (
            ('survey', 'acquired'),
        )

    def __str__(self):
        return 'Ticket %s. Survey: %s. Acquired: %s' % (self.pk, self.survey, self.acquired)


class SurveyQuestionHintType(models.Model):
    name = models.CharField(max_length=255)
    slug = models.SlugField()


class ValidatorType(models.Model):
    name = models.CharField(max_length=255)
    slug = models.SlugField(unique=True)
    answer_types = models.ManyToManyField('AnswerType', related_name='validator_types')
    is_external = models.BooleanField(default=False)


def get_show_conditions(survey):
    return (
        SurveyQuestionShowConditionNodeItem.objects.using(settings.DATABASE_ROLOCAL)
        .filter(
            survey_question_show_condition_node__survey_question__survey=survey,
        )
        .values_list(
            'survey_question_show_condition_node__survey_question',
            'survey_question',
        )
    )


def get_move_error_message(question_with_logic, question_with_new_position):
    return ('Не удалось выполнить перемещение. Логика вопроса «{question_with_logic}» зависит от '
            'ответа на вопрос «{question_with_new_position}». В случае перемещения логика показа будет зависеть от '
            'ответа на вопрос, который находится ниже. '
            '\n\n'
            'Удалите зависимость логики показа вопроса '
            '«{question_with_logic}» от ответа на вопрос «{question_with_new_position}», либо не '
            'выполняйте такого перемещения.'.format(
        question_with_logic=question_with_logic,
        question_with_new_position=question_with_new_position
    ))


class SurveyQuestion(TranslationModel):
    FIELDS_FOR_TRANSLATION = ('label', 'param_help_text')

    label = models.TextField()
    survey = models.ForeignKey(Survey, on_delete=models.CASCADE)
    group = models.ForeignKey(
        'self', related_name='group_questions',
        blank=True, null=True, on_delete=models.CASCADE,
    )
    answer_type = models.ForeignKey(AnswerType, on_delete=models.CASCADE)
    validator_type = models.ForeignKey(ValidatorType, blank=True, null=True, on_delete=models.DO_NOTHING)
    validator_options = JSONFieldWithFallback(blank=True, null=True)
    position = models.PositiveIntegerField(default=0)
    page = models.PositiveIntegerField(default=1)
    initial = JSONFieldForSimpleText(blank=True, null=True)
    is_deleted = models.BooleanField(default=False, db_index=True)
    label_image = models.ForeignKey(Image, blank=True, null=True, on_delete=models.CASCADE)

    param_is_required = models.BooleanField(default=True)
    param_is_allow_multiple_choice = models.BooleanField(default=False)
    param_is_allow_other = models.BooleanField(default=False)
    param_help_text = models.TextField(blank=True)
    param_is_section_header = models.BooleanField(default=False)
    param_max_file_size = models.PositiveSmallIntegerField('максимальный суммарный размер файлов (мегабайты)', default=20)
    param_max_files_count = models.PositiveSmallIntegerField('максимальное количество файлов', default=20)
    param_max = models.IntegerField('максимальное количество', blank=True, null=True)
    param_min = models.IntegerField('минимальное количество', blank=True, null=True)
    param_price = models.PositiveIntegerField('цена билета', blank=True, null=True)
    param_variables = SimpleTextField(blank=True, null=True)
    param_widget = models.CharField(max_length=255, choices=SURVEY_QUESTION_WIDGETS, default=SURVEY_QUESTION_WIDGETS[0][0])
    param_is_hidden = models.BooleanField(default=False)
    param_is_random_choices_position = models.BooleanField('Перемешивать варианты ответов', default=False)
    param_modify_choices = models.CharField(max_length=64, choices=SURVEY_MODIFY_CHOICES, default=SURVEY_MODIFY_CHOICES[0][0], blank=True)
    param_suggest_choices = models.BooleanField(default=False)
    param_slug = models.SlugField(blank=True, null=True)
    param_hint_type = models.ForeignKey(SurveyQuestionHintType, blank=True, null=True, on_delete=models.CASCADE)  # todo: remove me
    param_is_disabled_init_item = models.BooleanField('Первый пункт без значения', default=True)  # FORMS-455

    param_data_source = models.CharField(blank=True, null=True, max_length=100, default='survey_question_choice')
    param_data_source_params = JSONFieldForSimpleText(blank=True, null=True)

    param_hint_data_source = models.CharField(blank=True, null=True, max_length=100)
    param_hint_data_source_params = JSONFieldForSimpleText(blank=True, null=True)

    param_date_field_type = models.CharField(default='date', max_length=100)  # 'date' or 'daterange'
    param_date_field_min = models.DateField(blank=True, null=True)
    param_date_field_max = models.DateField(blank=True, null=True)
    param_payment = JSONFieldWithFallback(blank=True, null=True)
    param_quiz = JSONFieldWithFallback(blank=True, null=True)

    objects = SurveyQuestionManager()
    with_deleted_objects = models.Manager()

    class Meta:
        verbose_name = 'Вопрос'
        verbose_name_plural = 'Вопросы'
        ordering = ('page', 'position', 'id')
        unique_together = (
            ('survey', 'param_slug'),
        )

    def __str__(self):
        return self.label

    def has_quiz(self):
        if not self.param_quiz:
            return False
        return bool(self.param_quiz.get('enabled'))

    def get_scores(self):
        if not self.has_quiz():
            return 0.0
        param_quiz = ParamQuiz(self.param_quiz)
        # на случай пустого массива, чтобы функция max не бросала
        # ValueError добавляем значение по-умолчанию 0.0
        method = lambda it: max(itertools.chain([0.0], it))
        if self.param_is_allow_multiple_choice:
            method = sum
        return method(
            answer.scores
            for answer in param_quiz.answers or []
            if answer.correct
        )

    def get_default_language(self):
        return self.survey.get_default_language()

    def get_label(self):
        return self.get_translated_field('label') or self.param_slug

    def get_param_help_text(self):
        return self.get_translated_field('param_help_text')

    def save(self, *args, **kwargs):
        self.validate_model()
        self.set_position()
        return super(SurveyQuestion, self).save(*args, **kwargs)

    def get_hint_data_source_class(self):
        # todo: test me
        return data_sources_by_name[self.param_hint_data_source]

    @cached_property
    def param_has_logic(self):
        return self.show_condition_nodes.filter(items__survey_question__is_deleted=False).exists()

    def get_data_source_class(self):
        # todo: test me
        return data_sources_by_name[self.param_data_source]

    def clean_fields(self, *args, **kwargs):
        super(SurveyQuestion, self).clean_fields(*args, **kwargs)
        self.validate_param_min_and_param_max_values()
        self.validate_param_min_and_max_date_values()
        self.validate_param_slug()

    def validate_param_min_and_param_max_values(self):
        if self.param_max is not None and self.param_min is not None and self.param_min > self.param_max:
            min_msg = 'Минимальное значение должно быть меньше или равно максимальному'
            max_msg = 'Максимальное значение должно быть больше или равно минимальному'
            raise ValidationError({'param_min': [min_msg], 'param_max': [max_msg]})

    def validate_param_slug(self):
        if self.answer_type.kind == 'generic' and self.param_slug and self.param_slug.lower().startswith('param_'):
            raise ValidationError({
                'param_slug': 'Не давайте идентификаторам вопросов значения, начинающиеся на param_. '
                               'Они могут пересечься идентификаторами полей профиля.'
            })

    def validate_param_min_and_max_date_values(self):
        if self.param_date_field_min and self.param_date_field_max:
            if self.param_date_field_min >= self.param_date_field_max:
                raise ValidationError({
                    'param_date_field_min': 'Минимальная дата диапазона должна быть меньше максимальной'
                })

    def validate_group(self):
        if self.group_id:
            if self.answer_type.slug == 'answer_group':
                raise ValidationError(_('You cannot add group to another group'))
            if self.id and self.group_id == self.id:
                raise ValidationError(_('You cannot add group to itself'))
            try:
                answer_type_slug = (SurveyQuestion.objects
                                    .select_related('answer_type')
                                    .get(pk=self.group_id)
                                    .answer_type.slug
                                    )
            except SurveyQuestion.DoesNotExist:
                raise ValidationError(_('Question with such id does not exists'))
            if answer_type_slug != 'answer_group':
                raise ValidationError(_('You can only specify group question in this field'))

    def validate_model(self):
        self.validate_group()
        if self.answer_type.kind == 'profile':
            self.validate_profile_question()
        else:
            self.validate_generic_question()

    def validate_profile_question(self):
        """На каждый опрос может быть только 1 вариант ответа по определенному полю профиля

        """
        if SurveyQuestion.objects.db_manager('default').filter(
            survey=self.survey,
            answer_type=self.answer_type
        ).exclude(pk=self.pk).exists():
            raise ValidationError(
                'For same survey you can create only one profile question with answer_type.slug={0}'
                .format(self.answer_type.slug)
            )

    def validate_generic_question(self):
        if self.param_max is not None and self.param_min is not None and self.param_min > self.param_max:
            raise ValidationError('Минимальное значение должно быть меньше или равно максимальному')

    def set_position(self):
        # todo: test me
        if not self.pk and self.position == 0:
            self.position = SurveyQuestion.objects.filter(survey=self.survey, page=self.page).count() + 1

    def change_position(self, position, page=None, group_id=None):
        page = page or self.page

        if position <= 0:
            raise RestValidationError('Позиция вопроса не должна быть меньше или равна нулю')

        # проверяем, что код для серии вопросов не противоречив
        group = None
        if group_id is not None:
            try:
                group = SurveyQuestion.objects.get(pk=group_id)
            except SurveyQuestion.DoesNotExist:
                raise RestValidationError('Серия вопросов не существует')
            if group.survey_id != self.survey_id:
                raise RestValidationError('Изменяемый вопрос и серия не могут принадлежать разным формам')
            if self.answer_type.slug == 'answer_group':
                raise RestValidationError('Серия вопросов не может находиться в другой серии')
            page = group.page

        # запрашиваем список вопросов
        questions = {
            question.pk: question
            for question in SurveyQuestion.objects.filter(survey_id=self.survey_id).exclude(pk=self.pk)
        }
        questions[self.pk] = self

        # для вопросов вне серий, расставляем их по порядку страниц, номеров позиций
        # для вопросов внутри серий, заводим отдельный словарь со списками
        page_key = lambda question: question.page
        position_key = lambda question: question.position
        pages = []
        groups = defaultdict(list)
        for p, page_group in itertools.groupby(sorted(questions.values(), key=page_key), key=page_key):
            page_positions = []
            for question in sorted(page_group, key=position_key):
                if question.group_id is None:
                    if question != self:
                        page_positions.append(question)
                else:
                    if question != self:
                        groups[question.group_id].append(question)
            pages.append(page_positions)

        # вставляем вопрос в нужное место
        if group_id is None:
            if page > len(pages):
                pages.append([self])
            else:
                pages[page - 1].insert(position - 1, self)
        else:
            groups[group_id].insert(position - 1, self)

        # убираем пустые страницы
        pages = [positions for positions in pages if positions]

        # для вопросов вне серий вопросов, меняем номера страниц/позиции
        changed = []
        for new_page, positions in enumerate(pages, start=1):
            for new_position, question in enumerate(positions, start=1):
                if (
                    question.page != new_page
                    or question.position != new_position
                    or question.group_id is not None
                ):
                    question.page = new_page
                    question.position = new_position
                    question.group_id = None
                    changed.append(question)

        # для вопросов внутри серий, меняем номера страниц/позиции
        for grouping_question_pk, positions in groups.items():
            grouping_question = questions[grouping_question_pk]
            for new_position, question in enumerate(positions, start=1):
                if (
                    question.page != grouping_question.page
                    or question.position != new_position
                    or question.group_id != grouping_question.pk
                ):
                    question.page = grouping_question.page
                    question.position = new_position
                    question.group_id = grouping_question.pk
                    changed.append(question)

        # проверяем непротиворечивость условий
        condition_qs = (
            SurveyQuestionShowConditionNodeItem.objects.using(settings.DATABASE_ROLOCAL)
            .filter(
                survey_question_show_condition_node__survey_question__survey=self.survey_id,
            )
            .values_list(
                'survey_question_show_condition_node__survey_question_id',
                'survey_question_id',
            )
        )
        for source_id, target_id in condition_qs:
            source = questions.get(source_id)
            target = questions.get(target_id)
            if not source or not target:
                continue
            if source.group_id is None and target.group_id is not None:
                raise RestValidationError('Вопрос вне серии не может зависеть от вопроса в серии')
            elif source.page < target.page:
                raise RestValidationError('Вопрос в условии не может быть на следующей странице')
            elif source.group_id is not None:
                if target.group_id is not None:
                    if source.group_id != target.group_id:
                        raise RestValidationError('Вопрос в серии не может зависеть от вопроса в другой серии')
                    elif source.position < target.position:
                        raise RestValidationError('Вопрос в условии не может быть ниже на той же странице')
                elif source.page == target.page and questions.get(source.group_id).position < target.position:
                    raise RestValidationError('Вопрос в условии не может быть ниже на той же странице')
            elif source.page == target.page and source.position < target.position:
                raise RestValidationError('Вопрос в условии не может быть ниже на той же странице')

        # записываем измененные данные
        for question in changed:
            SurveyQuestion.objects.filter(pk=question.pk).update(
                page=question.page,
                position=question.position,
                group_id=question.group_id,
            )

    def generate_next_slug(self, next_counter=0):
        # todo: test me
        if self.answer_type.kind == 'generic':
            value = '{0}_{1}'.format(self.answer_type.slug, self.id)
        elif self.answer_type.kind == 'profile':
            value = self.answer_type.slug
        else:
            raise Exception('Unknown answer type')
        if next_counter:
            value += '_%s' % next_counter
        return value

    def get_form_field_name(self, group_counter=None):
        # todo: test me
        slug = self.param_slug
        if group_counter is not None:
            slug = '{}__{}'.format(force_str(slug), force_str(group_counter))
        return slug

    @timeit
    def get_show_conditions(self, questions_dict, field_name, field, with_additional_info=False):
        from events.surveyme import utils as surveyme_utils
        return surveyme_utils.get_condition_nodes(
            self,
            questions_dict=questions_dict,
            condition_queryset=self.show_condition_nodes.all(),
            with_additional_info=with_additional_info,
            field_name=field_name,
            field=field,
        )

    def render_variables(self, answer):
        if self.param_variables:
            template_text = self.param_variables.format(**dict([
                (key, '{%% variable config=config_%s answer=answer %%}' % key)
                for key in list(self.variables.keys())
            ]))
            context_data = {
                'answer': answer
            }
            for key, value in list(self.variables.items()):
                context_data['config_%s' % key] = value
            template_text = '{% load variable %}' + template_text
            return Template(template_text).render(context=Context(context_data))
        else:
            return ''

    def get_hint_data(self, user, questions_dict):
        # todo: test me
        return self._get_data_from_data_source_instance(
            self.get_hint_data_source_class()(user=user),
            params=self.param_hint_data_source_params,
            questions_dict=questions_dict,
        )

    @timeit
    def get_data_source_data(self, user, questions_dict, group_position_counter=None):
        # todo: test me
        return self._get_data_from_data_source_instance(
            self.get_data_source_class()(user=user),
            params=self.param_data_source_params,
            questions_dict=questions_dict,
            group_position_counter=group_position_counter
        )

    def _get_data_from_data_source_instance(self, instance, params, questions_dict, group_position_counter=None):
        # todo: test me
        if params is None:
            params = {}
        if not params and self.answer_type.slug == 'answer_choices':
            params = {
                'filters': [{
                    'type': 'specified_value',
                    'value': self.pk,
                    'filter': {'name': 'question'},
                }]
            }
        if self.answer_type.slug == 'answer_choices' and self.param_data_source == 'survey_question_choice':
            items = [
                {
                    'id': force_str(item.pk),
                    'text': item.get_label(),
                    'slug': item.slug,
                    'label_image': (
                        ImageSerializer(item.label_image).data
                        if item.label_image
                        else None
                    ),
                }
                for item in self.surveyquestionchoice_set.all()
                if not item.is_hidden
            ]
        elif self.answer_type.slug == 'answer_choices' and self.param_data_source == 'survey_question_matrix_choice':
            items = [
                {
                    'id': force_str(item.pk),
                    'text': item.get_label(),
                    'position': item.position,
                    'type': item.type,
                }
                for item in self.surveyquestionmatrixtitle_set.all()
            ]
        elif instance.is_with_pagination:
            return {
                'uri': instance.get_uri(),
                'content_type': None,
                'is_with_pagination': True,
                "filters": [self._prepare_filter(i, questions_dict, group_position_counter) for i in params.get('filters', [])],
            }
        else:
            items = [
                i for i in
                instance.serializer_class(
                    instance.get_filtered_queryset(
                        filter_data={
                            i['filter']['name']:i['value']
                            for i in params.get('filters', [])
                            if i['type'] == 'specified_value'
                        }
                    ),
                    many=True
                ).data
            ]
        if self.param_modify_choices == 'shuffle':
            random.shuffle(items)
        elif self.param_modify_choices == 'sort':
            items.sort(key=lambda it: it['text'])
        return {
            'content_type': instance.get_content_type_id(),
            'items': items
        }

    def _prepare_filter(self, filter, questions_dict, group_position_counter):
        # todo: test me
        response = {
            'type': filter['type'],
            'param_name': filter['filter']['name'],
        }
        if filter.get('value'):
            response['value'] = force_str(filter['value'])
        if filter.get('field'):
            question = questions_dict.get(int(filter['field']))
            if question:
                field_name = question.get_form_field_name()
                if group_position_counter is not None and self.group_id and question.group_id == self.group_id:
                    field_name = question.get_form_field_name(group_position_counter)

                response['field'] = field_name
            else:
                response['field'] = force_str(filter['field'])
        return response

    def add_additional_post_data(self, text_value, item_body):
        from events.surveyme.api_admin.v2.serializers import SurveyQuestionChoiceSerializer
        if self.answer_type.slug == 'answer_choices':
            item_body['question']['choices'] = {
                choice['label']: choice for choice in
                SurveyQuestionChoiceSerializer(
                    self.surveyquestionchoice_set.all(), many=True
                ).data
            }
            if isinstance(text_value, dict):
                choice_id = {
                    group: ','.join(
                        str(item_body['question']['choices'].get(choice.strip(), {}).get('id'))
                        for choice in choices.split(',')
                    )
                    for group, choices in text_value.items()
                }
            else:
                choice_id = item_body['question']['choices'].get(text_value, {}).get('id')
            item_body['choice_id'] = choice_id

    def get_choices(self):
        if self.answer_type.slug == 'answer_choices' and self.param_data_source == 'survey_question_choice':
            return {
                pk: label
                for pk, label in self.surveyquestionchoice_set.values_list('pk', 'label')
            }
        return {}

    def get_matrixtitles(self):
        if self.answer_type.slug == 'answer_choices' and self.param_data_source == 'survey_question_matrix_choice':
            return {
                pk: (type, label)
                for pk, type, label in self.surveyquestionmatrixtitle_set.values_list('pk', 'type', 'label')
            }
        else:
            return {}

    def get_answer_options(self):
        options = {}
        if self.param_is_required:
            options['required'] = True
        if self.answer_type.slug == 'answer_date':
            options['date_range'] = self.param_date_field_type == 'daterange'
        if self.answer_type.slug == 'answer_choices':
            options['data_source'] = self.param_data_source
            options['multiple'] = self.param_is_allow_multiple_choice
            options['ordering'] = self.param_modify_choices
        if self.answer_type.slug == 'answer_payment':
            param_payment = self.param_payment or {}
            options['account_id'] = param_payment.get('account_id')
        return options

    def get_answer_info(self):
        return {
            'id': self.pk,
            'slug': self.param_slug,
            'options': self.get_answer_options(),
            'answer_type': {
                'id': self.answer_type.pk,
                'slug': self.answer_type.slug,
            },
        }


class SurveyQuestionExportMetadata:
    def __init__(self, question):
        self.question_pk = question.pk
        self.old_metadata = {}
        self.new_metadata = {}
        self._update_metadata(self.old_metadata, question)

    def _update_metadata(self, metadata, question):
        metadata['question'] = self._get_question_metadata(question)
        metadata['choices'] = self._get_choices_metadata(question)
        metadata['matrixtitles'] = self._get_matrixtitles_metadata(question)

    def _get_question_metadata(self, question):
        return {
            'answer_type_slug': question.answer_type.slug,
            'label': question.label,
            'param_data_source': question.param_data_source,
            'param_widget': question.param_widget,
            'param_is_allow_multiple_choice': question.param_is_allow_multiple_choice,
        }

    def _get_choices_metadata(self, question):
        if question.answer_type.slug == 'answer_choices' \
                and question.param_data_source == 'survey_question_choice':
            return {
                pk: label
                for pk, label in question.surveyquestionchoice_set.values_list('pk', 'label')
            }
        return {}

    def _get_matrixtitles_metadata(self, question):
        if question.answer_type.slug == 'answer_choices' \
                and question.param_data_source == 'survey_question_matrix_choice':
            result = defaultdict(dict)
            for type, pk, label in question.surveyquestionmatrixtitle_set.values_list('type', 'pk', 'label'):
                result[type][pk] = label
            return result
        else:
            return {}

    def add_question(self, question):
        self.question_pk = question.pk
        self._update_metadata(self.new_metadata, question)

    def _check_if_questions_equal(self):
        return self.old_metadata.get('question') == self.new_metadata.get('question')

    def _get_need_to_upload(self, old, new):
        old_keys, new_keys = set(old.keys()), set(new.keys())
        need_to_upload = old_keys.symmetric_difference(new_keys)
        for key in old_keys.intersection(new_keys):
            if old.get(key) != new.get(key):
                need_to_upload.add(key)
        return need_to_upload

    def _get_choices_need_to_upload(self):
        old, new = self.old_metadata.get('choices', {}), self.new_metadata.get('choices', {})
        return self._get_need_to_upload(old, new)

    def _get_matrixtitles_need_to_upload(self):
        old = self.old_metadata.get('matrixtitles', {})
        old_row = old.get('row', {})
        old_column = old.get('column', {})

        new = self.new_metadata.get('matrixtitles', {})
        new_row = new.get('row', {})
        new_column = new.get('column', {})

        need_to_upload = set()
        old_pairs = set(
            (row_id, column_id)
            for row_id in old_row.keys()
            for column_id in old_column.keys()
        )
        new_pairs = set(
            (row_id, column_id)
            for row_id in new_row.keys()
            for column_id in new_column.keys()
        )
        for row_id, column_id in old_pairs.symmetric_difference(new_pairs):
            need_to_upload.add(row_id)
            need_to_upload.add(column_id)
        for row_id, column_id in old_pairs.intersection(new_pairs):
            if old_row[row_id] != new_row[row_id] or old_column[column_id] != new_column[column_id]:
                need_to_upload.add(row_id)
                need_to_upload.add(column_id)
        return need_to_upload


class SurveyQuestionMatrixTitle(TranslationModel):
    FIELDS_FOR_TRANSLATION = ('label', )

    survey_question = models.ForeignKey(SurveyQuestion, on_delete=models.CASCADE)
    label = models.CharField(max_length=255)
    type = models.CharField(choices=MATRIX_TITLE_TYPES, max_length=255)
    position = models.SmallIntegerField(default=1)
    is_deleted = models.BooleanField(default=False, db_index=True)

    objects = SurveyQuestionMatrixTitleManager()
    with_deleted_objects = models.Manager()

    def get_default_language(self):
        return self.survey_question.get_default_language()

    def get_label(self):
        return self.get_translated_field('label')


class SurveyText(TranslationModel):
    FIELDS_FOR_TRANSLATION = ('value', )

    survey = models.ForeignKey(Survey, related_name='texts', on_delete=models.CASCADE)
    slug = models.SlugField()
    max_length = models.PositiveSmallIntegerField(null=True)
    null = models.BooleanField(default=False)
    value = models.TextField(blank=True)

    class Meta:
        verbose_name = 'сообщение для опроса'
        unique_together = (('survey', 'slug'),)

    def get_default_language(self):
        return self.survey.get_default_language()

    def get_value(self):
        return self.get_translated_field('value')

    def clean_fields(self, *args, **kwargs):
        super(SurveyText, self).clean_fields(*args, **kwargs)
        self.validate_value_for_max_length()
        self.validate_value_for_null()

    def validate_value_for_max_length(self):
        if self.max_length is not None and self.value and len(self.value) > self.max_length:
            msg = 'Максимальная длина сообщения не должна превышать %s символов.' % self.max_length
            raise ValidationError({'value': [msg]})

    def validate_value_for_null(self):
        value = (self.value or '').strip()
        if not self.null and not value:
            msg = 'Это поле не может быть пустым.'
            raise ValidationError({'value': [msg]})

    @classmethod
    def get_default_texts_for_survey_type(cls, survey_type=None):
        return settings.SURVEYME_SURVEY_DEFAULT_TEXTS_BY_SURVEY_TYPE.get(
            survey_type, settings.SURVEYME_SURVEY_DEFAULT_TEXTS
        )


class AnswerTypeSettings(models.Model):
    column_name = models.SlugField(max_length=50, primary_key=True)


class SurveyQuestionChoice(TranslationModel):
    FIELDS_FOR_TRANSLATION = ('label', )

    survey_question = models.ForeignKey(SurveyQuestion, on_delete=models.CASCADE)
    label = models.CharField(max_length=1000)
    position = models.PositiveSmallIntegerField(default=1)
    slug = models.SlugField(_('slug'), blank=True, null=True)
    is_hidden = models.BooleanField('скрытое поле', default=False)
    label_image = models.ForeignKey(Image, blank=True, null=True, on_delete=models.CASCADE)
    is_deleted = models.BooleanField(default=False, db_index=True)

    objects = SurveyQuestionChoiceManager()
    with_deleted_objects = models.Manager()

    class Meta:
        verbose_name = 'Вариант ответа'
        verbose_name_plural = 'Варианты ответов'
        ordering = ('position', 'id')
        unique_together = [('survey_question', 'slug')]

    def __str__(self):
        return self.label

    def get_default_language(self):
        return self.survey_question.get_default_language()

    def get_label(self):
        return self.get_translated_field('label')


class AnswerExportYtStatus(models.Model):
    answer = models.OneToOneField(
        'ProfileSurveyAnswer',
        primary_key=True,
        on_delete=models.CASCADE,
        related_name='export_yt_status',
    )
    exported = models.BooleanField(default=False, db_index=True)


class ProfileSurveyAnswer(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, null=True)
    survey = models.ForeignKey(Survey, on_delete=models.CASCADE)

    source_request = JSONFieldForSimpleText(blank=True, null=True)

    # код идентификации ответа
    secret_code = models.CharField(
        max_length=255,
        default=generate_code,
        unique=True,
    )

    date_created = models.DateTimeField(auto_now_add=True, db_index=True)  # дата создания
    date_updated = models.DateTimeField(auto_now=True, db_index=True)  # дата изменения

    data = JSONFieldWithFallback(null=True)

    objects = ProfileSurveyAnswerManager()

    class Meta:
        verbose_name = 'Ответы пользователя на опрос'
        verbose_name_plural = 'Ответы пользователей на опросы'
        ordering = ['-date_updated']

    def as_dict(self):
        data = self.data or {}
        return {
            it.get('question', {}).get('id'): it
            for it in data.get('data') or []
        }

    def get_success_message_subject(self):
        # todo: test me unit
        return self.survey.user_notification_email_message_subject or 'Подтверждение регистрации'

    @timeit
    def rebuild_states_and_states_rating(self):
        pass

    def get_answer_language(self):
        if self.source_request:
            return (
                self.source_request.get('lang')
                or get_lang_from_query_params(self.source_request)
                or get_lang_from_accept_language(self.source_request)
            )

    def update_source_request(self, data, commit=True):
        if self.source_request:
            self.source_request.update(data)
        else:
            self.source_request = data
        if commit:
            self.save()

    @property
    @class_localcache
    def personal_data(self):
        if not self.user.uid and not self.user.cloud_uid:
            return None

        if settings.IS_BUSINESS_SITE:
            if not self.survey.org:
                return None
            return DirPersonalData(self.user.uid, self.user.cloud_uid, self.survey.org.dir_id)

        return PassportPersonalData(self.user.uid)


class ProfileSurveyAnswerCounter(models.Model):  # deprecated
    survey = models.OneToOneField(Survey, primary_key=True, on_delete=models.DO_NOTHING)
    answers_count = models.PositiveIntegerField(default=0)
    date_updated = models.DateTimeField(default=datetime.datetime(1970, 1, 1, tzinfo=timezone.utc))


def patched_has_changed_method(self, initial, data):
    if initial is None:
        initial = []
    if data is None:
        data = []
    if not isinstance(initial, list):
        initial = [initial]
    if len(initial) != len(data):
        return True
    initial_set = set(force_str(value) for value in self.prepare_value(initial))
    data_set = set(force_str(value) for value in data)
    return data_set != initial_set


ModelMultipleChoiceField.has_changed = patched_has_changed_method


class SurveyQuestionConditionItem(models.Model):
    operator = models.CharField(choices=OPERATOR_CHOICES, default=OPERATOR_CHOICES[0][0], max_length=100)
    survey_question = models.ForeignKey(SurveyQuestion, on_delete=models.CASCADE)
    condition = models.CharField(choices=CONDITION_CHOICES, default=CONDITION_CHOICES[0][0], max_length=100)
    survey_question_choice = models.ForeignKey(SurveyQuestionChoice, on_delete=models.CASCADE)
    position = models.PositiveSmallIntegerField(default=1)

    class Meta:
        verbose_name = 'condition node item'
        abstract = True
        ordering = ('position',)


class SurveyConditionNodeBase(models.Model):
    position = models.PositiveSmallIntegerField(default=1)

    class Meta:
        abstract = True
        ordering = ('position',)

    def is_true(self, questions, evaluated_conditions):
        state = None
        for item in self.items.all():
            choice_condition = (
                item.survey_question in questions and
                item.survey_question_choice in questions[item.survey_question] and
                evaluated_conditions.get(item.survey_question.get_form_field_name(), True)
            )
            if item.condition == 'neq':
                choice_condition = not choice_condition
            if state is None:
                state = True
            state = getattr(operator, '%s_' % item.operator.lower())(state, choice_condition)
        return bool(state)


# для опросов
class SurveyQuestionShowConditionNode(ConditionNodeBase):
    survey_question = models.ForeignKey(SurveyQuestion, related_name='show_condition_nodes', on_delete=models.CASCADE)

    @property
    def survey(self):
        return self.survey_question.survey


class SurveyQuestionShowConditionNodeItem(ConditionItemBase):
    survey_question_show_condition_node = models.ForeignKey(SurveyQuestionShowConditionNode, related_name='items', on_delete=models.CASCADE)
    survey_question = models.ForeignKey(SurveyQuestion, blank=True, null=True, on_delete=models.CASCADE)

    # deprecated fields
    survey_question_choice = models.ForeignKey(SurveyQuestionChoice, blank=True, null=True, on_delete=models.CASCADE)

    class Meta(ConditionItemBase.Meta):
        verbose_name = 'question show condition item'

    @property
    def survey(self):
        return self.survey_question_show_condition_node.survey


# Для условий возможности сабмита формы
class SurveySubmitConditionNode(ConditionNodeBase):
    survey = models.ForeignKey(Survey, related_name='submit_condition_nodes', on_delete=models.CASCADE)


class SurveySubmitConditionNodeItem(ConditionItemBase):
    survey_submit_condition_node = models.ForeignKey(SurveySubmitConditionNode, related_name='items', on_delete=models.CASCADE)

    # deprecated fields
    survey_question = models.ForeignKey(SurveyQuestion, blank=True, null=True, on_delete=models.CASCADE)
    survey_question_choice = models.ForeignKey(SurveyQuestionChoice, blank=True, null=True, on_delete=models.CASCADE)

    @property
    def survey(self):
        return self.survey_submit_condition_node.survey


class SurveyTemplate(TranslationModel):
    FIELDS_FOR_TRANSLATION = ('name', 'description')

    name = models.CharField(max_length=250)
    description = models.CharField(max_length=500, blank=True, null=True)
    slug = models.SlugField(unique=True)
    data = JSONFieldForSimpleText()
    user = models.ForeignKey(User, blank=True, null=True, verbose_name='Автор', on_delete=models.CASCADE)
    is_personal = models.BooleanField(default=False)
    position = models.PositiveIntegerField()
    image = models.ForeignKey(Image, blank=True, null=True, on_delete=models.SET_NULL)
    image_admin = models.ForeignKey(Image, blank=True, null=True, on_delete=models.SET_NULL, related_name='+')
    connect_template = models.BooleanField(default=False)

    class Meta:
        ordering = ['position']
        verbose_name = u'шаблон формы'
        verbose_name_plural = u'шаблоны форм'

    def get_default_language(self):
        return settings.MODELTRANSLATION_DEFAULT_LANGUAGE
