# -*- coding: utf-8 -*-

import random
import time
from collections import deque
from itertools import chain
from django.db import connection


class SqliteSequence(object):
    vendor = 'sqlite'

    def __init__(self, *args):
        random.seed(time.monotonic())

    def __call__(self, model, n):
        start = random.randint(100000, 999999)
        return deque(range(start, start + n))


class PostgresqlSequence(object):
    vendor = 'postgresql'

    def __init__(self, *models):
        self.column_defaults = self._get_column_defaults(models)

    def _get_column_defaults(self, models):
        table_names = 'or'.join(['(table_name = %s and column_name = %s)'] * len(models))
        sql = '''
            select table_name, column_default
            from information_schema.columns
            where ({table_names})
            and table_catalog = current_catalog
            and table_schema = current_schema
        '''.format(table_names=table_names)
        params = (
            (model._meta.db_table, model._meta.pk.column)
            for model in models
        )
        c = connection.cursor()
        c.execute(sql, list(chain(*params)))
        return {
            table_name: column_default
            for table_name, column_default in c.fetchall()
        }

    def _get_series(self, model, n):
        table = model._meta.db_table
        sql = '''
            select {column_default} from generate_series(1, %s);
        '''.format(column_default=self.column_defaults[table])
        c = connection.cursor()
        c.execute(sql, [n])
        for (val, ) in c.fetchall():
            yield val

    def __call__(self, model, n):
        """Вернуть список id для модели из последовательности
        model: класс модели, из списка переданного в конструктор
        n: количество значений, которые нужно вернуть из последовательности
        """
        return deque(self._get_series(model, n))


def get_sequence(*models):
    classes = {
        clazz.vendor: clazz
        for clazz in [PostgresqlSequence, SqliteSequence]
    }
    sequence_class = classes[connection.vendor]
    return sequence_class(*models)
