# -*- coding: utf-8 -*-
import dateutil
import urllib

from bson.objectid import ObjectId
from django.conf import settings
from json import loads as json_loads

from events.accounts.models import get_or_create_organization
from events.common_app.utils import disable_signals, lazy_re_compile
from events.countme.models import AnswerCount
from events.media.models import Image
from events.surveyme.models import (
    AnswerType,
    Survey,
    SurveySubmitConditionNode,
    SurveySubmitConditionNodeItem,
    SurveyQuestion,
    SurveyQuestionChoice,
    SurveyQuestionMatrixTitle,
    SurveyQuestionShowConditionNode,
    SurveyQuestionShowConditionNodeItem,
    SurveyStyleTemplate,
    SurveyText,
)
from events.surveyme_integration.models import (
    IntegrationFileTemplate,
    JSONRPCSubscriptionData,
    JSONRPCSubscriptionParam,
    ServiceSurveyHookSubscription,
    StartrekSubscriptionData,
    SubscriptionAttachment,
    SubscriptionHeader,
    SurveyHook,
    SurveyHookCondition,
    SurveyHookConditionNode,
    SurveyVariable,
    WikiSubscriptionData,
)
from events.surveyme_integration.utils import macros_re
from events.surveyme.sequences import get_sequence
from rest_framework.exceptions import ValidationError

CONTENT_TYPE_ATTRIBUTE_ANSWER_CHOICES = 2  # используется для проверки content_type_attribute

SURVEY_PARAMS = (
    'name',
    'need_auth_to_answer',
    'is_only_for_iframe',
    'is_allow_answer_editing',
    'is_allow_multiple_answers',
    'is_allow_answer_versioning',
    'metrika_counter_code',
    'captcha_display_mode',
    'maximum_answers_count',
    'auto_control_publication_status',
    'validator_url',
    'language',
    'type',
)

SURVEY_EXTRA_PARAMS = (
    'redirect',
    'footer',
    'quiz',
    'stats',
    'teaser',
)

TEXT_PARAMS = (
    'slug',
    'max_length',
    'null',
    'value',
)

QUESTION_PARAMS = (
    'answer_type_id',
    'validator_type_id',
    'position',
    'page',
    'initial',
    'param_is_required',
    'param_is_allow_multiple_choice',
    'param_is_allow_other',
    'param_max',
    'param_min',
    'param_is_section_header',
    'param_max_file_size',
    'param_price',
    'param_variables',
    'param_widget',
    'param_is_hidden',
    'param_slug',
    'param_hint_type_id',
    'param_data_source',
    'param_data_source_params',
    'param_hint_data_source',
    'param_hint_data_source_params',
    'param_is_random_choices_position',
    'param_modify_choices',
    'param_max_files_count',
    'param_is_disabled_init_item',
    'param_date_field_type',
    'param_date_field_min',
    'param_date_field_max',
    'param_suggest_choices',
    'param_help_text',
    'param_quiz',
    'label',
)

CHOICE_PARAMS = (
    'position',
    'slug',
    'is_hidden',
    'label',
)

MATRIX_PARAMS = (
    'position',
    'type',
    'label',
)

CONDITIONS_PARAMS = (
    'operator',
    'position',
    'condition',
    'value',
)

HOOKS_PARAMS = (
    'is_active',
    'name',
)

SUBSCRIPTIONS_PARAMS = (
    'is_synchronous',
    'is_active',
    'context_language',
    'title',
    'body',
    'email',
    'phone',
    'email_to_address',
    'email_from_address',
    'email_from_title',
    'email_spam_check',
    'http_url',
    'http_method',
    'http_format_name',
    'is_all_questions',
    'follow_result',
)

VARIABLES_PARAMS = (
    'var',
    'format_name',
    'arguments',
    'filters',
)

JSONRPC_PARAMS = (
    'method',
)

JSONRPC_PARAM_PARAMS = (
    'name',
    'value',
    'add_only_with_value',
)

STARTREK_PARAMS = (
    'queue',
    'parent',
    'author',
    'assignee',
    'type',
    'project',
    'priority',
    'followers',
    'tags',
    'components',
    'fields',
)

WIKI_PARAMS = (
    'supertag',
    'text',
)

HEADERS_PARAMS = (
    'value',
    'name',
    'add_only_with_value',
)

ATTACHMENT_TEMPLATES_PARAMS = (
    'name',
    'template',
    'type',
    'slug',
)

SURVEY_STYLES_TEMPLATE_PARAMS = (
    'name',
    'type',
    'styles',
)

DEFAULT_ANSWER_TYPE = 1
ANSWER_GROUP_IDS = (40, 1040)


class SurveyImporter(object):

    avatars_storage_path_re = lazy_re_compile(r'/([\d]{3,}/[\da-f]{32,})/')

    def __init__(self, data):
        self.data = data
        self.sequence = get_sequence(
            Image,
            IntegrationFileTemplate,
            JSONRPCSubscriptionData,
            ServiceSurveyHookSubscription,
            StartrekSubscriptionData,
            SurveyHook,
            SurveyHookConditionNode,
            SurveySubmitConditionNode,
            SurveyQuestion,
            SurveyQuestionChoice,
            SurveyQuestionShowConditionNode,
        )
        self.profile_answer_types = self._get_profile_answer_types()

    def import_survey(self, user_id, dir_id):
        with disable_signals():
            survey = self._create_survey(user_id, dir_id)
            self._create_texts(survey)
            self._create_questions(survey)
            self._create_show_conditions_nodes()
            self._create_attachment_templates(survey)
            self._create_hooks(survey)
            self._create_submit_condition_nodes(survey)
            self._create_counters(survey)
            return survey

    def _get_profile_answer_types(self):
        return set(AnswerType.objects.filter(kind='profile').values_list('pk', flat=True))

    def _get_params(self, params, data):
        prepared_params = {
            elem: data.get(elem)
            for elem in params
            if data.get(elem) is not None
        }
        return prepared_params

    def _add_translations(self, model, language, params):
        if hasattr(model, 'FIELDS_FOR_TRANSLATION'):
            translations = {}
            for field_name in model.FIELDS_FOR_TRANSLATION:
                field_value = params.get(field_name)
                if field_value is not None:
                    translations[field_name] = {
                        language: field_value,
                    }
            params['translations'] = translations

    def _create_survey(self, user_id, dir_id):
        survey_data = self.data.get('survey')
        if not survey_data or not isinstance(survey_data, dict):
            raise ValidationError({'errors': ['Invalid data format']})
        prepared_params = self._get_params(SURVEY_PARAMS, survey_data)
        prepared_params['extra'] = self._get_params(SURVEY_EXTRA_PARAMS, survey_data)
        prepared_params['user_id'] = user_id
        prepared_params['cached_unicode_value'] = prepared_params.get('name')
        prepared_params['is_published_external'] = False
        prepared_params['date_published'] = None
        if survey_data.get('datetime_auto_open'):
            prepared_params['datetime_auto_open'] = dateutil.parser.parse(survey_data['datetime_auto_open'])
        if survey_data.get('datetime_auto_close'):
            prepared_params['datetime_auto_close'] = dateutil.parser.parse(survey_data['datetime_auto_close'])
        if dir_id:
            prepared_params['org'] = get_or_create_organization(dir_id)
        styles_template_id = self._get_or_create_styles_template(survey_data)
        if styles_template_id:
            prepared_params['styles_template_id'] = styles_template_id
        self._add_translations(Survey, prepared_params.get('language', 'ru'), prepared_params)
        return Survey.objects.create(**prepared_params)

    def _create_texts(self, survey):
        texts = (
            self._create_text(survey, text_data)
            for text_data in self.data.get('texts', [])
        )
        SurveyText.objects.bulk_create(texts)

    def _create_text(self, survey, text_data):
        prepared_params = self._get_params(TEXT_PARAMS, text_data)
        prepared_params['survey_id'] = survey.pk
        self._add_translations(SurveyText, survey.language, prepared_params)
        return SurveyText(**prepared_params)

    def _create_questions(self, survey):
        self.questions_map, self.choices_map = {}, {}
        self.images = []
        questions, choices, matrix_choices = [], [], []
        questions_data = self.data.get('questions', [])
        questions_seq = self.sequence(SurveyQuestion, len(questions_data))

        choices_count = sum(
            len(question_data.get('choices'))
            for question_data in questions_data
        )
        choices_seq = self.sequence(SurveyQuestionChoice, choices_count)

        images_count = 0
        for question_data in questions_data:
            if question_data.get('label_image'):
                images_count += 1
            for choice_data in question_data.get('choices'):
                if choice_data.get('label_image'):
                    images_count += 1
        self.images_seq = self.sequence(Image, images_count)

        for question_data in questions_data:
            question = self._create_question(survey, question_data, questions_seq)
            self.questions_map[question_data.get('id')] = (question_data, question)
            questions.append(question)

            choices_data = question_data.get('choices')
            if choices_data:
                choices.extend(self._create_choices(survey, question, choices_data, choices_seq))

            matrixes_data = question_data.get('matrix_titles')
            if matrixes_data:
                matrix_choices.extend(self._create_matrix_choices(survey, question, matrixes_data))

        for question_data, question in self.questions_map.values():
            self._patch_question(question_data, question)

        Image.objects.bulk_create(self.images)

        SurveyQuestion.objects.bulk_create(questions)
        SurveyQuestionChoice.objects.bulk_create(choices)
        SurveyQuestionMatrixTitle.objects.bulk_create(matrix_choices)

    def _create_question(self, survey, question_data, seq):
        prepared_params = self._get_params(QUESTION_PARAMS, question_data)
        prepared_params['pk'] = seq.popleft()
        prepared_params['survey_id'] = survey.pk

        image_data = question_data.get('label_image')
        if image_data and image_data.get('links'):
            prepared_params['label_image'] = self._create_image(image_data)
        self._add_translations(SurveyQuestion, survey.language, prepared_params)

        return SurveyQuestion(**prepared_params)

    def _patch_question(self, question_data, question):
        # патчим поле group_id для вопросов членов группы
        old_group_id = question_data.get('group_id')
        if old_group_id:
            _, new_question = self.questions_map.get(old_group_id, (None, None))
            if new_question:
                question.group_id = new_question.pk

        # патчим код вопроса в фильтре источника данных
        param_data_source_params = question_data.get('param_data_source_params')
        if param_data_source_params and\
            isinstance(param_data_source_params.get('filters'), list) and\
                param_data_source_params['filters']:
            temp_filters = []
            for param in param_data_source_params['filters']:
                if param.get('filter').get('name') == 'question':
                    _, new_question = self.questions_map.get(param.get('value'), (None, None))
                    if new_question:
                        param['value'] = new_question.pk
                        temp_filters.append(param)
                else:
                    temp_filters.append(param)
            param_data_source_params['filters'] = temp_filters
            question.param_data_source_params = param_data_source_params

        # патчим профильные вопросы, переделываем их в Короткий текст
        if question.answer_type_id in self.profile_answer_types:
            question.answer_type_id = DEFAULT_ANSWER_TYPE
            question.param_slug = 'answer_short_text_%s' % question.pk

        # патчим серии вопросов, они не могут быть обязательными
        if question.answer_type_id in ANSWER_GROUP_IDS:
            question.param_is_required = False

    def _create_choices(self, survey, question, choices_data, choices_seq):
        choices = []
        # TODO Убрать строчку ниже если FORMS-1254 более не актуально (храним ответы по-другому)
        choices_data.sort(key=lambda item: item['position'], reverse=True)
        for choice_data in choices_data:
            choice = self._create_choice(survey, question, choice_data, choices_seq)
            self.choices_map[choice_data['id']] = (choice_data, choice)
            choices.append(choice)
        return choices

    def _create_choice(self, survey, question, choice_data, choices_seq):
        prepared_params = self._get_params(CHOICE_PARAMS, choice_data)
        prepared_params['pk'] = choices_seq.pop()
        prepared_params['survey_question_id'] = question.pk

        image_data = choice_data.get('label_image')
        if image_data and image_data.get('links'):
            prepared_params['label_image'] = self._create_image(image_data)
        self._add_translations(SurveyQuestionChoice, survey.language, prepared_params)

        return SurveyQuestionChoice(**prepared_params)

    def _create_image(self, image_data):
        prepared_params = {}

        prepared_params['name'] = image_data['name']
        prepared_params['pk'] = self.images_seq.pop()

        links_data = image_data['links']
        image_path = next(iter(links_data.values()))
        match_path = self.avatars_storage_path_re.search(image_path)
        if match_path:
            prepared_params['image'] = match_path.group(1)

            image = Image(**prepared_params)
            self.images.append(image)

            return image

    def _create_matrix_choices(self, survey, question, matrixes_data):
        matrix_choices = []
        for matrix_data in matrixes_data:
            prepared_params = self._get_params(MATRIX_PARAMS, matrix_data)
            prepared_params['survey_question_id'] = question.pk
            self._add_translations(SurveyQuestionMatrixTitle, question.survey.language, prepared_params)
            matrix_choice = SurveyQuestionMatrixTitle(**prepared_params)
            matrix_choices.append(matrix_choice)
        return matrix_choices

    def _create_show_conditions_nodes(self):
        nodes_data = self.data.get('questions_show_conditions', [])
        if nodes_data:
            nodes, items = [], []
            seq = self.sequence(SurveyQuestionShowConditionNode, len(nodes_data))
            for node_data in nodes_data:
                _, question = self.questions_map.get(node_data['survey_question'], (None, None))
                if question:
                    node = SurveyQuestionShowConditionNode(pk=seq.pop(), survey_question_id=question.pk)
                    nodes.append(node)
                    items_data = node_data.get('items')
                    items.extend(self._create_show_conditions_node_items(node, items_data))
            SurveyQuestionShowConditionNode.objects.bulk_create(nodes)
            SurveyQuestionShowConditionNodeItem.objects.bulk_create(items)

    def _create_show_conditions_node_items(self, node, items_data):
        items_list = []
        for item_data in items_data:
            _, question = self.questions_map.get(item_data['survey_question'], (None, None))
            if question:
                prepared_params = self._get_params(CONDITIONS_PARAMS, item_data)
                prepared_params['survey_question_show_condition_node_id'] = node.pk
                prepared_params['survey_question_id'] = question.pk
                content_type_attribute = item_data['content_type_attribute']
                prepared_params['content_type_attribute_id'] = content_type_attribute
                item_value = item_data.get('value') or ''
                if content_type_attribute == CONTENT_TYPE_ATTRIBUTE_ANSWER_CHOICES:
                    if item_value.isdigit():
                        _, choice = self.choices_map.get(int(item_value), (None, None))
                        if choice:
                            prepared_params['value'] = choice.id
                else:
                    prepared_params['value'] = item_value
                item = SurveyQuestionShowConditionNodeItem(**prepared_params)
                items_list.append(item)
        return items_list

    def _create_attachment_templates(self, survey):
        self.templates_data = self.data['survey'].get('integration_file_templates')
        self.variables = []
        self.variables_map = {}
        if self.templates_data:
            templates_count = len('templates_data')
            templates_seq = self.sequence(IntegrationFileTemplate, templates_count)

            templates = []
            self.templates_map = {}
            for template_data in self.templates_data:
                prepared_params = self._get_params(ATTACHMENT_TEMPLATES_PARAMS, template_data)
                prepared_params['survey_id'] = survey.pk
                prepared_params['pk'] = templates_seq.pop()

                variables_data = template_data.get('variables')
                if variables_data:
                    self.variables.extend(self._create_variables(variables_data, template_id=prepared_params['pk']))

                prepared_params['template'] = self._update_variables(prepared_params, 'template')

                template = IntegrationFileTemplate(**prepared_params)
                self.templates_map[template_data['id']] = (template_data, template)
                templates.append(template)

            IntegrationFileTemplate.objects.bulk_create(templates)

    def _create_hooks(self, survey):
        hooks_data = self.data['survey'].get('hooks')
        if hooks_data:
            seq = self.sequence(SurveyHook, len(hooks_data))
            hooks = []
            hooks_map = {}
            for hook_data in hooks_data:
                hook = self._create_hook(survey, hook_data, seq)
                hooks_map[hook_data.get('id')] = (hook_data, hook)
                hooks.append(hook)

            SurveyHook.objects.bulk_create(hooks)

            HookTriggers = SurveyHook.triggers.through
            triggers = []
            for hook_data, hook in hooks_map.values():
                for trigger_id in hook_data.get('triggers', []):
                    triggers.append(HookTriggers(surveyhook_id=hook.pk, surveyhooktrigger_id=trigger_id))

            if triggers:
                HookTriggers.objects.bulk_create(triggers)

            self._create_subscriptions(hooks_map)
            self._create_hook_conditions_nodes(hooks_map)

    def _create_hook(self, survey, hook_data, seq):
        prepared_params = self._get_params(HOOKS_PARAMS, hook_data)
        prepared_params['pk'] = seq.pop()
        prepared_params['survey_id'] = survey.pk
        return SurveyHook(**prepared_params)

    def _create_subscriptions(self, hooks_map):
        subscriptions_count = sum(
            len(hook_data.get('subscriptions'))
            for hook_data, _ in hooks_map.values()
        )
        subscriptions_seq = self.sequence(ServiceSurveyHookSubscription, subscriptions_count)

        subscriptions, attachments, = [], []
        subscriptions_map = {}
        for hook_data, hook in hooks_map.values():
            for subscription_data in hook_data.get('subscriptions'):
                subscription = self._create_subscription(hook, subscription_data, subscriptions_seq)
                subscriptions_map[subscription_data['id']] = (subscription_data, subscription)
                subscriptions.append(subscription)

                variables_data = subscription_data.get('variables')
                if variables_data:
                    self.variables.extend(self._create_variables(variables_data, subscription_id=subscription.pk))

                attachments_data = subscription_data.get('attachments')
                if attachments_data:
                    attachments.extend(self._create_attachments(subscription, attachments_data))

        self._patch_subscriptions_pre_save(hook, subscriptions_map)
        ServiceSurveyHookSubscription.objects.bulk_create(subscriptions)
        self._patch_subscriptions_post_save(hook, subscriptions_map)

        SurveyVariable.objects.bulk_create(self.variables)
        SubscriptionAttachment.objects.bulk_create(attachments)

        self._create_integrations(hooks_map, subscriptions_map, subscriptions_count)
        self._create_headers(hooks_map, subscriptions_map)

    def _create_subscription(self, hook, subscription_data, subscriptions_seq):
        prepared_params = self._get_params(SUBSCRIPTIONS_PARAMS, subscription_data)
        prepared_params['service_type_action_id'] = subscription_data.get('service_type_action')
        prepared_params['survey_hook_id'] = hook.pk
        prepared_params['pk'] = subscriptions_seq.pop()
        if settings.APP_TYPE != 'forms_int':
            # при импорте включаем проверку на спам
            prepared_params['email_spam_check'] = True
        return ServiceSurveyHookSubscription(**prepared_params)

    def _create_variables(self, variables_data, subscription_id=None, template_id=None):
        variables_list = []
        for variable_data_id, variable_data in variables_data.items():
            prepared_params = self._get_params(VARIABLES_PARAMS, variable_data)
            prepared_params['variable_id'] = str(ObjectId())
            if subscription_id:
                prepared_params['hook_subscription_id'] = subscription_id
            elif template_id:
                prepared_params['integration_file_template_id'] = template_id

            self._patch_variable(variable_data)

            variable = SurveyVariable(**prepared_params)
            self.variables_map[variable_data_id] = (variable_data, variable)
            variables_list.append(variable)
        return variables_list

    def _patch_variable(self, variable_data):
        variable_arg_question = variable_data['arguments'].get('question')
        variable_arg_questions = variable_data['arguments'].get('questions')
        variable_is_all_questions = variable_data['arguments'].get('is_all_questions')

        if variable_arg_question:
            _, question = self.questions_map.get(variable_arg_question, (None, None))
            if question:
                variable_data['arguments']['question'] = question.pk
            else:
                variable_data['arguments']['question'] = None
        elif variable_arg_questions:
            new_questions = []
            if variable_is_all_questions is not True:
                for var_question in variable_arg_questions:
                    _, question = self.questions_map.get(var_question, (None, None))
                    if question:
                        new_questions.append(question.pk)
            variable_data['arguments']['questions'] = new_questions

    def _update_variables(self, data, field):
        field_data = data.get(field, '')
        variable_ids = set(macros_re.findall(field_data))
        for variable_id in variable_ids:
            _, variable = self.variables_map.get(variable_id, (None, None))
            if variable:
                field_data = field_data.replace(variable_id, variable.variable_id)
        return field_data

    def _create_attachments(self, subscription, attachments_data):
        attachments_list = []
        prepared_params = {}
        for attachment_data in attachments_data:
            prepared_params['subscription'] = subscription

            attachment_path = attachment_data['path'].replace('%2F', '/')
            parsed = urllib.parse.urlsplit(attachment_path)
            path = urllib.parse.parse_qs(parsed.query).get('path')
            if path:
                prepared_params['file'] = path[0].rstrip('/')

            attachment = SubscriptionAttachment(**prepared_params)
            attachments_list.append(attachment)
        return attachments_list

    def _patch_subscriptions_pre_save(self, hook, subscriptions_map):
        fields = (
            'title',
            'body',
            'email_to_address',
            'email_from_address',
            'email_from_title',
            'http_url',
        )
        for subscription_data, subscription in subscriptions_map.values():
            for value in fields:
                setattr(subscription, value, self._update_variables(subscription_data, value))
            if settings.IS_BUSINESS_SITE:
                subscription.email_from_address = f'{hook.survey.pk}@forms-mailer.yaconnect.com'

    def _patch_subscriptions_post_save(self, hook, subscriptions_map):
        if self.templates_data:
            SubscriptionTemplates = ServiceSurveyHookSubscription.attachment_templates.through
            templates = []
            for subscription_data, subscription in subscriptions_map.values():
                for attachment_template in subscription_data.get('attachment_templates'):
                    _, template = self.templates_map.get(attachment_template['id'], (None, None))
                    if template:
                        templates.append(SubscriptionTemplates(
                            integrationfiletemplate_id=template.pk,
                            servicesurveyhooksubscription_id=subscription.pk,
                        ))
            if templates:
                SubscriptionTemplates.objects.bulk_create(templates)

        SubscriptionQuestions = ServiceSurveyHookSubscription.questions.through
        questions = []
        for subscription_data, subscription in subscriptions_map.values():
            for question_old_id in subscription_data.get('questions'):
                _, question = self.questions_map.get(question_old_id, (None, None))
                if question:
                    questions.append(SubscriptionQuestions(
                        surveyquestion_id=question.pk,
                        servicesurveyhooksubscription_id=subscription.pk,
                    ))
        if questions:
            SubscriptionQuestions.objects.bulk_create(questions)

    def _create_integrations(self, hooks_map, subscriptions_map, subscriptions_count):
        json_rpcs, startreks, wikis = [], [], []
        json_rpc_params = []

        json_rpc_seq = self.sequence(JSONRPCSubscriptionData, subscriptions_count)
        startrek_seq = self.sequence(StartrekSubscriptionData, subscriptions_count)

        for hook_data, hook in hooks_map.values():
            for subscription_data in hook_data.get('subscriptions'):
                _, subscription = subscriptions_map[subscription_data['id']]

                json_rpc_data = subscription_data.get('json_rpc')
                if json_rpc_data:
                    json_rpc = self._create_json_rpc_subscription(subscription, json_rpc_data, json_rpc_seq)
                    json_rpcs.append(json_rpc)
                    json_rpc_params.extend(self._create_json_rpc_params(json_rpc, json_rpc_data))

                startrek_data = subscription_data.get('startrek')
                if startrek_data:
                    startrek = self._create_startrek_subscription(subscription, startrek_data, startrek_seq)
                    startreks.append(startrek)

                wiki_data = subscription_data.get('wiki')
                if wiki_data:
                    wikis.append(self._create_wiki_subscription(subscription, wiki_data))

        if json_rpcs:
            JSONRPCSubscriptionData.objects.bulk_create(json_rpcs)
        if json_rpc_params:
            JSONRPCSubscriptionParam.objects.bulk_create(json_rpc_params)
        if startreks:
            StartrekSubscriptionData.objects.bulk_create(startreks)
        if wikis:
            WikiSubscriptionData.objects.bulk_create(wikis)

    def _create_json_rpc_subscription(self, subscription, json_rpc_data, json_rpc_seq):
        prepared_params = self._get_params(JSONRPC_PARAMS, json_rpc_data)
        prepared_params['subscription'] = subscription
        prepared_params['method'] = self._update_variables(prepared_params, 'method')
        prepared_params['pk'] = json_rpc_seq.pop()
        return JSONRPCSubscriptionData(**prepared_params)

    def _create_json_rpc_params(self, json_rpc, json_rpc_data):
        params_list = []
        for param_data in json_rpc_data.get('params'):
            prepared_params = self._get_params(JSONRPC_PARAM_PARAMS, param_data)
            prepared_params['subscription'] = json_rpc
            for value in ('name', 'value'):
                prepared_params[value] = self._update_variables(prepared_params, value)
            param = JSONRPCSubscriptionParam(**prepared_params)
            params_list.append(param)
        return params_list

    def _create_startrek_subscription(self, subscription, startrek_data, startrek_seq):
        prepared_params = self._get_params(STARTREK_PARAMS, startrek_data)
        for value in ('author', 'assignee'):
            prepared_params[value] = self._update_variables(prepared_params, value)
        prepared_params['subscription'] = subscription
        prepared_params['pk'] = startrek_seq.pop()
        return StartrekSubscriptionData(**prepared_params)

    def _create_wiki_subscription(self, subscription, wiki_data):
        prepared_params = self._get_params(WIKI_PARAMS, wiki_data)
        prepared_params['subscription'] = subscription
        for value in ('supertag', 'text'):
            prepared_params[value] = self._update_variables(prepared_params, value)
        return WikiSubscriptionData(**prepared_params)

    def _create_headers(self, hooks_map, subscriptions_map):
        headers = []
        for hook_data, hook in hooks_map.values():
            for subscription_data in hook_data.get('subscriptions'):
                _, subscription = subscriptions_map[subscription_data['id']]
                headers.extend(
                    self._create_header(subscription, header_data)
                    for header_data in subscription_data['headers']
                )
        SubscriptionHeader.objects.bulk_create(headers)

    def _create_header(self, subscription, header_data):
        prepared_params = self._get_params(HEADERS_PARAMS, header_data)
        prepared_params['subscription'] = subscription
        for value in ('value', 'name'):
            prepared_params[value] = self._update_variables(prepared_params, value)
        return SubscriptionHeader(**prepared_params)

    def _create_hook_conditions_nodes(self, hooks_map):
        nodes, node_items = [], []

        nodes_count = sum(
            len(hook_data.get('condition_nodes'))
            for hook_data, _ in hooks_map.values()
        )
        node_seq = self.sequence(SurveyHookConditionNode, nodes_count)

        for hook_data, hook in hooks_map.values():
            for node_data in hook_data.get('condition_nodes'):
                node = SurveyHookConditionNode(pk=node_seq.pop(), hook_id=hook.pk)
                nodes.append(node)
                node_items.extend(self._create_hook_conditions_node_items(node, node_data))
        SurveyHookConditionNode.objects.bulk_create(nodes)
        SurveyHookCondition.objects.bulk_create(node_items)

    def _create_hook_conditions_node_items(self, node, node_data):
        items_list = []
        for item_data in node_data.get('items'):
            prepared_params = self._get_params(CONDITIONS_PARAMS, item_data)
            _, question = self.questions_map.get(item_data['survey_question'], (None, None))
            if question:
                prepared_params['survey_question_id'] = question.pk
            content_type_attribute = item_data['content_type_attribute']
            item_value = item_data.get('value') or ''
            if content_type_attribute == CONTENT_TYPE_ATTRIBUTE_ANSWER_CHOICES:
                if item_value.isdigit():
                    _, choice = self.choices_map.get(int(item_value), (None, None))
                    if choice:
                        prepared_params['value'] = choice.id
            else:
                prepared_params['value'] = item_data['value']
            prepared_params['content_type_attribute_id'] = content_type_attribute
            prepared_params['condition_node'] = node
            item = SurveyHookCondition(**prepared_params)
            items_list.append(item)
        return items_list

    def _create_submit_condition_nodes(self, survey):
        nodes_data = self.data.get('nodes', [])
        if nodes_data:
            nodes, node_items = [], []
            seq = self.sequence(SurveySubmitConditionNode, len(nodes_data))
            for node_data in nodes_data:
                node = SurveySubmitConditionNode(pk=seq.pop(), survey_id=survey.pk)
                nodes.append(node)
                node_items.extend(self._create_submit_condition_node_items(node, node_data))
            SurveySubmitConditionNode.objects.bulk_create(nodes)
            SurveySubmitConditionNodeItem.objects.bulk_create(node_items)

    def _create_submit_condition_node_items(self, node, node_data):
        items_list = []
        for item_data in node_data.get('items', []):
            prepared_params = self._get_params(CONDITIONS_PARAMS, item_data)
            _, question = self.questions_map.get(item_data['survey_question'], (None, None))
            if question:
                prepared_params['survey_question_id'] = question.pk
            prepared_params['survey_submit_condition_node_id'] = node.pk
            _, choice = self.choices_map.get(item_data['survey_question_choice'], (None, None))
            if choice:
                prepared_params['survey_question_choice_id'] = choice.pk
            content_type_attribute = item_data['content_type_attribute']
            item_value = item_data.get('value') or ''
            if content_type_attribute == CONTENT_TYPE_ATTRIBUTE_ANSWER_CHOICES:
                if item_value.isdigit():
                    _, choice = self.choices_map.get(int(item_value), (None, None))
                    if choice:
                        prepared_params['value'] = choice.id
            else:
                prepared_params['value'] = item_value
            prepared_params['content_type_attribute_id'] = content_type_attribute
            item = SurveySubmitConditionNodeItem(**prepared_params)
            items_list.append(item)
        return items_list

    def _get_or_create_styles_template(self, survey_data):
        styles_template_id = survey_data.get('styles_template_id')
        if styles_template_id:
            styles_template_data = survey_data.get('styles_template')
            if styles_template_data:
                prepared_params = self._get_params(SURVEY_STYLES_TEMPLATE_PARAMS, styles_template_data)
                if prepared_params.get('type') == 'custom':
                    styles_template = SurveyStyleTemplate.objects.create(**prepared_params)
                    styles_template_id = styles_template.pk
        return styles_template_id

    def _create_counters(self, survey):
        AnswerCount.objects.create(survey=survey)

    @classmethod
    def from_string(cls, text):
        data = json_loads(text)
        instance = cls(data)
        return instance

    @classmethod
    def from_template(cls, template):
        instance = cls(template.data)
        return instance
