# -*- coding: utf-8 -*-
import datetime
import logging

from celery.signals import after_task_publish
from django.conf import settings
from django.contrib.auth.models import Group
from django.core.exceptions import ObjectDoesNotExist
from django.core.files.base import ContentFile
from django.contrib.contenttypes.models import ContentType
from django.db import connection
from django.utils.encoding import force_str
from guardian.models import UserObjectPermission
from io import BytesIO

from events.accounts.models import User
from events.common_app.utils import (
    get_backoffice_url_for_obj,
)
from events.common_storages.proxy_storages import ProxyStorage
from events.common_storages.storage import SaveError
from events.surveyme.drop_utils import (
    DropNotifications,
    DropDeletedAnswers,
    DropUnusedAnswers,
)
from events.surveyme.logic import access
from events.surveyme.utils import (
    PrepareAnswersToExportCommand,
    RemoveAnswersFromExportCommand,
    RemoveTableFromYtCommand,
    SurveyAutoPublicationCommand,
    get_survey,
    send_email_with_retry,
)
from events.celery_app import app

logger = logging.getLogger(__name__)


TAKEOUT_RETRY_DELAY = 300
TAKEOUT_RETRY_ERRORS = (SaveError, )
TAKEOUT_RETRIES = 5

EXPORT_ANSWERS_TIME_LIMIT = 90 * 60  # 90 min
EXPORT_ANSWERS_SORT_TIME_LIMIT = 60 * 60  # 60 min


@app.task(
    time_limit=EXPORT_ANSWERS_TIME_LIMIT,
    soft_time_limit=EXPORT_ANSWERS_SORT_TIME_LIMIT,
)
def export_survey_answers(survey_id, **kwargs):
    from events.surveyme.export_answers_v2 import export_answers
    return export_answers(survey_id, **kwargs)


@app.task
def send_information_about_robot_actions(survey_id, action_text):
    # todo: test me
    from events.surveyme.models import Survey

    survey = Survey.objects.get(id=survey_id)

    subject = '[Конструктор форм] Форма: %s' % survey.cached_unicode_value
    html_content = """
    Форма <a href="{survey_href}">{survey_title}</a> была автоматически {action_text}.
    """.format(
        survey_href=get_backoffice_url_for_obj(survey, action_text=action_text),
        survey_title=survey.cached_unicode_value,
        action_text=force_str(action_text)
    )

    emails = [follower.get_email() for follower in survey.followers.all()]
    send_email_with_retry(subject, html_content, emails)


@app.task
def check_surveys_publication_status():
    command = SurveyAutoPublicationCommand()
    while command.execute() > 0:
        pass


def get_answers(profile_pk):
    sql = '''
        select
          cast(row_number() over() as varchar),
          cast(a.date_created as varchar),
          a.survey_id as survey,
          a.data as answer
        from surveyme_profilesurveyanswer a
        where a.user_id = %s
        group by a.id
    '''
    c = connection.cursor()
    c.execute(sql, [profile_pk])
    for row in c.fetchall():
        yield row


@app.task(
    bind=True,
    default_retry_delay=TAKEOUT_RETRY_DELAY,
    autoretry_for=TAKEOUT_RETRY_ERRORS,
    retry_kwargs={'max_retries': TAKEOUT_RETRIES},
)
def takeout(self, user_pk):
    from events.surveyme.export_answers import CsvExporter
    task = app.AsyncResult(self.request.id)
    logger.info('takeout task %s %s', self.request.id, task.status)
    with BytesIO() as f:
        with CsvExporter(f) as exporter:
            exporter.writeheader(('id', 'date_created', 'answers'))
            for values in get_answers(user_pk):
                exporter.writerow(values)
        filename = 'takeout.csv'
        path = ProxyStorage().save(content=ContentFile(f.getvalue(), name=filename), name=filename)
        return {
            'path': path,
            'filename': filename,
            'content_type': 'text/csv',
            'user_pk': user_pk,
        }


@app.task
def prepare_answers_to_export(survey_id):
    if settings.IS_BUSINESS_SITE:
        return  # ничего не делаем в б2б
    survey = get_survey(survey_id)
    if not survey or survey.is_deleted:
        return  # форма не существует или удалена
    if survey.save_logs_for_statbox:
        # добавить статус экспорта для ответов на форму
        command = PrepareAnswersToExportCommand(survey)
        command.execute()


@app.task
def remove_answers_from_export(survey_id):
    if settings.IS_BUSINESS_SITE:
        return  # ничего не делаем в б2б
    survey = get_survey(survey_id, with_deleted=True)
    if not survey:
        return  # форма не найдена
    if survey.is_deleted or not survey.save_logs_for_statbox:
        # удалить статус экспорта с ответов на форму
        command = RemoveAnswersFromExportCommand(survey)
        command.execute()


@app.task
def remove_table_from_yt(survey_id):
    if settings.IS_BUSINESS_SITE:
        return  # ничего не делаем в б2б
    survey = get_survey(survey_id, with_deleted=True)
    if not survey:
        return  # форма не найдена
    if survey.is_deleted or not survey.save_logs_for_statbox:
        # удалить таблицу из YT
        command = RemoveTableFromYtCommand(survey)
        command.execute()


@after_task_publish.connect
def update_sent_state(sender=None, headers=None, **kwargs):
    task = app.tasks.get(sender)
    if task:
        task.backend.store_result(headers['id'], None, 'SENT')


@app.task(time_limit=4 * 3600, soft_time_limit=3 * 3600)
def drop_notifications():
    last_day = datetime.date.today() - datetime.timedelta(days=90)
    limit = 250
    operation = DropNotifications(last_day, limit)
    operation.execute()


@app.task(time_limit=4 * 3600, soft_time_limit=3 * 3600)
def drop_deleted_answers():
    last_day = datetime.date.today() - datetime.timedelta(days=30)
    limit = 250
    operation = DropDeletedAnswers(last_day, limit)
    operation.execute()


@app.task(time_limit=4 * 3600, soft_time_limit=3 * 3600)
def drop_unused_answers():
    last_day = datetime.date.today() - datetime.timedelta(days=90)
    limit = 250
    operation = DropUnusedAnswers(last_day, limit)
    operation.execute()


@app.task
def reject_roles(request_user_id, model_type, object_pk, deleted_only=True):
    if deleted_only:
        if model_type.objects.filter(pk=object_pk).exists():
            return

    ct = ContentType.objects.get_for_model(model_type)
    users = list(User.objects.filter(
        userobjectpermission__content_type=ct,
        userobjectpermission__object_pk=str(object_pk),
    ))
    groups = list(Group.objects.filter(
        groupobjectpermission__content_type=ct,
        groupobjectpermission__object_pk=str(object_pk),
    ))
    perm = access.get_change_permission(ct)
    request_user = User.objects.get(pk=request_user_id)
    access.reject_roles(request_user, perm, users=users, groups=groups, content_type=ct, object_pk=object_pk)


@app.task
def request_roles(request_user_id, model_type, object_pk):
    ct = ContentType.objects.get_for_model(model_type)
    perm = access.get_change_permission(ct)
    request_user = User.objects.get(pk=request_user_id)
    try:
        obj = (
            model_type.objects.select_related('user')
            .get(pk=object_pk)
        )
    except ObjectDoesNotExist:
        pass
    else:
        user_granted = (
            UserObjectPermission.objects.filter(
                content_type=ct,
                user=obj.user,
                object_pk=str(obj.pk),
            )
            .exists()
        )
        if not user_granted:
            access.request_roles(request_user, perm, users=[obj.user], content_type=ct, object_pk=obj.pk)
