# -*- coding: utf-8 -*-
from django.core import validators
from django.utils.translation import gettext_lazy as _
from django.core.exceptions import ValidationError

from events.common_app.utils import lazy_re_compile
from events.common_app.inn import is_valid as is_inn_valid


class BaseValidator(object):
    required_options = ()

    def __init__(self, **options):
        pass


class DecimalValidator(BaseValidator, validators.RegexValidator):
    regex = validators._lazy_re_compile(r'^([\+-])?(0|[1-9]\d*)([\.,]\d+)?$')
    message = _('enter_a_valid_decimal_number')


class INNValidator(BaseValidator):
    message = _('enter_a_valid_inn_number')

    def __call__(self, value):
        valid_inn = is_inn_valid(value)
        if not valid_inn:
            raise ValidationError(self.message)


class RussianSymbolsValidator(BaseValidator):
    message = _('enter_russian_symbols')
    symbols_re = lazy_re_compile(r'^[А-Яа-яЁё\d\.,; \(\)_-]*$')

    def __call__(self, value):
        if not self.symbols_re.match(value):
            raise ValidationError(self.message)


class RegexpValidator(BaseValidator, validators.RegexValidator):
    message = _('enter_value_matching_regular_expression')
    # нужные ключи для передачи в атрибут 'SurveyQuestion.validator_options'
    required_options = ('regexp',)

    def __init__(self, **options):
        super().__init__(**options)

        exp = options['regexp']
        self.regex = validators._lazy_re_compile(exp)
