# -*- coding: utf-8 -*-
import json
import logging

from django.core.files.uploadedfile import UploadedFile
from django.http import JsonResponse
from rest_framework import generics
from rest_framework.response import Response

from events.surveyme.serializers import ValidatorDataSerializer
from events.rest_framework_contrib.mixins import GenericViewV1Mixin

logger = logging.getLogger(__name__)


class QuestionValidator(generics.GenericAPIView):
    serializer_class = ValidatorDataSerializer

    def get_headers(self, request):
        return {
            key: value
            for key, value in request.META.items()
            if key.startswith('HTTP') and not key.endswith('AUTHORIZATION')
        }

    def post(self, request, *args, **kwargs):
        logger.warn(json.dumps(request.data))
        logger.warn(json.dumps(self.get_headers(request)))
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data
        errors = {}
        for question in data.get('questions', []):
            messages = []
            if question.get('answer_type', {}).get('slug') != 'answer_short_text':
                messages.append('Incorrect answer type, must be `answer_short_text`')
            if question.get('value') != '42':
                messages.append('Incorrect value, must be `42`')
            if messages:
                errors[question.get('slug')] = messages
        result = {
            'status': 'OK',
        }
        if errors:
            result['status'] = 'ERROR'
            result['errors'] = errors

        return Response(data=result)


class RequestInfoMixin:
    ignore_headers = {'authorization', 'x-ya-service-ticket', 'x-ya-user-ticket'}

    def log_request(self, request):
        _headers = {
            header: value
            for (header, value) in request.headers.items()
            if header.lower() not in self.ignore_headers
        }
        logger.warn('Request data %s', request.data)
        logger.warn('Request headers %s', _headers)


class ErrorView(GenericViewV1Mixin, RequestInfoMixin, generics.GenericAPIView):
    def get(self, request, *args, **kwargs):
        return JsonResponse({'status': 'ERROR'}, status=500)

    def post(self, request, *args, **kwargs):
        self.log_request(request)
        return JsonResponse({'status': 'ERROR'}, status=500)

    def put(self, request, *args, **kwargs):
        self.log_request(request)
        return JsonResponse({'status': 'ERROR'}, status=500)

    def patch(self, request, *args, **kwargs):
        self.log_request(request)
        return JsonResponse({'status': 'ERROR'}, status=500)


class SuccessView(GenericViewV1Mixin, RequestInfoMixin, generics.GenericAPIView):
    def get(self, request, *args, **kwargs):
        return JsonResponse({'status': 'OK'})

    def post(self, request, *args, **kwargs):
        self.log_request(request)
        for it in request.data.values():
            if isinstance(it, UploadedFile):
                with open(f'/tmp/{it.name}', 'wb') as f:
                    f.write(it.read())
        return JsonResponse({'status': 'OK'})

    def put(self, request, *args, **kwargs):
        self.log_request(request)
        return JsonResponse({'status': 'OK'})

    def patch(self, request, *args, **kwargs):
        self.log_request(request)
        return JsonResponse({'status': 'OK'})


class ConflictView(GenericViewV1Mixin, RequestInfoMixin, generics.GenericAPIView):
    status = 409

    def _get_status(self, request):
        try:
            return int(request.query_params.get('status', self.status))
        except ValueError:
            return self.status

    def get(self, request, *args, **kwargs):
        status = self._get_status(request)
        return JsonResponse({'status': 'CONFLICT'}, status=status)

    def post(self, request, *args, **kwargs):
        self.log_request(request)
        status = self._get_status(request)
        return JsonResponse({'status': 'CONFLICT'}, status=status)

    def put(self, request, *args, **kwargs):
        self.log_request(request)
        status = self._get_status(request)
        return JsonResponse({'status': 'CONFLICT'}, status=status)

    def patch(self, request, *args, **kwargs):
        self.log_request(request)
        status = self._get_status(request)
        return JsonResponse({'status': 'CONFLICT'}, status=status)
