# -*- coding: utf-8 -*-
import logging
import yenv

from django.http import Http404, JsonResponse
from django.views import View
from django.conf import settings
from django.db.models import Q, Prefetch
from django.utils.translation import get_language, ugettext as _
from django.core.cache import cache
from rest_framework import viewsets, views, status, generics, mixins
from rest_framework.response import Response
from rest_framework.exceptions import PermissionDenied
from rest_framework.decorators import action
from urllib.parse import urlencode

from events.common_app.views import Http404Response
from events.surveyme.api.throttling import FormPostUserRateThrottle
from events.surveyme.dataclasses import SurveyQuiz
from events.surveyme.models import (
    Survey,
    SurveyQuestion,
    SurveyQuestionChoice,
    SurveyQuestionShowConditionNodeItem,
    ProfileSurveyAnswer,
    SurveySubmitConditionNodeItem,
)
from events.surveyme.utils import ConditionEvaluator
from events.surveyme.permissions import IsProfileCanAnswerToSurvey
from events.surveyme.serializers import (
    SurveySerializer,
    SurveySuccessSerializer,
    SurveySerializerDetail,
    ScoreResultsSerializer,
    AnswerResultsSerializer,
)

from events.rest_framework_contrib.mixins import (
    ExternalGenericApiViewV1Mixin,
    ManyLookupFieldsMixin,
    DetailSerializerMixin,
)
from events.rest_framework_contrib.permissions import IsAuthenticatedForBusiness
from events.surveyme_keys.models import Key
from events.surveyme.exceptions import MaxSurveyAnswersException
from events.common_app.utils import timeit
from events.surveyme_integration.tasks import start_integrations
from events.surveyme.logic.query import is_true


logger = logging.getLogger(__name__)


class SurveyCommonMixin(object):
    # todo: test me
    def get_user(self):
        return self.request.user

    def get_active_key_from_survey_bundles(self, survey):
        if not hasattr(self, '_key'):
            self._key = self._get_active_key_from_survey_bundles(survey)
        return self._key

    def _get_active_key_from_survey_bundles(self, survey):
        key = self.request.query_params.get('key')
        if key:
            try:
                return Key.objects.get(
                    value=key,
                    is_active=True,
                    bundle__survey__id=survey.id,
                )
            except Key.DoesNotExist:
                pass

    def get_what_to_ignore_while_checking_user_ability_to_answer(self, survey):
        key = self.get_active_key_from_survey_bundles(survey)
        ignore = []
        if key:
            if key.bundle.allow_access_to_unpublished_form:
                ignore.append('is_ready_for_answer')
        return ignore

    def get_is_profile_can_answer(self, survey):
        return survey.is_profile_can_answer(
            user=self.get_user(),
            ignore=self.get_what_to_ignore_while_checking_user_ability_to_answer(survey),
            preview=self.get_preview_status(),
        )

    def get_why_profile_cant_answer(self, survey):
        return survey.why_profile_cant_answer(
            user=self.get_user(),
            ignore=self.get_what_to_ignore_while_checking_user_ability_to_answer(survey),
            preview=self.get_preview_status(),
        )

    def get_preview_status(self):
        return is_true(self.request.query_params.get('preview', ''))


class CaptchaController(object):
    def __init__(self, request, view):
        self.request = request
        self.view = view
        self.throttle = FormPostUserRateThrottle()
        self.is_show_captcha_for_user_key = 'show_captcha_for_%s' % self.throttle.get_identity(self.request)

    def is_enabled(self):
        return bool(settings.IS_ENABLE_CAPTCHA and cache.get(self.is_show_captcha_for_user_key))

    def is_should_be_enabled(self):
        return not self.throttle.allow_request(self.request, view=self.view)

    def set_enabled(self, value):
        cache.set(self.is_show_captcha_for_user_key, value, timeout=10 * 60)

    def flush_submit_history(self):
        key = self.throttle.get_cache_key(self.request, self.view)
        cache.delete(key)


class SurveyViewSet(ManyLookupFieldsMixin,
                    DetailSerializerMixin,
                    SurveyCommonMixin,
                    ExternalGenericApiViewV1Mixin,
                    mixins.RetrieveModelMixin,
                    viewsets.GenericViewSet,
                    ):
    # todo: test me
    create_on_missing = False
    serializer_class = SurveySerializer
    serializer_detail_class = SurveySerializerDetail
    queryset = Survey.objects.filter_by_ready_for_answer()
    queryset_detail = (
        Survey.objects.all()
        .prefetch_related(
            'submit_condition_nodes__items',
            'texts',
        )
        .select_related(
            'group',
            'styles_template',
            'org',
        )
    )
    lookup_fields = ('pk', 'slug')
    permission_classes = (IsAuthenticatedForBusiness,)

    def get_queryset(self, *args, **kwargs):
        queryset = super(SurveyViewSet, self).get_queryset(*args, **kwargs)
        if settings.IS_BUSINESS_SITE:
            query = None
            if not self.request.yauser.is_authenticated():
                query = Q(is_public=True)
            else:
                user = self.request.user
                if not user.is_superuser:
                    query = Q()
                    if getattr(user, 'pk', None):
                        query = Q(org_id__isnull=True, user=user)
                    if self.request.orgs:
                        query = query | Q(org__dir_id__in=self.request.orgs)
                    if not self.get_preview_status():
                        query = query | Q(is_public=True)
            if query:
                queryset = queryset.filter(query)
        return queryset

    @action(url_path='tickets', detail=True)
    def tickets_info(self, request, *args, **kwargs):
        data = self.get_object().get_tickets_status()
        if data:
            return Response(data=data)
        else:
            raise Http404()

    @action(url_path='stats', detail=True)
    def stats(self, request, *args, **kwargs):
        from events.countme.stats import get_stats_info
        survey = self.get_object()
        enabled = False
        extra = survey.extra or {}
        stats = extra.get('stats')
        if isinstance(stats, dict):
            enabled = stats.get('enabled') is True
        if not enabled:
            raise Http404()
        return Response(data=get_stats_info(survey))

    @action(url_path='success', detail=True)
    def success(self, request, *args, **kwargs):
        survey = self.get_object()
        if survey.is_ban_detected:
            raise Http404
        answer_key = request.query_params.get('answer_key')
        if not survey.is_published_external and not answer_key:
            raise Http404
        serializer = SurveySuccessSerializer(survey, context={'answer_key': answer_key})
        return Response(data=serializer.data)


class SurveyFormView(SurveyCommonMixin, ExternalGenericApiViewV1Mixin, views.APIView):
    create_on_missing = False
    choices_qs = (
        SurveyQuestionChoice.objects.all()
        .select_related(
            'label_image',
        )
    )
    submit_condition_items_qs = (
        SurveySubmitConditionNodeItem.objects.all()
        .select_related(
            'content_type_attribute__content_type',
        )
    )
    show_condition_items_qs = (
        SurveyQuestionShowConditionNodeItem.objects.all()
        .select_related(
            'content_type_attribute__content_type',
        )
    )
    questions_qs = (
        SurveyQuestion.objects.all()
        .select_related(
            'answer_type',
            'label_image',
            'validator_type',
            # 'group',
        )
    )
    queryset = (
        Survey.objects.all()
        .prefetch_related(
            Prefetch('surveyquestion_set', queryset=questions_qs),
            Prefetch('surveyquestion_set__surveyquestionchoice_set', queryset=choices_qs),
            Prefetch('surveyquestion_set__show_condition_nodes__items', queryset=show_condition_items_qs),
            Prefetch('submit_condition_nodes__items', queryset=submit_condition_items_qs),
            'surveyquestion_set__surveyquestionmatrixtitle_set',
            'agreements',
        )
    )
    lookup_fields = ('pk', 'slug')
    permission_classes = (IsProfileCanAnswerToSurvey, IsAuthenticatedForBusiness)

    @timeit
    def get_object(self):
        for lookup_field in self.lookup_fields:
            try:
                return self._get_object_for_field(lookup_field)
            except Http404:
                pass
        raise Http404

    def get_queryset(self, *args, **kwargs):
        queryset = super(SurveyFormView, self).get_queryset(*args, **kwargs)
        if settings.IS_BUSINESS_SITE:
            if self.request.yauser.is_authenticated():
                if not self.request.user.is_superuser and self.request.orgs:
                    queryset = queryset.filter(Q(org__dir_id__in=self.request.orgs) | Q(is_public=True))
            else:
                queryset = queryset.filter(is_public=True)
        return queryset

    @timeit
    def _get_object_for_field(self, lookup_field):
        lookup = self.kwargs.get(lookup_field, None)
        if not lookup:
            raise Http404

        filter_kwargs = {lookup_field: lookup}

        obj = generics.get_object_or_404(self.queryset, **filter_kwargs)
        # May raise a permission denied
        self.check_object_permissions(self.request, obj)
        return obj

    def get(self, request, *args, **kwargs):
        from events.surveyme.forms_v2 import FormClass

        survey_id = self.kwargs.get('pk') or self.kwargs.get('slug')
        params = {
            'user': request.user,
            'orgs': request.orgs,
            'is_preview': request.query_params.get('preview') == '1',
            'survey_key': request.query_params.get('key'),
            'captcha_controller': CaptchaController(request=request, view=self),
        }

        try:
            form_class = FormClass(survey_id, **params)
        except Survey.DoesNotExist:
            raise Http404

        return Response(form_class.as_dict(get_language()))

    @timeit
    def is_valid(self, form):
        return form.is_valid()

    @timeit
    def post(self, request, *args, **kwargs):
        survey = self.get_object()
        captcha_controller = CaptchaController(request=request, view=self)
        is_show_captcha = self._is_show_captcha(survey, captcha_controller)

        # хак для тестирования форм во время выкатки
        # TECH-2793 "Скрипт для отслеживания падений сервиса во время миграций/выкатки"
        if survey.pk == 1234 and yenv.type == 'testing':
            is_show_captcha = False

        user = self.get_user()
        form = survey.get_form(
            data=request.data,
            files=request.data,  # TODO check files param
            instance=user,
            is_with_captcha=is_show_captcha,
            request=request,
        )

        is_form_valid = self.is_valid(form)
        is_submit_conditions_is_satisfied = self._is_submit_conditions_is_satisfied(survey, form)
        request._request.additional_data_to_log = {
            'is_form_valid': is_form_valid,
            'is_submit_conditions_is_satisfied': is_submit_conditions_is_satisfied,
            'form_errors': form.errors,
        }
        dry_run = request.query_params.get('dry_run')
        if dry_run:
            if is_form_valid and is_submit_conditions_is_satisfied:
                status_code = status.HTTP_200_OK
            else:
                status_code = status.HTTP_400_BAD_REQUEST
            return Response(form.as_dict(), status=status_code)

        if is_form_valid and is_submit_conditions_is_satisfied:
            if survey.captcha_display_mode == 'auto' and not Survey.get_spam_detected(survey.pk):
                if is_show_captcha:
                    captcha_controller.set_enabled(False)
                    captcha_controller.flush_submit_history()
                else:
                    captcha_controller.set_enabled(captcha_controller.is_should_be_enabled())
            return self.save_answer(
                request=request,
                survey=survey,
                form=form,
            )
        return Response(form.as_dict(), status=status.HTTP_400_BAD_REQUEST)

    def _is_show_captcha(self, survey, captcha_controller):
        if survey.captcha_display_mode == 'always' or Survey.get_spam_detected(survey.pk):
            return True
        else:
            return captcha_controller.is_enabled()

    @timeit
    def save_answer(self, request, survey, form):
        try:
            save_response = form.save()
        except MaxSurveyAnswersException:
            raise PermissionDenied

        answer = save_response['profile_survey_answer']

        integrations = start_integrations(
            survey_id=survey.id,
            answer_id=answer.id,
            trigger_slug='create' if save_response['is_profile_survey_answer_created'] else 'update',
        )

        # todo: test me
        active_survey_key_from_bundles = self.get_active_key_from_survey_bundles(survey)
        if active_survey_key_from_bundles:
            active_survey_key_from_bundles.deactivate_with_answer(answer)

        payment_url = None
        payment_question = survey.get_payment_question()
        if payment_question:
            answer_data = answer.as_dict()
            payment_answer = answer_data.get(payment_question.pk)
            if payment_answer:
                account_id = payment_question.param_payment.get('account_id') or ''
                amount = payment_answer.get('value', {}).get('amount')
                if amount:
                    comment = _('Оплата заказа %(order_id)s на сумму %(amount)s руб') % {
                        'order_id': answer.pk,
                        'amount': amount,
                    }
                    order = {
                        'label': str(answer.pk),
                        'paymentType': payment_answer.get('value', {}).get('payment_method'),
                        'quickpay-form': 'shop',
                        'receiver': account_id,
                        'sum': amount,
                        'targets': comment,
                        'successURL': self._get_success_url(),
                    }
                    payment_url = '{url}?{params}'.format(
                        url=settings.YOOMONEY_QUICKPAY_URL,
                        params=urlencode(order),
                    )

        extra = survey.extra or {}
        quiz = SurveyQuiz(extra.get('quiz'))
        return Response(
            data={
                'messages': [],
                'answer_id': answer.id,
                'answer_key': answer.secret_code,
                'integrations': integrations,
                'payment_url': payment_url,
                'show_results': quiz.show_results,
            },
        )

    @timeit
    def _is_submit_conditions_is_satisfied(self, survey, form):
        is_allowed = ConditionEvaluator(
            condition_nodes=survey.get_submit_conditions(with_additional_info=True),
            source_data=form.data,
            fields_by_name=form.get_fields_by_name()
        ).evaluate()
        if is_allowed is not None:
            return is_allowed
        else:
            return True

    def _get_success_url(self):
        if self.request.frontend:
            return self.request.data.get('success_url')


class ScoreResultsView(ExternalGenericApiViewV1Mixin, generics.RetrieveAPIView):
    lookup_field = 'secret_code'
    queryset = ProfileSurveyAnswer.objects.select_related('survey')
    serializer_class = ScoreResultsSerializer


class AnswerResultsView(ExternalGenericApiViewV1Mixin, generics.RetrieveAPIView):
    lookup_field ='secret_code'
    queryset = ProfileSurveyAnswer.objects.select_related('survey')
    serializer_class = AnswerResultsSerializer


class SurveyStatusView(View):
    def get(self, request, pk):
        queryset = (
            Survey.objects.select_related('org')
            .filter(Q(is_ban_detected__isnull=True) | Q(is_ban_detected=False))
            .filter(is_published_external=True)
            .values('pk', 'is_public', 'org__dir_id')
        )
        try:
            survey = queryset.get(pk=pk)
        except Survey.DoesNotExist:
            return Http404Response()

        return JsonResponse({
            'id': survey['pk'],
            'is_public': survey['is_public'],
            'org_id': survey['org__dir_id'],
        })
