# -*- coding: utf-8 -*-
import json

from django.conf import settings
from rest_framework import serializers
from urllib.parse import urlparse, urljoin

from events.rest_framework_contrib.serializers import InternalModelSerializerV2Mixin, DictIdSerializer
from events.rest_framework_contrib.fields import FilterHeadersJSONField
from events.surveyme_integration.models import (
    SurveyHookTrigger,
    SurveyHook,
    ServiceType,
    ServiceTypeAction,
    ServiceSurveyHookSubscription,
    IntegrationFileTemplate,
    HookSubscriptionNotification,
)
from events.followme.api_admin.v2.serializers import UserFollowerSerializer


class ServiceTypeSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = ServiceType
        fields = (
            'id',
            'title',
            'slug',
        )


class ServiceTypeActionSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    service_type = ServiceTypeSerializer()

    class Meta:
        model = ServiceTypeAction
        fields = (
            'id',
            'service_type',
            'title',
            'slug',
        )


class SurveyHookTriggerSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = SurveyHookTrigger
        fields = (
            'id',
            'title',
            'slug',
        )


class SurveyHookSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = SurveyHook
        fields = (
            'id',
            'survey',
            'is_active',
        )


class VariableCategorySerializer(serializers.Serializer):
    name = serializers.CharField()
    title = serializers.CharField()


class VariableRendererClassesSerializer(serializers.Serializer):
    format_name = serializers.CharField()
    format_title = serializers.CharField()


class VariableSerializer(serializers.Serializer):
    name = serializers.CharField()
    title = serializers.CharField()
    arguments = serializers.ReadOnlyField()
    filters = serializers.ReadOnlyField()
    connect_only = serializers.ReadOnlyField()
    category = VariableCategorySerializer()
    renderer_classes = VariableRendererClassesSerializer(many=True)


class FilterSerializer(serializers.Serializer):
    name = serializers.CharField()
    title = serializers.CharField()


class StatusSerializer(serializers.Serializer):
    name = serializers.CharField()
    title = serializers.CharField()
    could_be_canceled = serializers.BooleanField()
    could_be_restarted = serializers.BooleanField()


class SurveyHookForSubscriptionSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    survey_id = serializers.IntegerField(read_only=True)

    class Meta:
        model = SurveyHook
        fields = (
            'id',
            'survey_id'
        )


class ServiceSurveyHookSubscriptionSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    survey_hook = SurveyHookForSubscriptionSerializer(read_only=True)

    class Meta:
        model = ServiceSurveyHookSubscription
        fields = (
            'id',
            'service_type_action',
            'survey_hook'
        )


class IntegrationFileTemplateSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = IntegrationFileTemplate
        fields = (
            'id',
            'name',
            'template',
            'type',
            'slug',
        )


class HookSubscriptionNotificationSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    ID_FIELDS = {'celery_task_id', '_id', 'notification_id', }

    profile = UserFollowerSerializer(read_only=True, source='user')
    error_message = serializers.SerializerMethodField()
    _id = serializers.CharField(source='id')
    response = serializers.JSONField(required=False, allow_null=True)
    trigger_data = serializers.JSONField(required=False, allow_null=True)
    context = FilterHeadersJSONField(required=False, allow_null=True)
    error = serializers.JSONField(required=False, allow_null=True)

    def get_error_message(self, obj):
        error = obj.error or {}
        error_message = error.get('message', '')
        return error_message.split('\n')[0]

    def build_field(self, field_name, info, model_class, nested_depth):
        if field_name.endswith('_id') and field_name not in self.ID_FIELDS:
            field_name = field_name[:-3]
        return super(HookSubscriptionNotificationSerializer, self).build_field(field_name, info, model_class, nested_depth)

    class Meta:
        model = HookSubscriptionNotification
        fields = (
            'id',
            '_id',
            'notification_id',
            'status',
            'profile',
            'answer',
            'answer_id',
            'max_retries',
            'survey_group_id',
            'trigger_slug',
            'celery_task_id',
            'survey',
            'survey_id',
            'subscription',
            'subscription_id',
            'date_created',
            'error_message',
            'is_visible',
            'error',
            'date_updated',
            'date_finished',
            'date_retry',
            'retries',
            'response',
            'trigger_data',
            'context',
        )


class HookSubscriptionNotificationShortSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    profile = UserFollowerSerializer(read_only=True, source='user')
    error_message = serializers.SerializerMethodField()

    def get_error_message(self, obj):
        error = obj.error or {}
        error_message = error.get('message', '')
        return error_message.split('\n')[0]

    class Meta:
        model = HookSubscriptionNotification
        fields = (
            'id',
            'profile',
            'answer',
            'survey',
            'subscription',
            'date_created',
            'error_message',
        )


class FieldRestrictionSerializer(serializers.Serializer):
    field_name = serializers.CharField()
    filters = serializers.DictField()
    variables = serializers.ReadOnlyField()


class HookSubscriptionNotificationResultSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    id = serializers.CharField(source='subscription_id')
    integration_type = serializers.CharField(source='subscription__service_type_action__service_type__slug')
    resources = serializers.SerializerMethodField()

    def parse_json(self, obj, field_name):
        obj = obj.get(field_name) or {}
        if isinstance(obj, str):
            try:
                obj = json.loads(obj)
            except ValueError:
                obj = {}
        return obj

    def set_startrek_resource(self, obj, resources):
        if obj.get('status') == 'success':
            response = self.parse_json(obj, 'response')
            issue = response.get('issue') or {}
            key = issue.get('key')
            title = issue.get('summary')
            if not title:
                context = self.parse_json(obj, 'context')
                title = context.get('title')
            resources['key'] = f'{key}: {title}' if title else key
            resources['link'] = urljoin(settings.STARTREK_HOST, key)

    def set_http_resource(self, obj, resources):
        context = self.parse_json(obj, 'context')
        url = context.get('url')
        resources['url'] = urlparse(url).netloc

    def set_json_rpc_resource(self, obj, resources):
        self.set_http_resource(obj, resources)

    def set_wiki_resource(self, obj, resources):
        context = self.parse_json(obj, 'context')
        supertag = context.get('supertag')
        resources['wiki_page'] = supertag
        resources['link'] = urljoin(settings.WIKI_HOST, supertag)

    def set_email_resource(self, obj, resources):
        context = self.parse_json(obj, 'context')
        resources['to_address'] = context.get('to_address')

    def get_resources(self, obj):
        resources = {}
        status = obj.get('status')
        if status != 'pending':
            integration_type = obj.get('subscription__service_type_action__service_type__slug')
            method = getattr(self, f'set_{integration_type}_resource')
            method(obj, resources)
            if status == 'error':
                error = self.parse_json(obj, 'error')
                resources['message'] = error.get('message')
        return resources

    class Meta:
        model = HookSubscriptionNotification
        list_serializer_class = DictIdSerializer
        fields = (
            'id',
            'status',
            'integration_type',
            'resources',
        )
