# -*- coding: utf-8 -*-
from django.conf import settings
from django.utils import timezone
from rest_framework import viewsets, views, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.settings import api_settings

from events.rest_framework_contrib.mixins import (
    InternalGenericApiViewV2Mixin,
    InternalGenericApiViewV2MixinWithPermissions,
)
from events.rest_framework_contrib.pagination import LargePagination
from events.surveyme.models import Survey
from events.surveyme_integration.api_admin.v2.filters import (
    ServiceSurveyHookSubscriptionFilter,
    IntegrationFileTemplateFilter,
    HookSubscriptionNotificationFilter,
    HookSubscriptionNotificationFilterBackend,
)
from events.surveyme_integration.api_admin.v2.serializers import (
    SurveyHookTriggerSerializer,
    SurveyHookSerializer,
    ServiceTypeSerializer,
    ServiceTypeActionSerializer,
    VariableSerializer,
    FilterSerializer,
    VariableCategorySerializer,
    ServiceSurveyHookSubscriptionSerializer,
    StatusSerializer,
    IntegrationFileTemplateSerializer,
    HookSubscriptionNotificationSerializer,
    FieldRestrictionSerializer,
)
from events.surveyme_integration.filters import filters_list
from events.surveyme_integration.models import (
    SurveyHookTrigger,
    SurveyHook,
    ServiceType,
    ServiceTypeAction,
    ServiceSurveyHookSubscription,
    IntegrationFileTemplate,
    HookSubscriptionNotification,
)
from events.surveyme_integration.statuses import statuses_list, statuses_could_be_canceled, \
    statuses_could_be_restarted
from events.surveyme_integration.tasks import start_notification
from events.surveyme_integration.utils import update_notification_counter
from events.surveyme_integration.variables import variables_list, variable_categories_list
from events.surveyme_integration.variables.fields_restrictions import fields_restrictions_list
from events.history.models import HistoryRawEntry
from events.guardian_contrib.api_admin.v2.views_api import UsersWithPermissionsMixin


import logging


logger = logging.getLogger(__name__)


class IntegrationFileTemplateViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = IntegrationFileTemplateSerializer
    queryset = IntegrationFileTemplate.objects.all()
    filter_class = IntegrationFileTemplateFilter


class SurveyHookTriggerViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = SurveyHookTriggerSerializer
    queryset = SurveyHookTrigger.objects.all()


class SurveyHookViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = SurveyHookSerializer
    queryset = SurveyHook.objects.all()


class ServiceTypeViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = ServiceTypeSerializer
    queryset = ServiceType.objects.all()


class ServiceTypeActionViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = ServiceTypeActionSerializer
    queryset = ServiceTypeAction.objects.all().select_related('service_type')


class ServiceSurveyHookSubscriptionViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = ServiceSurveyHookSubscriptionSerializer
    queryset = ServiceSurveyHookSubscription.objects.all().select_related('survey_hook')
    filter_class = ServiceSurveyHookSubscriptionFilter


class VariableCategoryView(InternalGenericApiViewV2Mixin, views.APIView):
    def get(self, request, *args, **kwargs):
        return Response(data=VariableCategorySerializer(variable_categories_list, many=True).data)


class VariableView(InternalGenericApiViewV2Mixin, views.APIView):
    def is_b2b(self, survey_id):
        try:
            org_id = Survey.objects.values_list('org_id', flat=True).get(pk=survey_id)
            return org_id is not None
        except Survey.DoesNotExist:
            return False

    def check_granted(self, var, b2b):
        if var.connect_only:
            return b2b
        return True

    def get(self, request, *args, **kwargs):
        _variables_list = variables_list
        if settings.IS_BUSINESS_SITE:
            b2b = self.is_b2b(request.query_params.get('survey'))
            _variables_list = [
                var
                for var in _variables_list
                if self.check_granted(var, b2b)
            ]
        return Response(data=VariableSerializer(_variables_list, many=True).data)


class FilterView(InternalGenericApiViewV2Mixin, views.APIView):
    def get(self, request, *args, **kwargs):
        return Response(data=FilterSerializer(filters_list, many=True).data)


class StatusView(InternalGenericApiViewV2Mixin, views.APIView):
    def get(self, request, *args, **kwargs):
        return Response(data=StatusSerializer(statuses_list, many=True).data)


class HookSubscriptionNotificationViewSet(UsersWithPermissionsMixin,
                                          InternalGenericApiViewV2MixinWithPermissions,
                                          viewsets.ModelViewSet):
    serializer_class = HookSubscriptionNotificationSerializer
    queryset = (
        HookSubscriptionNotification.objects.all()
        .select_related('user')
    )
    filter_backends = api_settings.DEFAULT_FILTER_BACKENDS + [HookSubscriptionNotificationFilterBackend]
    filter_class = HookSubscriptionNotificationFilter
    ordering = ('-date_created',)
    RESTART_STATUSES = [i.name for i in statuses_could_be_restarted]
    CANCEL_STATUSES = [i.name for i in statuses_could_be_canceled]

    @action(methods=['post'], url_path='cancel', detail=True)
    def cancel(self, request, pk):
        error_msg = 'Notification with status "%s" could not be canceled'
        action = '_cancel'
        return self._single_action(
            action=action, message=error_msg,
            request=request, statuses=self.CANCEL_STATUSES,
        )

    @action(methods=['post'], url_path='cancel', detail=False)
    def cancel_list(self, request):
        error_msg = (
            'В списке интеграций есть интеграции, статус которых не позволяет выполнить их отмену. '
            'Разрешена отмена интеграций со статусами:%s' % ''.join([
                '\n- %s' % i.title
                for i in statuses_could_be_canceled
            ])
        )
        action = '_cancel'
        return self._list_action(action=action, message=error_msg, statuses=self.CANCEL_STATUSES)

    def _cancel(self, obj):
        obj.date_finished = timezone.now()
        obj.status = 'canceled'
        obj.save(update_fields=['date_finished', 'status'])

        self._save_action_to_history(obj)

    def _list_action(self, action, message, statuses):
        queryset = self.filter_queryset(self.get_queryset())
        if queryset.exclude(status__in=statuses).count():
            return Response(
                {'error_detail': message},
                status=status.HTTP_400_BAD_REQUEST
            )
        else:
            subscriptions = set()
            for notification in queryset:
                getattr(self, action)(notification)
                subscriptions.add(notification.subscription_id)
            for subscription in subscriptions:
                update_notification_counter(subscription)
            return Response({'status': 'ok'})

    def _single_action(self, action, message, request, statuses):
        obj = self.get_object()
        if obj.status in statuses:
            subscription = obj.subscription_id
            getattr(self, action)(obj)
            update_notification_counter(subscription)
            return self.retrieve(request, id=obj.id)
        else:
            return Response(
                {'error_detail': message % obj.status},
                status=status.HTTP_400_BAD_REQUEST
            )

    @action(methods=['post'], url_path='restart', detail=True)
    def restart(self, request, pk):
        error_msg = 'Notification with status "%s" could not be restarted'
        action = '_restart'
        return self._single_action(
            action=action, message=error_msg,
            request=request, statuses=self.RESTART_STATUSES,
        )

    @action(methods=['post'], url_path='restart', detail=False)
    def restart_list(self, request):
        error_msg = (
            'В списке интеграций есть интеграции, статус которых не позволяет выполнить их перезапуск. '
            'Разрешен перезапуск интеграций со статусами:%s' % ''.join([
                '\n- %s' % i.title
                for i in statuses_could_be_restarted
            ])
        )
        action = '_restart'
        return self._list_action(action=action, message=error_msg, statuses=self.RESTART_STATUSES)

    def _restart(self, obj):
        obj.status = 'pending'
        obj.retries = 0
        obj.save(update_fields=['status', 'retries'])

        start_notification(obj.id)
        self._save_action_to_history(obj)

    def _save_action_to_history(self, obj):
        HistoryRawEntry.objects.create_entry(obj)

    def perform_update(self, serializer):
        obj = self.get_object()
        if 'is_visible' in serializer.validated_data:
            obj.make_visible(serializer.validated_data['is_visible'])


class FieldRestrictionView(InternalGenericApiViewV2Mixin, views.APIView):
    pagination_class = LargePagination

    def get(self, request, *args, **kwargs):
        return Response(data=FieldRestrictionSerializer(fields_restrictions_list, many=True).data)
