# -*- coding: utf-8 -*-
import base64
import hashlib
import lazy_object_proxy
import re
import urllib.parse

from django.utils.http import urlquote
from django.utils.encoding import force_str
from django.utils.translation import ugettext_lazy as _
from enum import Enum
from json import dumps as json_dumps


class BaseFilter(object):
    name = None
    title = None

    def clean_value(self, value):
        if not value:
            return ''
        else:
            return force_str(value)

    def apply_filter(self, value):
        return self.clean_value(value)


class MD5Filter(BaseFilter):
    name = 'md5'
    title = _('MD5')

    def apply_filter(self, value):
        value = super(MD5Filter, self).apply_filter(value)
        return hashlib.md5(value.encode('utf-8')).hexdigest()


class Base64Filter(BaseFilter):
    name = 'base64'
    title = _('Base64')

    def apply_filter(self, value):
        value = super(Base64Filter, self).apply_filter(value)
        return force_str(base64.b64encode(value.encode('utf-8')))


class LowerFilter(BaseFilter):
    name = 'lower'
    title = _('В нижний регистр')

    def apply_filter(self, value):
        value = super(LowerFilter, self).apply_filter(value)
        return value.lower()


class UpperFilter(BaseFilter):
    name = 'upper'
    title = _('В верхний регистр')

    def apply_filter(self, value):
        value = super(UpperFilter, self).apply_filter(value)
        return value.upper()


class URLEncodeFilter(BaseFilter):
    name = 'urlencode'
    title = _('URL-encode')

    def apply_filter(self, value):
        value = super(URLEncodeFilter, self).apply_filter(value)
        return urlquote(value)


class PunycodeFilter(BaseFilter):
    name = 'punycode'
    title = _('Punycode')

    def apply_filter(self, value):
        value = super(PunycodeFilter, self).apply_filter(value)
        parsed_url = urllib.parse.urlparse(value)
        if not parsed_url.netloc:
            return force_str(force_str(value).encode('idna'))
        parsed_url = parsed_url._replace(netloc=force_str(force_str(parsed_url.netloc).encode('idna')))
        return parsed_url.geturl()


class LoginFilter(BaseFilter):
    name = 'login'
    title = _('Логин на стаффе')
    filter_re = lazy_object_proxy.Proxy(lambda: re.compile(r'\(([^)]+)\)$'))

    def get_login(self, value):
        m = self.filter_re.search(value)
        if m:
            value = m.group(1)
        return value.lower()

    def split_value(self, value):
        return [
            self.get_login(it.strip())
            for it in value.split(',')
        ]

    def apply_filter(self, value):
        value = super(LoginFilter, self).apply_filter(value)
        return ', '.join(self.split_value(value))


class HeaderSanitizeFilter(BaseFilter):
    name = 'sanitize'
    title = _('Sanitize string')
    filter_re = lazy_object_proxy.Proxy(lambda: re.compile(r'[^\w\.,;_-]+', re.UNICODE))

    def apply_filter(self, value):
        value = super(HeaderSanitizeFilter, self).apply_filter(value)
        return self.filter_re.sub(' ', value).strip()


class JsonFilter(BaseFilter):
    name = 'json'
    title = _('Json string')

    def apply_filter(self, value):
        value = json_dumps(super().apply_filter(value))
        if value:
            return value[1:-1]


filters_list = [
    MD5Filter,
    Base64Filter,
    LowerFilter,
    UpperFilter,
    URLEncodeFilter,
    PunycodeFilter,
    LoginFilter,
    HeaderSanitizeFilter,
    JsonFilter,
]


filters_by_name = dict([(i.name, i) for i in filters_list])


class _StrEnum(str, Enum):
    pass

FilterTypes = _StrEnum('FilterTypes', {name: name for name in filters_by_name})
