# -*- coding: utf-8 -*-
import base64
import json
import os
import responses
import time

from collections import OrderedDict
from django.conf import settings
from django.core.files.base import ContentFile
from django.utils.encoding import force_str
from rest_framework.throttling import SimpleRateThrottle
from unittest.mock import patch

from events.accounts.factories import UserFactory
from events.accounts.helpers import YandexClient
from events.surveyme.factories import (
    SurveyFactory,
    SurveyQuestionFactory,
    ProfileSurveyAnswerFactory,
)
from events.surveyme.models import AnswerType, SurveyQuestionChoice
from events.surveyme_integration.exceptions import RetriableError
from events.surveyme_integration.factories import (
    SurveyHookFactory,
    ServiceSurveyHookSubscriptionFactory,
    SurveyVariableFactory,
)
from events.surveyme_integration.filters import (
    Base64Filter,
    MD5Filter,
    URLEncodeFilter,
    UpperFilter,
)
from events.surveyme_integration.variables import (
    FormQuestionAnswerVariable,
    FormIdVariable,
    FormNameVariable,
    FormQuestionsAnswersVariable,
    StaffMetaUserVariable,
)
from events.surveyme_integration.variables.staff import (
    StaffLoginVariableRenderer,
    StaffBirthDateVariableRenderer,
)
from events.surveyme_integration.services.http.services import HTTPService
from events.surveyme_integration.services.base.context_processors.base import (
    ServiceContextInstance,
    ContextProcessorBase,
)
from events.common_storages.models import ProxyStorageModel
from events.common_storages.proxy_storages import ProxyStorage
from events.common_storages.storage import MdsStorage
from events.arc_compat import read_asset


class IntegrationTestMixin:
    fixtures = ['initial_data.json']
    client_class = YandexClient

    def setUp(self):
        self.former_timer = SimpleRateThrottle.timer
        SimpleRateThrottle.timer = time.time
        self.user = UserFactory(username='web-chib', uid='1120000000018124')
        self.survey = SurveyFactory(
            name='Some_feedback_form',
            is_public=True,
            is_published_external=True,
            user=self.user,
        )
        self.questions = {
            'name': SurveyQuestionFactory(
                label='name',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_short_text'),
                param_is_required=False,
            ),
            'some_text': SurveyQuestionFactory(
                label='some_text',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_short_text'),
                param_is_required=False,
            ),
            'career': SurveyQuestionFactory(
                label='career',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_choices')
            ),
            'email': SurveyQuestionFactory(
                label='email',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_short_text')
            ),
        }

        # create choices for career
        self.career_choices = {
            'working': SurveyQuestionChoice.objects.create(
                label='working',
                survey_question=self.questions['career'],
                slug='working_choice',
            ),
            'studying': SurveyQuestionChoice.objects.create(
                label='studying',
                survey_question=self.questions['career'],
                slug='studying_choice',
            ),
        }

        self.data = {
            self.questions['email'].get_form_field_name(): 'web-chib@ya.com',
            self.questions['name'].get_form_field_name(): 'Gennady',
            self.questions['career'].get_form_field_name(): [self.career_choices['working'].id],
            self.questions['some_text'].get_form_field_name(): 'hello me',
        }

        # create integration hook
        self.survey_hook = SurveyHookFactory(survey=self.survey)

        self.subscription = ServiceSurveyHookSubscriptionFactory(survey_hook=self.survey_hook)

        form_name_variable = SurveyVariableFactory(
            variable_id='575176514507946508000002',
            hook_subscription=self.subscription,
            var=FormNameVariable.name,
        )
        email_answer_value_variable = SurveyVariableFactory(
            variable_id='57554dc3c67f552562000002',
            hook_subscription=self.subscription,
            var=FormQuestionAnswerVariable.name,
            arguments={'question': self.questions['email'].id},
        )
        staff_birthday_variable = SurveyVariableFactory(
            variable_id='57554dc3c32f559862000002',
            hook_subscription=self.subscription,
            var=StaffMetaUserVariable.name,
            format_name=StaffBirthDateVariableRenderer.format_name,
        )
        form_answers_variable = SurveyVariableFactory(
            variable_id='57554dc3c32f559844000002',
            hook_subscription=self.subscription,
            var=FormQuestionsAnswersVariable.name,
            arguments={'is_all_questions': True},
        )
        staff_login_variable = SurveyVariableFactory(
            variable_id='57554dc3c67f559862000002',
            hook_subscription=self.subscription,
            var=StaffMetaUserVariable.name,
            format_name=StaffLoginVariableRenderer.format_name,
        )

        # variables
        self.variables_ids = {
            'form_name': form_name_variable.variable_id,
            'email_answer_value': email_answer_value_variable.variable_id,
            'form_answers': form_answers_variable.variable_id,
            'staff_login': staff_login_variable.variable_id,
            'staff_birthday': staff_birthday_variable.variable_id,
        }
        self.variables = {
            variable.variable_id: variable
            for variable in (
                form_name_variable, email_answer_value_variable,
                form_answers_variable, staff_birthday_variable,
                staff_login_variable
            )
        }
        self.variables_result_data = {
            'form_name': self.survey.name,
            'email_answer_value': 'web-chib@ya.com',
            'form_answers': ('name:\n'
                             'Gennady\n'
                             '\n'
                             'some_text:\n'
                             'hello me\n'
                             '\n'
                             'career:\n'
                             'working\n'
                             '\n'
                             'email:\n'
                             'web-chib@ya.com'),
            'staff_login': 'zomb-prj-124',
            'staff_birthday': '2013-05-20',
        }
        self.variables_filtered_result_data = {}
        self.variables_filtered_result_data['form_name'] = (
            MD5Filter().apply_filter(
                self.variables_result_data['form_name']
            )
        )
        self.variables_filtered_result_data['email_answer_value'] = (
            self.variables_result_data['email_answer_value']
        )
        self.variables_filtered_result_data['form_answers'] = (
            URLEncodeFilter().apply_filter(
                UpperFilter().apply_filter(
                    self.variables_result_data['form_answers']
                )
            )
        )

    def tearDown(self):
        SimpleRateThrottle.timer = self.former_timer

    def post_data(self, client=None):
        client = client or self.client
        response = client.post(f'/v1/surveys/{self.survey.pk}/form/', data=self.data)
        self.assertEqual(response.status_code, 200)
        return response

    def get_message_id(self, response):
        notification = self.subscription.get_all_notifications().first()
        return '%s.%s.%s.%s@%s' % (
            settings.APP_TYPE,
            self.survey.pk,
            response.data['answer_id'],
            notification.pk,
            settings.HOSTNAME,
        )

    def add_filters_to_variables(self):
        form_name_variable = self.variables[self.variables_ids['form_name']]
        form_name_variable.filters = [MD5Filter.name]
        form_name_variable.save()

        email_variable = self.variables[self.variables_ids['email_answer_value']]
        email_variable.filters = []
        email_variable.save()

        form_answers_variable = self.variables[self.variables_ids['form_answers']]
        form_answers_variable.filters = [UpperFilter.name, URLEncodeFilter.name]
        form_answers_variable.save()


class AddVariablesToSubscriptionMixin:
    def setUp(self):
        # variables

        self.variable_value = self.survey.id

        self.variable_with_filters_value = MD5Filter().apply_filter(
            Base64Filter().apply_filter(
                self.survey.id
            )
        )
        form_id_variable = SurveyVariableFactory(
            variable_id='575176514507946508000004',
            hook_subscription=self.subscription,
            var=FormIdVariable.name,
        )
        form_id_filter_variable = SurveyVariableFactory(
            variable_id='575176514507946508000005',
            hook_subscription=self.subscription,
            var=FormIdVariable.name,
            filters=[Base64Filter.name, MD5Filter.name],
        )
        self.variable_id = form_id_variable.variable_id
        self.variable_with_filters_id = form_id_filter_variable.variable_id


class RenderedCharFieldTestMixin(AddVariablesToSubscriptionMixin):
    serializer_class = None
    field_name = None
    subscription_source = None

    def setUp(self):
        self.unprocessed_value = 'Привет'
        self.subscription = ServiceSurveyHookSubscriptionFactory()
        setattr(self.subscription, self.subscription_source, self.unprocessed_value)
        self.answer = ProfileSurveyAnswerFactory()
        self.survey = self.answer.survey
        self.trigger_data = {}
        self.notification_unique_id = '123'
        self.service_context_instance = ServiceContextInstance(
            subscription=self.subscription,
            answer=self.answer,
            trigger_data=self.trigger_data,
            notification_unique_id=self.notification_unique_id,
        )
        super(RenderedCharFieldTestMixin, self).setUp()

    def test_simple(self):
        serializer = self.serializer_class(self.service_context_instance)
        self.assertEqual(serializer.data[self.field_name], self.unprocessed_value)

    def test_with_variables(self):
        expected = '%s, %s' % (self.unprocessed_value, self.variable_value)
        setattr(
            self.subscription,
            self.subscription_source,
            '%s, {%s}' % (self.unprocessed_value, self.variable_id)
        )
        serializer = self.serializer_class(self.service_context_instance)
        self.assertEqual(serializer.data[self.field_name], expected)

    def test_with_variables_and_filters(self):
        expected = '%s, %s' % (self.unprocessed_value, self.variable_with_filters_value)
        setattr(
            self.subscription,
            self.subscription_source,
            '%s, {%s}' % (self.unprocessed_value, self.variable_with_filters_id)
        )
        serializer = self.serializer_class(self.service_context_instance)
        self.assertEqual(serializer.data[self.field_name], expected)


class HeadersFieldMixin(AddVariablesToSubscriptionMixin):
    serializer_class = None
    field_name = None
    required_default_headers = None

    def setUp(self):
        self.subscription = ServiceSurveyHookSubscriptionFactory()
        self.answer = ProfileSurveyAnswerFactory()
        self.survey = self.answer.survey

        # create service context instance
        self.trigger_data = {}
        self.notification_unique_id = '456'
        self.service_context_instance = ServiceContextInstance(
            subscription=self.subscription,
            answer=self.answer,
            trigger_data=self.trigger_data,
            notification_unique_id=self.notification_unique_id,
        )

        # add some headers
        self.headers = [
            self.subscription.headers.create(name='X-GEOBASE', value='1'),
            self.subscription.headers.create(name='X-ID', value='2'),
        ]
        self.headers_unprocessed = [
            {
                'name': self.headers[0].name,
                'value': self.headers[0].value
            },
            {
                'name': self.headers[1].name,
                'value': self.headers[1].value
            },
        ]

        if self.required_default_headers:
            self.required_default_headers_dict = self.required_default_headers(service_context_instance=self.service_context_instance)
        else:
            self.required_default_headers_dict = {}

        super(HeadersFieldMixin, self).setUp()

    def get_message_id(self, notification_id):
        return '%s.%s.%s.%s@%s' % (
            settings.APP_TYPE,
            self.survey.pk,
            self.answer.pk,
            notification_id,
            settings.HOSTNAME,
        )

    def test_simple(self):
        serializer = self.serializer_class(self.service_context_instance)
        expected = {
            self.headers[0].name: self.headers[0].value,
            self.headers[1].name: self.headers[1].value
        }
        expected.update(self.required_default_headers_dict)
        response = serializer.data[self.field_name]
        response.pop('X-SO-Form-Spam', None)
        self.assertEqual(response, expected)

    def test_with_variables(self):
        self.headers[0].name = '%s, name {%s}' % (self.headers_unprocessed[0]['name'], self.variable_id)
        self.headers[0].value = '%s, value {%s}' % (self.headers_unprocessed[0]['value'], self.variable_id)
        self.headers[0].save()
        self.headers[1].name = '%s, name {%s}' % (self.headers_unprocessed[1]['name'], self.variable_id)
        self.headers[1].value = '%s, value {%s}' % (self.headers_unprocessed[1]['value'], self.variable_id)
        self.headers[1].save()
        expected = {
            self.headers_unprocessed[0]['name'] + ', name %s' % self.variable_value:
                self.headers_unprocessed[0]['value'] + ', value %s' % self.variable_value,
            self.headers_unprocessed[1]['name'] + ', name %s' % self.variable_value:
                self.headers_unprocessed[1]['value'] + ', value %s' % self.variable_value,
        }
        expected.update(self.required_default_headers_dict)
        serializer = self.serializer_class(self.service_context_instance)
        response = serializer.data[self.field_name]
        response.pop('X-SO-Form-Spam', None)
        self.assertEqual(response, expected)

    def test_with_variables_and_filters(self):
        self.headers[0].name = '%s, name {%s}' % (self.headers_unprocessed[0]['name'], self.variable_with_filters_id)
        self.headers[0].value = '%s, value {%s}' % (self.headers_unprocessed[0]['value'], self.variable_with_filters_id)
        self.headers[0].save()
        self.headers[1].name = '%s, name {%s}' % (self.headers_unprocessed[1]['name'], self.variable_with_filters_id)
        self.headers[1].value = '%s, value {%s}' % (self.headers_unprocessed[1]['value'], self.variable_with_filters_id)
        self.headers[1].save()
        expected = {
            self.headers_unprocessed[0]['name'] + ', name %s' % self.variable_with_filters_value:
                self.headers_unprocessed[0]['value'] + ', value %s' % self.variable_with_filters_value,
            self.headers_unprocessed[1]['name'] + ', name %s' % self.variable_with_filters_value:
                self.headers_unprocessed[1]['value'] + ', value %s' % self.variable_with_filters_value,
        }
        expected.update(self.required_default_headers_dict)
        serializer = self.serializer_class(self.service_context_instance)
        response = serializer.data[self.field_name]
        response.pop('X-SO-Form-Spam', None)
        self.assertEqual(response, expected)


class PrepareTextAndFileQuestionsWithAnswersMixin:
    def setUp(self):
        self.subscription = ServiceSurveyHookSubscriptionFactory()
        self.answer = ProfileSurveyAnswerFactory()
        self.survey = self.answer.survey

        # create service context instance
        self.trigger_data = {}
        self.notification_unique_id = '456'
        self.service_context_instance = ServiceContextInstance(
            subscription=self.subscription,
            answer=self.answer,
            trigger_data=self.trigger_data,
            notification_unique_id=self.notification_unique_id,
        )

        # add some questions
        self.questions = {
            'name': SurveyQuestionFactory(
                label='Имя',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_short_text')
            ),
            'surname': SurveyQuestionFactory(
                label='Фамилия',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_short_text')
            ),
            'avatar': SurveyQuestionFactory(
                label='Аватар',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_files')
            ),
            'resume': SurveyQuestionFactory(
                label='Резюме',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_files'),
                param_is_required=False
            ),
            'photos': SurveyQuestionFactory(
                label='Фотографии',
                survey=self.survey,
                answer_type=AnswerType.objects.get(slug='answer_files'),
                param_is_required=False
            ),
        }

        # we need to create file answers because of post_save signal (need meta backend info about object)
        with patch('events.common_storages.storage.generate_code') as mock_generate:
            mock_generate.return_value = 'b9j2a642b859b570557ab890943c999g'
            with patch.object(MdsStorage, '_save') as mock_save:
                mock_save.return_value = '401/{}_{}'.format('b9j2a642b859b570557ab890943c999g', 'ava.jpeg')
                self.ava_file = self._save_file('ava.jpeg')

        with patch('events.common_storages.storage.generate_code') as mock_generate:
            mock_generate.return_value = 'b9j2a642b859b330557ab191943c999g'
            with patch.object(MdsStorage, '_save') as mock_save:
                mock_save.return_value = '401/{}_{}'.format('b9j2a642b859b330557ab191943c999g', 'resume.txt')
                self.resume_file = self._save_file('resume.txt')

        with patch('events.common_storages.storage.generate_code') as mock_generate:
            mock_generate.return_value = 'b9j2a242b859b371557ab890943c399g'
            with patch.object(MdsStorage, '_save') as mock_save:
                mock_save.return_value = '401/{}_{}'.format('b9j2a242b859b371557ab890943c399g', 'photo_1.jpeg')
                photo_1 = self._save_file('photo_1.jpeg')

        with patch('events.common_storages.storage.generate_code') as mock_generate:
            mock_generate.return_value = 'b9j1a642b859b378557ab890943c499g'
            with patch.object(MdsStorage, '_save') as mock_save:
                mock_save.return_value = '401/{}_{}'.format('b9j1a642b859b378557ab890943c499g', 'photo_2.jpeg')
                photo_2 = self._save_file('photo_2.jpeg')

        self.file_question_answers = {
            'avatar': [self.ava_file],
            'resume': [self.resume_file],
            'photos': [photo_1, photo_2],
        }

        # создаем словарь вида {filename: path}
        self.file_question_answer_paths = {}
        for key, value in list(self.file_question_answers.items()):
            for meta_info in value:
                filename = os.path.basename(meta_info.original_name)
                self.file_question_answer_paths[filename] = self.questions[key], meta_info.path

        self.answer.data = {
            'data': [
                {
                    'question': self.questions['name'].get_answer_info(),
                    'value': 'Gennady',
                },
                {
                    'question': self.questions['surname'].get_answer_info(),
                    'value': 'Chibisov',
                },
            ],
        }
        self.answer.save()

    def _save_file(self, filename):
        proxy_storage = ProxyStorage()
        buf = read_asset(os.path.join(settings.FIXTURES_DIR, 'files', filename))
        f = ContentFile(content=buf, name=filename)
        path = proxy_storage.save(filename, f)
        return ProxyStorageModel.objects.get(path=path)

    def init_files(self, *files):
        answer_data = self.answer.data['data']
        for file_name in files:
            answer_data.append({
                'question': self.questions[file_name].get_answer_info(),
                'value': [
                    {
                        'path': meta_info.path,
                    }
                    for meta_info in self.file_question_answers[file_name]
                ],
            })
        self.answer.data['data'] = answer_data
        self.answer.save()


class AttachmentsFieldMixin(PrepareTextAndFileQuestionsWithAnswersMixin):
    serializer_class = None
    field_name = None
    selective = None

    def setUp(self):
        super(AttachmentsFieldMixin, self).setUp()
        if self.selective:
            self.subscription.is_all_questions = True
            self.subscription.save()

    def assert_files_has_right_format(self, file_names):
        expected = []
        for file_name in file_names:
            question, path = self.file_question_answer_paths[file_name]
            ProxyStorageModel.objects.get(path=path)
            expected.append(self.get_expected_item(path, question))
        serializer = self.serializer_class(self.service_context_instance)
        response = serializer.data[self.field_name]

        for item in (response, expected):
            for data in item:
                data.pop('content_type', None)
                for key, value in data.items():
                    if key == 'question':
                        data['question'] = dict(data['question'])
                        data['question']['label'] = dict(data['question']['label'])
                    if isinstance(value, dict):
                        try:
                            need_decode = value.get('X-META-INFO')
                            if need_decode:
                                value['X-META-INFO'] = json.loads(need_decode)
                            value.pop('X-META-INFO-BASE64', None)
                        except json.decoder.JSONDecodeError:
                            pass
        checked = 0
        for item in response:
            for expected_item in expected:
                if item['path'] == expected_item['path']:
                    self.assertDictEqual(item, expected_item)
                    checked += 1

        self.assertEqual(checked, len(response))

    def get_expected_item(self, path, question):
        frontend_url = ProxyStorage().url(path, site_type='internal')
        meta_backend_object = ProxyStorageModel.objects.get(path=path)
        return {
            'filename': meta_backend_object.original_name or path.split('/')[-1],
            'path': path,
            'namespace': 'forms',
            'frontend_url': frontend_url,
            'question': {
                'id': question.id,
                'label': {
                    'ru': question.label,
                },
            },
        }

    def test_format(self):
        self.init_files('avatar', 'resume', 'photos')
        self.assert_files_has_right_format(['ava.jpeg', 'resume.txt', 'photo_1.jpeg', 'photo_2.jpeg'])

    def test_with_empty_path_to_file(self):
        self.init_files('resume', 'photos')
        self.file_question_answers['avatar'] = []
        self.assert_files_has_right_format(['resume.txt', 'photo_1.jpeg', 'photo_2.jpeg'])

    def test_with_not_existing_answer_for_file_question(self):
        self.init_files('resume')
        del self.file_question_answers['avatar']
        del self.file_question_answers['photos']
        self.assert_files_has_right_format(['resume.txt'])

    def test_should_return_only_checked_files_if_selective_and_selected_only_specific_questions(self):
        if not self.selective:
            return
        self.subscription.is_all_questions = False
        self.subscription.questions.set([self.questions['resume']])
        self.subscription.save()
        self.init_files('resume')
        self.assert_files_has_right_format(['resume.txt'])

    def test_should_return_only_checked_files_if_selective_and_selected_only_specific_questions_for_multiattach(self):
        if not self.selective:
            return
        self.subscription.is_all_questions = False
        self.subscription.questions.set([self.questions['photos']])
        self.subscription.save()
        self.assert_files_has_right_format(['photo_1.jpeg', 'photo_2.jpeg'])

    def test_should_return_empty_list_if_no_attachments(self):
        serializer = self.serializer_class(self.service_context_instance)
        self.assertEqual(serializer.data[self.field_name], [])


class HTTPAttachmentsFieldMixin(AttachmentsFieldMixin):
    expected_formats = None

    def get_expected_item(self, path, question_answer):
        response = super(HTTPAttachmentsFieldMixin, self).get_expected_item(path, question_answer)
        meta_info_header = force_str(self.subscription.render_data_to_http_format(OrderedDict([
            ('frontend_url', response['frontend_url']),
            ('question', response['question']),
        ])))
        response['headers'] = {
            'X-META-INFO': meta_info_header,
            'X-META-INFO-BASE64': force_str(base64.b64encode(meta_info_header.encode('utf-8'))),
        }
        return response

    def test_format(self):
        for expected_format in self.expected_formats:
            self.subscription.http_format_name = expected_format
            self.subscription.save()
            super(HTTPAttachmentsFieldMixin, self).test_format()


class HTTPBodyDataFieldMixin(PrepareTextAndFileQuestionsWithAnswersMixin):
    # todo: me
    expected_formats = None

    def test_format(self):
        # todo: test for different formats
        # todo: test for field name (position)
        serializer = self.serializer_class(self.service_context_instance)
        self.subscription.http_format_name = 'xml'
        self.subscription.save()
        serializer.data[self.field_name]


class ServiceBaseTestCaseMixin:
    service_class = None

    def setUp(self):
        self.service_instance = self.service_class()

        # add files
        buf = read_asset(os.path.join(self.get_files_path(), 'ava.jpeg'))
        avatar_file_object = ContentFile(buf, name='ava.jpeg')

        buf = read_asset(os.path.join(self.get_files_path(), 'resume.txt'))
        resume_file_object = ContentFile(buf, name='resume.txt')

        buf = read_asset(os.path.join(self.get_files_path(), 'resume2.txt'))
        resume2_file_object = ContentFile(buf, name='resume2.txt')

        buf = read_asset(os.path.join(self.get_files_path(), 'test.eml'))
        testeml_file_object = ContentFile(buf, name='test.eml')

        self.files = {
            'avatar': {
                'filename': 'ava.jpeg',
                'object': avatar_file_object,
                'content_type': 'image/jpeg',
            },
            'resume': {
                'filename': 'resume.txt',
                'object': resume_file_object,
                'content_type': 'text/plain;charset=utf-8',
            },
            'resume2': {
                'filename': 'resume2.txt',
                'object': resume2_file_object,
                'content_type': 'text/plain;charset=utf-8',
            },
            'testeml': {
                'filename': 'test.eml',
                'object': testeml_file_object,
                'content_type': 'application/octet-stream',
            },
        }
        self.storage = ProxyStorage()
        pregenerate_id = 'b9j2a642b859b330557ab191943c999g'
        with patch('events.common_storages.storage.generate_code') as mock_generate:
            mock_generate.return_value = pregenerate_id
            for file_item in list(self.files.values()):
                with patch.object(MdsStorage, '_save') as mock_save:
                    mock_save.return_value = '401/{}_{}'.format(pregenerate_id, file_item['filename'])
                    file_item['path'] = self.storage.save(
                        file_item['filename'],
                        file_item['object']
                    )

    def get_files_path(self):
        return os.path.join(settings.FIXTURES_DIR, 'files')

    def do_service_action(self, action, context, status=None):
        if status is None:
            status = 'pending'
        return getattr(self.service_instance, action)(
            data=context,
            status=status
        )

    def test_integraion_retraibleerror_should_be_in_retriable_list(self):
        self.assertIn(RetriableError, self.service_instance.retriable_exceptions)


class ContextProcessorMixin:
    fixtures = ['initial_data.json']
    context_processor_class = ContextProcessorBase

    def setUp(self):
        user = UserFactory()
        self.subscription = ServiceSurveyHookSubscriptionFactory()
        self.answer = ProfileSurveyAnswerFactory()
        self.survey = self.answer.survey
        self.survey.user = user
        self.survey.save()
        self.trigger_data = {}
        self.notification_unique_id = '123'
        self.service_context_instance = ServiceContextInstance(
            subscription=self.subscription,
            answer=self.answer,
            trigger_data=self.trigger_data,
            notification_unique_id=self.notification_unique_id,
        )

    def test_default_fields(self):
        response = self.context_processor_class(self.service_context_instance).data
        self.assertEqual(response['notification_unique_id'], self.notification_unique_id)
        self.assertEqual(response['survey_id'], self.answer.survey.id)
        self.assertEqual(response['answer_id'], self.answer.id)
        self.assertEqual(response['subscription_id'], self.subscription.id)

    def test_force_render_default_field(self):
        response = self.context_processor_class(self.service_context_instance).data
        self.assertEqual(response['force_render'], False)

        self.service_context_instance = ServiceContextInstance(
            subscription=self.subscription,
            answer=self.answer,
            trigger_data=self.trigger_data,
            notification_unique_id=self.notification_unique_id,
            force_render=True,
        )
        response = self.context_processor_class(self.service_context_instance).data
        self.assertEqual(response['force_render'], True)


class HTTPServiceBaseTestCaseMixin(ServiceBaseTestCaseMixin):
    service_class = HTTPService
    expected_status_after_first_submission = 'success'

    def setUp(self):
        super().setUp()
        self.context_data = {
            'url': 'http://yandex.ru/test_url/',
            'body_data': {
                'hello': 'world'
            },
            'body_data_items_count': 1,
            'headers': None,
            'attachments': None
        }

    def register_uri(self, url, status_code=200, headers=None, content='hello', method=responses.POST):
        headers = headers or {}
        responses.add_callback(
            method,
            url,
            callback=self.get_request_callback(status_code, headers=headers, content=content),
        )

    def get_request_callback(self, status_code, headers, content):
        def callback(request):
            _headers = dict(request.headers)
            _headers.update(headers)
            _headers.update({
                'x-mark': 'mark'
            })
            return status_code, _headers, content
        return callback
