# -*- coding: utf-8 -*-
import datetime
import json
import logging

from django.core.exceptions import ValidationError
from django.db import models
from django.utils.functional import cached_property
from django.db.models import Q, F
from django.utils.encoding import force_str
from django.conf import settings
from django.utils.translation import ugettext as _
from django.utils import timezone
from urllib.parse import urlparse

from events.common_storages.models import ProxyStorageModel
from events.common_storages.proxy_storages import ProxyStorage
from events.common_app.fields import JSONFieldForTinyText, JSONFieldForSimpleText
from events.common_app.jsonfield.fields import JSONField as JSONFieldWithFallback
from events.common_app import utils as common_app_utils
from events.common_app.wiki import get_wiki_client
from events.common_app.mixins import TimeStampMixin
from events.surveyme_integration.services.startrek.services import StartrekService
from events.conditions.models import ConditionNodeBase, ConditionItemBase
from events.surveyme.models import Survey, SurveyQuestion
from events.surveyme_integration.exceptions import IntegrationError
from events.surveyme_integration.managers import HookSubscriptionNotificationManager
from events.surveyme_integration.services.http.services import HTTPService
from events.surveyme_integration.services.email.services import EmailService
from events.surveyme_integration.services.json_rpc.services import JSONRPCService
from events.surveyme_integration.services.wiki.services import WikiService


logger = logging.getLogger(__name__)


FORMAT_NAME_CHOICES = (
    ('json', 'json'),
    ('xml', 'xml'),
)

FORMAT_METHOD_CHOICES = (
    ('get', 'GET'),
    ('post', 'POST'),
    ('put', 'PUT'),
    ('delete', 'DELETE'),
    ('patch', 'PATCH'),
)


ATTACHMENT_TYPE_CHOICES = (
    ('pdf', 'pdf'),
    ('txt', 'txt'),
)


NOTIFICATION_STATUS_CHOICES = (
    ('pending', 'pending'),
    ('success', 'success'),
    ('error', 'error'),
    ('canceled', 'canceled'),
)


class ObjectWithVariableMixin(object):

    @cached_property
    def variables_map(self):
        return {
            variable.variable_id: variable
            for variable in self.surveyvariable_set.all()
        }


class ServiceType(models.Model):
    title = models.CharField(max_length=100)
    slug = models.SlugField(unique=True)

    service_classes = {
        'email': EmailService,
        'http': HTTPService,
        'json_rpc': JSONRPCService,
        'startrek': StartrekService,
        'wiki': WikiService,
    }

    def get_service_class(self):
        return self.service_classes[self.slug]


class ServiceTypeAction(models.Model):
    service_type = models.ForeignKey(ServiceType, on_delete=models.CASCADE)
    title = models.CharField(max_length=100)
    slug = models.SlugField()

    class Meta:
        unique_together = (
            ('service_type', 'slug'),
        )

    def do_action(self, data, status):
        from events.surveyme_integration.exceptions import reraise
        try:
            service_instance = self.service_type.get_service_class()()
            return getattr(service_instance, self.slug)(data=data, status=status)
        except IntegrationError:
            raise
        except Exception as e:
            reraise(self.service_type, e)

    def __str__(self):
        # todo: test me
        return '{0}. {1}'.format(self.service_type.title, self.title)


class SurveyHookTrigger(models.Model):
    title = models.CharField(max_length=100)
    slug = models.SlugField(unique=True)


class SurveyHook(ConditionNodeBase):
    triggers = models.ManyToManyField(SurveyHookTrigger, blank=True)
    survey = models.ForeignKey(Survey, related_name='hooks', on_delete=models.CASCADE)
    is_active = models.BooleanField(default=True)
    name = models.CharField(max_length=255, blank=True, null=True)

    class Meta:
        ordering = ('id',)

    def is_true(self, **kwargs):
        if not self.condition_nodes.exists():
            return True
        for node in self.condition_nodes.all():
            if node.is_true(**kwargs):
                return True
        return False

    def notify_subscriptions(self, answer, trigger_slug, trigger_data=None):
        # todo: test me
        subscriptions_data = []
        notifications = []
        for service_hook_subscription in self.subscriptions.all():
            if service_hook_subscription.is_active:
                notifications.append(
                    service_hook_subscription.notify(
                        answer=answer,
                        trigger_slug=trigger_slug,
                        trigger_data=trigger_data
                    )
                )
                if service_hook_subscription.follow_result:
                    subscriptions_data.append(
                        {
                            'id': str(service_hook_subscription.id),
                            'type': service_hook_subscription.service_type_action.service_type.slug
                        }
                    )
        return subscriptions_data, notifications


class SurveyHookWithoutDeletedQuestionsManager(models.Manager):
    def get_queryset(self):
        qs = super(SurveyHookWithoutDeletedQuestionsManager, self).get_queryset()
        qs = qs.filter(
            Q(items__survey_question_id__isnull=True) |
            Q(items__survey_question__is_deleted=False)
        ).distinct()
        return qs


class SurveyHookConditionNode(ConditionNodeBase):
    hook = models.ForeignKey(SurveyHook, related_name='condition_nodes', on_delete=models.CASCADE)

    objects = SurveyHookWithoutDeletedQuestionsManager()
    default = models.Manager()

    def is_true(self, **kwargs):
        if not self.items.exists():
            return True
        else:
            return super(SurveyHookConditionNode, self).is_true(**kwargs)


class SurveyHookConditionSourceFromRequestMixin(object):
    # todo: рефакторинг в нормальные условия
    def _is_true_from_source_request(self, **kwargs):
        # todo: рефакторинг в нормальные условия
        profile_survey_answer = self.get_obj(**kwargs)
        if profile_survey_answer.source_request and isinstance(profile_survey_answer.source_request, dict):
            if self.content_type_attribute.lookup_field.lower() == 'accept-language':
                return self._is_true_for_accept_language(profile_survey_answer)
            elif self.content_type_attribute.lookup_field.lower() == 'parent_origin':
                return self._is_true_for_parent_origin(profile_survey_answer)
        return False

    def _is_true_for_accept_language(self, profile_survey_answer):
        # todo: рефакторинг в нормальные условия
        lang = profile_survey_answer.get_answer_language()
        if lang:
            if self.condition == 'eq':
                return lang == self.value
            else:
                return lang != self.value
        return False

    def _is_true_for_parent_origin(self, profile_survey_answer):
        # todo: рефакторинг в нормальные условия
        parent_origin = getattr(profile_survey_answer, 'source_request', {}).get('parent_origin') or ''
        if isinstance(parent_origin, list):
            if len(parent_origin):
                parent_origin = parent_origin[0]
            else:
                parent_origin = ''
        if not parent_origin.startswith('http://') and not parent_origin.startswith('https://'):
            parent_origin = 'http://%s' % parent_origin
        parsed_tld = urlparse(parent_origin).netloc.split(':')[0].split('.')[-1]
        if self.condition == 'eq':
            return parsed_tld == self.value
        else:
            return parsed_tld != self.value


class SurveyHookConditionWithoutDeletedQuestionsManager(models.Manager):
    def get_queryset(self):
        qs = super(SurveyHookConditionWithoutDeletedQuestionsManager, self).get_queryset()
        qs = qs.filter(
            Q(survey_question_id__isnull=True) |
            Q(survey_question__is_deleted=False)
        )
        return qs


class SurveyHookCondition(SurveyHookConditionSourceFromRequestMixin, ConditionItemBase):
    node = models.ForeignKey(SurveyHook, related_name='items', null=True, blank=True, on_delete=models.CASCADE)  # deprecated
    condition_node = models.ForeignKey(SurveyHookConditionNode, related_name='items', on_delete=models.CASCADE)  # you must add node FK
    survey_question = models.ForeignKey(SurveyQuestion, null=True, blank=True, on_delete=models.CASCADE)

    objects = SurveyHookConditionWithoutDeletedQuestionsManager()
    default = models.Manager()

    def get_obj(self, profile_survey_answer):
        if self.content_type_attribute.attr == 'source_request':
            return profile_survey_answer
        elif self.content_type_attribute.attr in ('groups', 'user.groups'):
            return None
        return self._get_question_answer_or_none(profile_survey_answer)

    def _get_question_answer_or_none(self, profile_survey_answer):
        answer = profile_survey_answer.as_dict()
        question_answer = answer.get(self.survey_question.pk, {})
        return question_answer.get('value')

    def is_true(self, **kwargs):
        # todo: рефакторинг в нормальные условия
        if self.content_type_attribute.attr == 'source_request':
            return self._is_true_from_source_request(**kwargs)
        elif self.content_type_attribute.attr in ('groups', 'user.groups'):
            return True
        return super(SurveyHookCondition, self).is_true(**kwargs)

    class Meta:
        ordering = ('position',)


class SubscriptionDataGenericFieldBase(models.Model):
    position = models.PositiveIntegerField(default=1)
    chars_value = models.CharField(max_length=255, blank=True, null=True)
    question = models.ForeignKey(SurveyQuestion, blank=True, null=True, on_delete=models.CASCADE)

    class Meta:
        abstract = True
        ordering = ('position',)


class IntegrationFileTemplate(ObjectWithVariableMixin, models.Model):
    survey = models.ForeignKey(Survey, related_name='integration_file_templates', on_delete=models.CASCADE)
    name = models.CharField(max_length=255, blank=True)
    template = models.TextField(blank=True)
    type = models.CharField(
        max_length=255,
        choices=ATTACHMENT_TYPE_CHOICES,
        default=ATTACHMENT_TYPE_CHOICES[0][0],
    )
    slug = models.SlugField(blank=True, null=True)

    class Meta:
        unique_together = (
            ('survey', 'slug'),
        )

    def __str__(self):
        # todo: test me forms-666
        return 'IntegrationFileTemplate "{0}.{1}" for survey "{2}"'.format(self.name, self.type, self.survey)

    @classmethod
    def render_content(cls, type, markdown_text):
        # todo: test me forms-666
        if type == 'pdf':
            html_content = common_app_utils.render_markdown(markdown_text, with_new_line_break=True)
            return common_app_utils.render_to_pdf(html_content).read()
        else:
            return force_str(markdown_text.replace('\n', '\r\n')).encode()

    @classmethod
    def get_filename(cls, name, type):
        # todo: test me forms-666
        return '%s.%s' % (name, type)

    def generate_next_slug(self, next_counter=0):
        slug = '%s_%s' % ('template', self.pk)
        if next_counter:
            slug = '%s_%s' % (slug, next_counter)
        return slug


class ServiceSurveyHookSubscription(ObjectWithVariableMixin, TimeStampMixin, models.Model):
    survey_hook = models.ForeignKey(SurveyHook, related_name='subscriptions', on_delete=models.CASCADE)
    service_type_action = models.ForeignKey(ServiceTypeAction, on_delete=models.CASCADE)
    is_synchronous = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)
    # значения: языковой код или from_request
    context_language = models.CharField(max_length=12, default=settings.MODELTRANSLATION_DEFAULT_LANGUAGE)
    follow_result = models.BooleanField(default=False)

    # generic fields
    title = models.CharField(max_length=255, blank=True)
    body = models.TextField(blank=True)
    email = models.CharField(max_length=255, blank=True)
    phone = models.CharField(max_length=255, blank=True)
    questions = models.ManyToManyField(SurveyQuestion, blank=True)
    is_all_questions = models.BooleanField(default=True)
    attachment_templates = models.ManyToManyField(IntegrationFileTemplate, blank=True, related_name='subscriptions')

    # email
    email_to_address = models.CharField(max_length=255, blank=True)
    email_from_address = models.CharField(max_length=255, blank=True)
    email_from_title = models.CharField(max_length=255, blank=True)
    email_spam_check = models.BooleanField(default=settings.IS_BUSINESS_SITE or settings.APP_TYPE == 'forms_ext_admin')

    # http
    http_url = models.CharField(max_length=255, blank=True)
    http_method = models.CharField(
        choices=FORMAT_METHOD_CHOICES,
        max_length=10,
        null=True,
        blank=True,
        default='get'
    )
    http_format_name = models.CharField(
        choices=FORMAT_NAME_CHOICES,
        max_length=10,
        null=True,
        blank=True,
        default='json'
    )
    tvm2_client_id = models.CharField(max_length=25, blank=True, null=True)

    class Meta:
        ordering = ('id',)

    @property
    def survey(self):
        return self.survey_hook.survey

    def __str__(self):
        # todo: test me
        return 'Subscription {0} for action "{1}"'.format(self.id, self.service_type_action)

    def get_slug(self):
        service_slug = self.service_type_action.service_type.slug
        action_slug = self.service_type_action.slug
        return f'{service_slug}/{action_slug}'

    def save(self, *args, **kwargs):
        if not self.email_from_address:
            survey_id = self.survey_hook.survey.pk
            if settings.IS_BUSINESS_SITE:
                self.email_from_address = f'{survey_id}@forms-mailer.yaconnect.com'
            elif settings.APP_TYPE == 'forms_int':
                self.email_from_address = 'devnull@yandex-team.ru'
            else:
                self.email_from_address = f'{survey_id}@forms.yandex.ru'
            if 'update_fields' in kwargs:
                kwargs['update_fields'].append('email_from_address')
        return super().save(*args, **kwargs)

    def notify(self, answer, trigger_slug, trigger_data=None, force=False):
        from events.surveyme_integration.tasks import start_notification
        # todo: test me
        notification = self.create_notification(
            answer=answer,
            trigger_slug=trigger_slug,
            trigger_data=trigger_data,
        )

        if force:
            notification.save()
            start_notification(notification.pk)

        return notification

    def validate_notification_data(self, trigger_slug, trigger_data):
        # todo: test me
        if trigger_slug == 'successful_payment':
            self.validate_data_for_successful_payment_notification(trigger_data)

    def validate_data_for_successful_payment_notification(self, trigger_data):
        # todo: test me
        trigger_data = trigger_data or {}
        if not trigger_data.get('order_id'):
            raise ValueError('Successful payment notification data must have "order_id" value')

    def create_notification(self, answer, trigger_slug, trigger_data=None):
        # todo: test me
        self.validate_notification_data(trigger_slug, trigger_data)
        status = 'pending'
        notification = HookSubscriptionNotification(
            user=answer.user,
            answer=answer,
            survey=answer.survey,
            survey_group_id=answer.survey.group_id,
            subscription=self,
            status=status,
            trigger_slug=trigger_slug,
            trigger_data=trigger_data,
        )
        return notification

    def get_all_notifications(self):
        # todo: test me
        return HookSubscriptionNotification.objects.filter(subscription_id=self.id)

    def render_data_to_http_format(self, data):
        # todo: test me
        return self._render_data_for_format(
            data=data,
            format_name=self.http_format_name
        )

    def _render_data_for_format(self, data, format_name):
        # todo: test me
        if format_name == 'xml':
            response = common_app_utils.render_to_xml(data)
        else:
            response = json.dumps(data)
        return force_str(response)


class SubscriptionHeader(models.Model):
    subscription = models.ForeignKey(ServiceSurveyHookSubscription, related_name='headers', on_delete=models.CASCADE)
    name = models.CharField(max_length=255)
    value = models.CharField(max_length=255)
    add_only_with_value = models.BooleanField(default=False)

    class Meta:
        ordering = ['pk']


class ProxyStorageFileField(models.FileField):
    pass
    # def value_to_string(self, obj):
    #     if isinstance(obj, ProxyStorageModel):
    #         return obj.path
    #     return super().value_to_string(obj)

    # def get_prep_value(self, value):
    #     if isinstance(value, ProxyStorageModel):
    #         value = value.path
    #     return super().get_prep_value(value)


class SubscriptionAttachment(models.Model):
    subscription = models.ForeignKey(ServiceSurveyHookSubscription, related_name='attachments', on_delete=models.CASCADE)
    file = ProxyStorageFileField(storage=ProxyStorage())

    @property
    def file_meta_info(self):
        try:
            meta_info = ProxyStorageModel.objects.get(path=self.file.name)
        except ProxyStorageModel.DoesNotExist:
            return {}
        return {
            'file_size': meta_info.file_size,
            'content_uri': self.get_file_internal_url(),
            'original_name': meta_info.original_name,
            'namespace': meta_info.namespace or settings.MDS_OLD_NAMESPACE,
        }

    def get_file_internal_url(self):
        return ProxyStorage().url(self.file.name, site_type='internal')

    def get_file_size(self):
        return ProxyStorage().open(self.file.name).size


class JSONRPCSubscriptionData(models.Model):
    subscription = models.OneToOneField(ServiceSurveyHookSubscription, related_name='json_rpc', on_delete=models.CASCADE)
    method = models.CharField(max_length=255, blank=True)


class JSONRPCSubscriptionParam(models.Model):
    subscription = models.ForeignKey(JSONRPCSubscriptionData, related_name='params', on_delete=models.CASCADE)
    name = models.CharField(max_length=255)
    value = models.CharField(max_length=255)
    add_only_with_value = models.BooleanField(default=False)

    class Meta:
        ordering = ['pk']


class StartrekSubscriptionData(models.Model):
    subscription = models.OneToOneField(ServiceSurveyHookSubscription, related_name='startrek', on_delete=models.CASCADE)
    queue = models.CharField(max_length=255, blank=True)
    parent = models.CharField(max_length=255, blank=True)
    author = models.CharField(max_length=255, blank=True)
    assignee = models.CharField(max_length=255, blank=True)
    followers = JSONFieldForTinyText(blank=True, null=True)
    tags = JSONFieldForTinyText(blank=True, null=True)
    type = models.PositiveIntegerField(blank=True, null=True)
    components = JSONFieldForTinyText(blank=True, null=True)
    project = models.PositiveIntegerField(blank=True, null=True)
    priority = models.PositiveIntegerField(blank=True, null=True)
    fields = JSONFieldWithFallback(blank=True, null=True)


class WikiSubscriptionData(models.Model):
    subscription = models.OneToOneField(ServiceSurveyHookSubscription, related_name='wiki', on_delete=models.CASCADE)
    supertag = models.CharField(max_length=255, blank=True)
    text = models.TextField(blank=True)

    def clean_fields(self, *args, **kwargs):
        super(WikiSubscriptionData, self).clean_fields(*args, **kwargs)

        if self.supertag:
            self.validate_supertag()
            self.validate_that_page_exists()

    def validate_supertag(self):
        tag = urlparse(self.supertag).path.strip().strip('/')

        client = get_wiki_client()
        if not client.is_valid_supertag(tag):
            raise ValidationError(_('Введите правильный URL вики-страницы или супертег.'))

        self.supertag = tag

    def validate_that_page_exists(self):
        client = get_wiki_client()
        if not client.is_page_exists(self.supertag):
            raise ValidationError(
                _('Указанной в интеграции вики-страницы не существует. Пожалуйста, измените адрес или создайте эту страницу на Вики.')
            )


class IntegrationNotification(models.Model):
    pass


class HookSubscriptionNotification(models.Model):
    notification_id = models.CharField(max_length=32, blank=True, null=True)
    user = models.ForeignKey('accounts.User', on_delete=models.CASCADE, null=True)
    answer = models.ForeignKey('surveyme.ProfileSurveyAnswer', on_delete=models.CASCADE)
    survey = models.ForeignKey('surveyme.Survey', on_delete=models.CASCADE)
    survey_group = models.ForeignKey('surveyme.SurveyGroup', null=True, on_delete=models.SET_NULL)
    subscription = models.ForeignKey(ServiceSurveyHookSubscription, null=True, on_delete=models.SET_NULL)
    status = models.CharField(max_length=32, choices=NOTIFICATION_STATUS_CHOICES, default=NOTIFICATION_STATUS_CHOICES[0][0])
    date_created = models.DateTimeField(auto_now_add=True, db_index=True)
    date_updated = models.DateTimeField(auto_now_add=True, null=True, blank=True)
    date_finished = models.DateTimeField(null=True)
    date_next_processing = models.DateTimeField(null=True)
    date_retry = models.DateTimeField(null=True)
    retries = models.IntegerField(null=True)
    max_retries = models.IntegerField(null=True)
    trigger_slug = models.CharField(max_length=50)
    trigger_data = JSONFieldForSimpleText(null=True)
    response = JSONFieldForSimpleText(null=True)
    context = JSONFieldForSimpleText(null=True)
    error = JSONFieldForSimpleText(null=True)
    is_visible = models.BooleanField(default=True, db_index=True)
    celery_task_id = models.CharField(max_length=50, null=True, blank=True)

    objects = HookSubscriptionNotificationManager()

    def update_fields(self, commit=True, **fields):
        for (key, value) in fields.items():
            setattr(self, key, value)
        if commit:
            self.save(update_fields=fields.keys())
        return self

    def increment_counter(self):
        counter_qs = HookSubscriptionNotificationCounter.objects.filter(subscription=self.subscription)
        counter_qs.update(
            errors_count=F('errors_count') + 1,
            date_updated=timezone.now(),
        )

    def decrement_counter(self):
        counter_qs = HookSubscriptionNotificationCounter.objects.filter(subscription=self.subscription)
        counter_qs.update(
            errors_count=F('errors_count') - 1,
            date_updated=timezone.now(),
        )

    def make_visible(self, is_visible):
        if self.is_visible == is_visible:
            return

        self.is_visible = is_visible
        self.save(update_fields=['is_visible'])

        if self.status == 'error':
            if self.is_visible:
                self.increment_counter()
            else:
                self.decrement_counter()

    @cached_property
    def is_success(self):
        return self.status == 'success'

    @cached_property
    def is_finished(self):
        return self.status != 'pending'

    @cached_property
    def integration_type(self):
        return self.subscription.service_type_action.service_type.slug


class SurveyVariable(models.Model):
    variable_id = models.CharField(max_length=32, primary_key=True)
    hook_subscription = models.ForeignKey(
        ServiceSurveyHookSubscription, null=True,
        blank=True, on_delete=models.CASCADE,
    )
    integration_file_template = models.ForeignKey(
        IntegrationFileTemplate, null=True,
        blank=True, on_delete=models.CASCADE,
    )
    var = models.CharField(max_length=50)
    format_name = models.CharField(max_length=50, null=True, blank=True)
    arguments = JSONFieldForSimpleText(null=True)
    filters = JSONFieldForSimpleText(null=True)

    def save(self, *args, **kwargs):
        if self.hook_subscription is None and self.integration_file_template is None:
            raise ValidationError(
                'Either hook_subscription or '
                'integration_file_template must be provided'
            )
        super(SurveyVariable, self).save(*args, **kwargs)


class HookSubscriptionNotificationCounter(models.Model):
    subscription = models.OneToOneField(ServiceSurveyHookSubscription, primary_key=True, on_delete=models.CASCADE)
    survey = models.ForeignKey(Survey, on_delete=models.CASCADE)
    errors_count = models.PositiveIntegerField(default=0)
    date_updated = models.DateTimeField(default=datetime.datetime(1970, 1, 1, tzinfo=timezone.utc))
