# -*- coding: utf-8 -*-
import base64
import itertools
import logging
import mimetypes
import os
import requests

from django.conf import settings
from django.utils.translation import ugettext as _
from email.utils import getaddresses
from requests import HTTPError, ConnectionError
from requests.structures import CaseInsensitiveDict

from events.common_app.utils import get_user_ip_address
from events.common_storages.proxy_storages import ProxyStorage
from events.common_storages.storage import ReadError
from events.surveyme.models import Survey, ProfileSurveyAnswer
from events.surveyme_integration.services.base.action_processors import ActionProcessorBase
from events.surveyme_integration.services.email.check_form import CheckFormV2
from events.surveyme_integration.exceptions import (
    EmailRecipientError,
    EmailSpamError,
    EmailSpamCheckError,
    EmailInternalError,
    EmailValidationError,
    EmailCompaignError,
    EmailMaxSizeExceededError,
)
from events.surveyme_integration.utils import (
    encode_long_email,
    encode_string,
    get_message_id,
    parse_response,
)
from events.yauth_contrib.auth import TvmAuth


logger = logging.getLogger(__name__)


class EmailActionProcessor(ActionProcessorBase):
    default_mime_type = 'application/octet-stream'
    text_mime_type = 'text/plain;charset=utf-8'
    valid_system_headers = {
        'message-id',
        'reply-to',
        'x-form-id',
        'x-so-form-spam',
    }
    invalid_system_headers = {
        'from',
        'to',
    }

    def do_action(self):
        sender_data = {
            'async': True,
            'args': {
                'subject': self.data['subject'],
                'body': self.data['body'],
            },
            'to': self.get_to_email(),
            'from_email': self.get_from_email(),
            'from_name': encode_string(self.get_from_name()),
            'headers': self.get_headers(),
            'attachments': self.get_attachments(),
            'ignore_empty_email': False,
            'has_ugc': True,
        }
        sender_url = settings.SENDER_URL.format(
            account=settings.SENDER_ACCOUNT,
            campaign=settings.APP_TYPE,
        )
        try:
            response = requests.post(
                sender_url,
                json=sender_data,
                auth=TvmAuth(settings.SENDER_TVM2_CLIENT),
                headers={
                    'X-Sender-Real-User-IP': get_user_ip_address(),
                },
                verify=settings.YANDEX_ROOT_CERTIFICATE,
                timeout=settings.DEFAULT_TIMEOUT,
            )
            response.raise_for_status()
        except Exception as e:
            self.handle_error(e)
        return {
            'status': 'success',
            'response': parse_response(response),
        }

    def get_error_message(self, response_json):
        result = response_json.get('result')
        errors = []
        error = result.get('error')
        if error:
            to_email = error.pop('to', None)
            if to_email:
                errors.append(_('Ошибка в адресе получателя.'))

            from_email = error.pop('from_email', None)
            if from_email:
                errors.append(_('Ошибка в адресе отправителя.'))

            if error:
                errors.extend([
                    _('Ошибка в поле %(field_name)s.') % {'field_name': field_name}
                    for field_name in error.keys()
                ])
        if not errors:
            errors.append(_('Ошибка валидации запроса к сервису отправки почты.'))
        return ' '.join(errors)

    def handle_error(self, e):
        if isinstance(e, HTTPError):
            if e.response.status_code >= 500:
                raise EmailInternalError(cause=e) from e
            elif e.response.status_code == 404:
                raise EmailCompaignError(cause=e) from e
            elif e.response.status_code >= 400:
                try:
                    response_json = e.response.json()
                except Exception:
                    response_json = {}
                msg = self.get_error_message(response_json)
                raise EmailValidationError(msg, e) from e
        elif isinstance(e, ConnectionError):
            raise EmailInternalError(cause=e) from e
        raise

    def _get_punnycode_address(self, address):
        parts = address.split('@')
        if len(parts) != 2:
            return address
        return '@'.join([parts[0], parts[1].encode('idna').decode()])

    def get_to_email(self):
        emails = []
        for name, email in getaddresses([self.data['to_address']]):
            if email:
                item = {
                    'email': self._get_punnycode_address(email),
                }
                if name:
                    item['name'] = name
                emails.append(item)
        if not emails:
            raise EmailRecipientError
        return emails

    def get_from_name(self):
        from events.surveyme.utils import check_email_title
        from_name = self.data.get('from_title', '')
        if not check_email_title(from_name):
            from_name = ''
        if settings.IS_BUSINESS_SITE:
            if not from_name:
                from_name = _('Яндекс.Формы')
        return from_name

    def _get_default_email(self):
        survey_id = self.data['survey_id']
        if settings.IS_BUSINESS_SITE:
            return f'{survey_id}@forms-mailer.yaconnect.com'
        elif settings.APP_TYPE == 'forms_int':
            return 'devnull@yandex-team.ru'
        return f'{survey_id}@forms.yandex.ru'

    def get_from_email(self):
        if settings.IS_BUSINESS_SITE:
            from_address = self._get_default_email()
        else:
            from_address = self.data.get('from_address', '')
            if '@' not in from_address:
                from_address = self._get_default_email()
        return from_address

    def get_headers(self):
        raw_headers = self.data.get('headers') or {}
        headers = CaseInsensitiveDict({
            name: value
            for name, value in raw_headers.items()
            if self._is_valid_header_name(name)
        })
        if 'Reply-To' in headers:
            headers['Reply-To'] = encode_long_email(headers['Reply-To'])
        if 'MESSAGE-ID' not in headers:
            headers['MESSAGE-ID'] = get_message_id(self.data)
        if 'X-Form-ID' not in headers:
            headers['X-Form-ID'] = str(self.data['survey_id'])
        self._update_cf_receipt(headers)
        return dict(headers.items())

    def _is_valid_header_name(self, header_name):
        header_name = header_name.lower()
        if settings.IS_BUSINESS_SITE:
            return header_name in self.valid_system_headers
        else:
            return header_name not in self.invalid_system_headers

    def _check_form(self):
        from events.surveyme_integration.models import ServiceSurveyHookSubscription
        answer = (
            ProfileSurveyAnswer.objects.using(settings.DATABASE_ROLOCAL)
            .get(pk=self.data['answer_id'])
        )
        subscription = (
            ServiceSurveyHookSubscription.objects.using(settings.DATABASE_ROLOCAL)
            .prefetch_related('attachments', 'headers')
            .get(pk=self.data['subscription_id'])
        )

        check_form_v2 = CheckFormV2(answer, subscription)
        result = check_form_v2.check()
        if not result:
            raise EmailSpamCheckError

        if settings.IS_BUSINESS_SITE:
            self._update_spam_status(result.get('spam'), result.get('ban'))
        return result

    def _update_cf_receipt(self, headers):
        if settings.IS_BUSINESS_SITE or self.data.get('spam_check'):
            # в случае подозрения на спам всегда выбрасываем исключение
            check_result = self._check_form()
            if check_result.get('ban') or check_result.get('spam'):
                raise EmailSpamError
            else:
                headers['X-Yandex-CF-Receipt'] = check_result.get('receipt', '')

    def _update_spam_status(self, is_spam_detected, is_ban_detected):
        fields = {}
        if is_spam_detected:
            Survey.set_spam_detected(self.data['survey_id'], True)
        if is_ban_detected:
            fields['is_ban_detected'] = True
            if settings.SWITCH_ON_BAN:
                fields['captcha_type'] = 'ocr'
                fields['is_published_external'] = False
        if fields:
            Survey.objects.filter(pk=self.data['survey_id']).update(**fields)

    def get_attachments(self):
        total_size = 0
        attachments = []
        for file_name, mime_type, content in itertools.chain(
            self._add_attachments(self.data['attachments']),
            self._add_attachments(self.data['static_attachments']),
            self._add_attachment_templates(self.data['attachment_templates']),
        ):
            total_size += len(content)
            if total_size >= settings.MAX_SENDER_ATTACHMENT_SIZE:
                raise EmailMaxSizeExceededError
            attachments.append({
                'filename': file_name,
                'mime_type': mime_type,
                'data': base64.b64encode(content).decode(),
            })
        return attachments

    def _get_mime_type(self, file_name):
        mime_type = None
        ext = os.path.splitext(file_name)[1]
        if ext == '.txt':
            mime_type = self.text_mime_type
        elif ext == '.eml':
            mime_type = self.default_mime_type
        else:
            mime_type = mimetypes.types_map.get(ext, self.default_mime_type)
        return mime_type

    def _add_attachments(self, attachments):
        if attachments:
            for attachment in attachments:
                file_name = attachment['filename']
                mime_type = self._get_mime_type(file_name)
                try:
                    namespace = attachment['namespace']
                    body = ProxyStorage(namespace).open(attachment['path']).read()
                except ReadError as e:
                    logger.warn(e)
                else:
                    yield file_name, mime_type, body

    def _add_attachment_templates(self, attachments):
        from events.surveyme_integration.models import IntegrationFileTemplate
        if attachments:
            for attachment in attachments:
                file_name = attachment['name']
                mime_type = self._get_mime_type(file_name)
                content = IntegrationFileTemplate.render_content(attachment['type'], attachment['content'])
                yield file_name, mime_type, content
