# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from email.utils import getaddresses
from io import StringIO
from itertools import chain
from uuid import uuid1

from events.common_app.utils import (
    get_user_ip_address,
    lazy_re_compile,
    requests_session,
)
from events.common_storages.models import ProxyStorageModel
from events.surveyme_integration.utils import (
    get_variables_keys,
    render_variables,
)

logger = logging.getLogger(__name__)


class CheckFormV2(object):
    # разбираем паттерн `<spam>1</spam><ban>0</ban>`
    response_re = lazy_re_compile(r'\<(\w+)\>(\w+)<\/\1>')

    def __init__(self, answer, subscription):
        self.answer = answer
        self.subscription = subscription

    def _get_capture_type(self, survey):
        if survey.captcha_display_mode == 'auto':
            return 'auto'
        if survey.captcha_display_mode == 'always':
            return settings.CAPTCHA_TYPE

    def _get_form_type(self, survey):
        if settings.IS_BUSINESS_SITE:
            if survey.org:
                return 'b2b'
            else:
                return 'user'
        else:
            return 'yandex'

    def _get_form_recipients(self):
        return [
            address
            for _, address in getaddresses([self.subscription.email_to_address])
        ]

    def _render(self, value):
        return render_variables(
            value=value,
            object_with_variables=self.subscription,
            answer=self.answer,
            trigger_data=None,
            notification_unique_id=None,
            force_render=True,
        )

    def _get_from_template(self):
        from_title = self.subscription.email_from_title
        from_address = self.subscription.email_from_address
        if not from_title:
            return from_address
        return '{from_title} <{from_address}>'.format(
            from_title=from_title,
            from_address=from_address,
        )

    def _get_file_meta(self, file_path):
        try:
            return ProxyStorageModel.objects.get(path=file_path)
        except ProxyStorageModel.DoesNotExist:
            pass

    def _get_static_attachments(self):
        for attachment in self.subscription.attachments.all():
            file_meta = self._get_file_meta(attachment.file.name)
            if file_meta:
                yield {
                    'id': file_meta.path,
                    'sha256': file_meta.sha256,
                    'source': 'form',
                }

    def _get_attachments(self):
        answer_data = self.answer.as_dict()
        for question_answer in answer_data.values():
            slug = question_answer.get('question', {}).get('answer_type', {}).get('slug')
            if slug == 'answer_files':
                for item in question_answer.get('value') or []:
                    file_meta = self._get_file_meta(item.get('path'))
                    if file_meta:
                        yield {
                            'id': file_meta.path,
                            'sha256': file_meta.sha256,
                            'source': 'ugc',
                        }

    def _get_variables(self):
        full_template = StringIO()
        full_template.write(str(self.subscription.title))
        full_template.write(str(self.subscription.body))
        full_template.write(str(self.subscription.email_to_address))
        full_template.write(str(self.subscription.email_from_address))
        full_template.write(str(self.subscription.email_from_title))
        full_template.write(str(self.subscription.email_from_title))
        for headers in self.subscription.headers.all():
            full_template.write(str(headers.name))
            full_template.write(str(headers.value))
        return get_variables_keys(full_template.getvalue())

    def _get_form_fields(self):
        from events.surveyme_integration.models import SurveyVariable
        variables_qs = (
            SurveyVariable.objects.filter(
                hook_subscription=self.subscription,
                variable_id__in=self._get_variables()
            ))
        return {
            variable.variable_id: {
                'value': self._render('{%s}' % variable.variable_id),
                'type': 'string',
                'filled_by': 'user' if 'form.question' in variable.var else 'automatic',
            }
            for variable in variables_qs
        }

    def _get_request_id(self, source_request):
        return source_request.get('request_id') or uuid1().hex

    def _get_client_ip(self, source_request):
        return source_request.get('ip') or get_user_ip_address()

    def _get_form_redir(self, survey):
        extra = survey.extra or {}
        redirect = extra.get('redirect') or {}
        if redirect.get('enabled'):
            return redirect.get('url')

    def _get_form_body(self, survey):
        texts = {
            text.slug: text.get_value()
            for text in survey.texts.all()
        }
        return '%s\n%s' % (
            texts.get('successful_submission_title'),
            texts.get('successful_submission'),
        )

    def check(self):
        source_request = self.answer.source_request or {}
        form_id = str(self.answer.survey.pk)
        data = {
            'attachments': list(chain(self._get_static_attachments(), self._get_attachments())),
            'body': self._render(self.subscription.body),
            'body_template': self.subscription.body,
            'capture_type': self._get_capture_type(self.answer.survey),
            'client_email': self.answer.user.email,
            'client_ip': self._get_client_ip(source_request),
            'client_uid': self.answer.user.uid,
            'form_author': self.answer.survey.user.uid,
            'form_fields': self._get_form_fields(),
            'form_id': form_id,
            'form_realpath': source_request.get('url'),
            'form_recipients': self._get_form_recipients(),
            'form_type': self._get_form_type(self.answer.survey),
            'form_body': self._get_form_body(self.answer.survey),
            'form_redir': self._get_form_redir(self.answer.survey),
            'from': self._render(self._get_from_template()),
            'from_template': self._get_from_template(),
            'subject': self._render(self.subscription.title),
            'subject_template': self.subscription.title,
            'yandexuid': source_request.get('cookies', {}).get('yandexuid'),
        }
        params = {
            'form_id': form_id,
            'request_id': self._get_request_id(source_request),
            'service': 'FEEDBACK',
            'format': 'json',
        }
        url = settings.CHECKFORM_URL
        response = requests_session.post(url, params=params, json=data, timeout=settings.DEFAULT_TIMEOUT)
        if response.status_code != 200:
            logger.warning('Check-form answer=%s subscription=%s error status %s',
                           self.answer.pk, self.subscription.pk, response.status_code)
            return None
        logging.info('Check-form answer=%s subscription=%s response %s',
                     self.answer.pk, self.subscription.pk, response.text)
        return self._get_check_result(response.json())

    def _get_check_result(self, data):
        from events.surveyme.logic.query import is_true
        check_result = data.get('check')
        ban = check_result.get('ban')
        if isinstance(ban, str):
            check_result['ban'] = is_true(ban)
        return check_result
