# -*- coding: utf-8 -*-
from email.utils import parseaddr
from django.conf import settings
from django.db.models import QuerySet
from django.utils.translation import ugettext_lazy as _
from requests.packages.urllib3.fields import guess_content_type
from rest_framework import serializers

from events.common_storages.models import ProxyStorageModel
from events.surveyme_integration.services.base.context_processors.base import (
    ContextProcessorBase,
)
from events.surveyme_integration.services.base.context_processors.fields import (
    RenderedCharField,
    SafeRenderedFieldMixin,
    SafeRenderedCharField,
    AttachmentFromQuestionSerializer,
    AttachmentBaseSerializer,
    AttachmentFileObjectWrapper,
    IntegrationFileTemplateField,
    SafeBooleanField,
    SafeCharField,
)
from events.surveyme_integration.utils import (
    get_variables_keys,
    get_message_id,
)


class EmailHeadersSerializer(SafeRenderedFieldMixin, serializers.Serializer):
    def to_representation(self, value):
        service_context_instance = self.parent.instance
        headers = {
            'MESSAGE-ID': get_message_id(service_context_instance),
            'X-Form-ID': str(service_context_instance.answer.survey_id),
        }
        for header in value.all():
            native_name = super(EmailHeadersSerializer, self).to_representation(header.name)
            native_value = super(EmailHeadersSerializer, self).to_representation(header.value)
            if not self.is_need_to_skip(header, native_value):
                headers[native_name] = native_value
        return headers


class EmailStaticAttachmentSerializer(AttachmentBaseSerializer):
    def to_representation(self, value):
        if isinstance(value, QuerySet):
            return self.get_list(value)
        return self.get_item(value)

    def get_list(self, value):
        return [self.get_item(i) for i in value.all()]

    def get_item(self, value):
        file_meta_data = ProxyStorageModel.objects.get(path=value.file.name)
        storage = file_meta_data.get_original_storage()
        return super(EmailStaticAttachmentSerializer, self).to_representation(AttachmentFileObjectWrapper(
            file_object=value.file,
            path=value.file.name,
            filename=file_meta_data.original_name,
            frontend_url=storage.url(value.file.name, site_type='internal'),
            content_type=guess_content_type(file_meta_data.original_name),
            question=None,
            namespace=file_meta_data.namespace or settings.MDS_OLD_NAMESPACE,
        ))


class FromAddressRenderedCharField(RenderedCharField):
    def to_representation(self, value):
        result = super(FromAddressRenderedCharField, self).to_representation(value)
        _, addr = parseaddr(result)
        if '@' not in addr:
            result = result.replace('"', '')
        return result


class EmailBodyField(RenderedCharField):
    disclaimer = _('Это письмо содержит ответы на опрос, созданный пользователем Яндекс.Форм. Яндекс не несёт ответственности за содержание письма.')

    def check_if_contains_user_data(self, value):
        from events.surveyme_integration.models import SurveyVariable
        keys = get_variables_keys(value)
        return SurveyVariable.objects.filter(variable_id__in=keys, var__contains='question').exists()

    def add_disclaimer(self, value):
        return '%s\n\n%s' % (value, self.disclaimer)

    def to_representation(self, value):
        result = super().to_representation(value)
        if settings.IS_BUSINESS_SITE:
            result = self.add_disclaimer(result)
        elif settings.APP_TYPE != 'forms_int':
            if self.check_if_contains_user_data(value):
                result = self.add_disclaimer(result)
        return result


class EmailContextProcessor(ContextProcessorBase):
    subject = SafeRenderedCharField(source='subscription.title')
    body = EmailBodyField(source='subscription.body')
    to_address = RenderedCharField(source='subscription.email_to_address')
    from_address = FromAddressRenderedCharField(source='subscription.email_from_address')
    from_title = RenderedCharField(source='subscription.email_from_title')
    spam_check = SafeBooleanField(source='subscription.email_spam_check', default=False)
    headers = EmailHeadersSerializer(source='subscription.headers')
    attachments = AttachmentFromQuestionSerializer(
        source='answer.survey.surveyquestion_set.filter_by_answer_type_with_files',
        selective_source='subscription.questions.filter_by_answer_type_with_files',
    )
    static_attachments = EmailStaticAttachmentSerializer(source='subscription.attachments', many=True)
    attachment_templates = IntegrationFileTemplateField(source='subscription.attachment_templates', many=True)
    login = SafeCharField(source='answer.survey.user.email', default='')

    class Meta(ContextProcessorBase.Meta):
        fields = ContextProcessorBase.Meta.fields + [
            'subject',
            'body',
            'to_address',
            'from_address',
            'spam_check',
            'headers',
            'attachments',
            'static_attachments',
            'attachment_templates',
            'login',
        ]
