# -*- coding: utf-8 -*-
import json
import logging
from collections import OrderedDict
from urllib.parse import urlparse
from requests import RequestException, HTTPError
from django.conf import settings

from events.common_storages.proxy_storages import ProxyStorage
from events.common_storages.storage import ReadError
from events.common_app.zora import get_session
from events.surveyme_integration.services.base.action_processors import ActionProcessorBase
from events.surveyme_integration.exceptions import RemoteServiceError, IntegrationError
from events.surveyme_integration.utils import parse_response
from events.yauth_contrib.auth import TvmAuth, OAuth


logger = logging.getLogger(__name__)


def prepare_headers(headers):
    encode_string = lambda s: s.encode('unicode_escape').decode().strip()
    for header, value in (headers or {}).items():
        encoded_header = encode_string(header)
        if encoded_header:
            yield encoded_header, encode_string(value)


class HTTPBaseActionProcessor(ActionProcessorBase):
    certificate = None
    verify_certificate = True
    force_multipart = True
    convert_body_to_json = False  # todo: test me
    max_content_size = 500

    def do_action(self):
        response = None
        try:
            response = self.make_request()
            response.raise_for_status()
        except HTTPError as e:
            status_code = e.response.status_code
            if status_code == 409:
                logger.info('Got 409 in response, marking integration as successful')
            elif status_code != 404 and 400 <= status_code < 500:
                raise IntegrationError(e)
            else:
                raise RemoteServiceError(str(e), e)
        except RequestException as e:
            raise RemoteServiceError(str(e), e)
        except Exception as e:
            raise IntegrationError(e)
        return self.get_action_result_from_response(response)

    def get_action_result_from_response(self, response):
        return {
            'status': 'success',
            'response': parse_response(response),
        }

    def make_request(self):
        raise NotImplementedError()

    def get_auth(self):
        if self.data.get('tvm2_client_id'):
            return TvmAuth(self.data['tvm2_client_id'])
        if self.is_good_for_oauth(self.data['url']):
            return OAuth(settings.HTTP_INTEGRATION_OAUTH_TOKEN)

    def is_good_for_oauth(self, url):
        domain_name = urlparse(url).netloc
        return domain_name.endswith('.yandex-team.ru') or domain_name.endswith('.yandex.net')


class PostHTTPActionProcessor(HTTPBaseActionProcessor):
    method = 'post'

    def make_request(self):
        params = self.get_request_context()
        return self._make_request(self.method, **params)

    def _make_request(self, method, **params):
        request_id = self.data.get('request_id')
        session = get_session(self.data['url'], request_id)
        request_method = getattr(session, method, None)
        return request_method(**params)

    def get_request_context(self):
        context = self.get_base_context()
        context.update(self.get_additional_context())
        return context

    def get_base_context(self):
        headers = self.data['headers']
        return {
            'url': self.data['url'],
            'headers': dict(prepare_headers(headers)),
            'auth': self.get_auth(),
            'cert': self.certificate,
        }

    def get_additional_context(self):
        return {
            'data': self._prepare_body_data(self.data['body_data']),
            'files': self._prepare_files_for_request(
                self.data.get('attachments'),
                start=self.data.get('body_data_items_count', 0) + 1
            ),
        }

    def _prepare_body_data(self, data):
        if self.convert_body_to_json:
            return json.dumps(data)
        return data

    def _prepare_files_for_request(self, attachments, start=1):
        files = OrderedDict()
        if attachments:
            for attachment in attachments:
                try:
                    namespace = attachment['namespace']
                    body = ProxyStorage(namespace).open(attachment['path']).read()
                except ReadError as e:
                    logger.warn(e)
                else:
                    file_content = (attachment['filename'], body)
                    if 'content_type' in attachment:
                        file_content += (attachment['content_type'], )
                    if 'headers' in attachment:
                        file_content += (dict(prepare_headers(attachment['headers'])), )
                    files[f'field_{start}'] = file_content
                    start += 1
        if self.force_multipart:
            files['noop'] = ('', 'content')
        return files


class PutHTTPActionProcessor(PostHTTPActionProcessor):
    method = 'put'


class ArbitraryHTTPActionProcessor(PostHTTPActionProcessor):
    def make_request(self):
        self.method = self.data['method']
        return super(ArbitraryHTTPActionProcessor, self).make_request()

    def get_additional_context(self):
        context = {}
        data = self.data.get('body_data')
        if data:
            prepared_data, data_type = self._prepare_body_data(self.data['body_data'])
            context[data_type] = prepared_data
        return context

    def _prepare_body_data(self, data):
        content_type = self.data['headers'].get('Content-Type')
        if self._is_json_type(content_type):
            return json.loads(data), 'json'
        return data.encode(), 'data'

    def _is_json_type(self, header):
        return header == 'application/json'
