# -*- coding: utf-8 -*-
import base64

from django.db.models import QuerySet
from django.utils.encoding import force_str
from django.utils import translation

from rest_framework import serializers

from events.surveyme_integration.services.base.context_processors.base import ContextProcessorBase
from events.surveyme_integration.services.base.context_processors.fields import (
    RenderedCharField,
    RenderedFieldMixin,
    AttachmentFromQuestionSerializer,
    SelectiveSourceMixin,
    SafeCharField,
)
from events.surveyme.answer_value import get_value_for_question


class HTTPHeadersSerializer(RenderedFieldMixin, serializers.Serializer):
    def to_representation(self, value):
        service_context_instance = self.parent.instance
        headers = {
            'X-DELIVERY-ID': service_context_instance.notification_unique_id,
            'X-FORM-ID': force_str(service_context_instance.answer.survey_id),
            'X-FORM-ANSWER-ID': force_str(service_context_instance.answer.id),
        }
        for header in value.all():
            native_name = super(HTTPHeadersSerializer, self).to_representation(header.name)
            native_value = super(HTTPHeadersSerializer, self).to_representation(header.value)
            if not self.is_need_to_skip(header, native_value):
                headers[native_name] = native_value
        return headers


class HTTPAttachmentFromQuestionSerializer(AttachmentFromQuestionSerializer):
    def to_representation(self, value):
        service_context_instance = self.parent.instance
        attachments = super(HTTPAttachmentFromQuestionSerializer, self).to_representation(value)
        for attach in attachments:
            meta_info_header = service_context_instance.subscription.render_data_to_http_format({
                'question': attach['question'],
                'frontend_url': attach['frontend_url']
            })
            attach['headers'] = {
                'X-META-INFO': meta_info_header,
                'X-META-INFO-BASE64': force_str(base64.b64encode(meta_info_header.encode('utf-8'))),
            }
        return attachments


class HTTPBodyDataSerializer(SelectiveSourceMixin, serializers.Serializer):
    def to_representation(self, value):
        if isinstance(value, QuerySet):
            return self.get_list(value)
        return self.get_item(value)

    def get_list(self, value):
        response = {}
        for i, body in enumerate(value.all(), start=1):
            response.update(self.get_item(body, i))
        return response

    def get_item(self, value, i=None):
        from events.surveyme.api_admin.v2.serializers import AnswerTypeSerializer
        service_context_instance = self.parent.instance
        answer = service_context_instance.answer
        subscription = service_context_instance.subscription
        lang = translation.get_language()
        unfold_group = False
        if value.group_id:
            unfold_group = True
        text_value = get_value_for_question(answer, value.pk, unfold_group=unfold_group)
        item_body = {
            'value': text_value,
            'question': {
                'id': value.pk,
                'slug': value.param_slug,
                'label': {
                    lang: value.label
                },
                'group_id': value.group_id,
                'type': AnswerTypeSerializer(value.answer_type).data
            }
        }
        value.add_additional_post_data(text_value=text_value, item_body=item_body)
        item_data = {
            'field_%s' % i or value.pk: subscription.render_data_to_http_format(item_body)
        }
        return item_data


class HTTPContextProcessor(ContextProcessorBase):
    url = RenderedCharField(source='subscription.http_url')
    headers = HTTPHeadersSerializer(source='subscription.headers')
    attachments = HTTPAttachmentFromQuestionSerializer(
        source='answer.survey.surveyquestion_set.filter_by_answer_type_with_files',
        selective_source='subscription.questions.filter_by_answer_type_with_files',
    )
    body_data = HTTPBodyDataSerializer(
        source='answer.survey.surveyquestion_set.exclude_with_file_answer_type',
        selective_source='subscription.questions.exclude_with_file_answer_type',
    )  # todo: test me
    tvm2_client_id = SafeCharField(source='subscription.tvm2_client_id', allow_blank=True)

    class Meta(ContextProcessorBase.Meta):
        fields = ContextProcessorBase.Meta.fields + [
            'url',
            'headers',
            'attachments',
            'body_data',  # todo: test me
            'tvm2_client_id',
        ]

    def to_representation(self, value):
        # todo: test me
        response = super(HTTPContextProcessor, self).to_representation(value)
        response['body_data_items_count'] = len(response['body_data'])
        return response


class ArbitraryHTTPContextProcessor(ContextProcessorBase):
    url = RenderedCharField(source='subscription.http_url')
    headers = HTTPHeadersSerializer(source='subscription.headers')
    method = RenderedCharField(source='subscription.http_method')
    body_data = RenderedCharField(source='subscription.body')
    tvm2_client_id = SafeCharField(source='subscription.tvm2_client_id', allow_blank=True)

    class Meta(ContextProcessorBase.Meta):
        fields = ContextProcessorBase.Meta.fields + [
            'url',
            'headers',
            'method',
            'body_data',
            'tvm2_client_id',
        ]
