# -*- coding: utf-8 -*-
from rest_framework import serializers
from events.surveyme_integration.services.base.context_processors.base import ContextProcessorBase
from events.surveyme_integration.services.base.context_processors.fields import (
    RenderedCharField,
    RenderedFieldMixin,
    SafeCharField,
)
from events.surveyme_integration.services.http.context_processors import HTTPHeadersSerializer


class JSONRPCHeadersSerializer(HTTPHeadersSerializer):
    def to_representation(self, value):
        headers = super().to_representation(value)
        if 'content-type' not in set(map(str.lower, headers.keys())):
            headers['content-type'] = 'application/json'
        return headers


class JSONRPCBodyDataSerializer(RenderedFieldMixin, serializers.Serializer):
    def to_representation(self, value):
        service_context_instance = self.parent.instance
        params = {}
        for param in value.all():
            native_name = super(JSONRPCBodyDataSerializer, self).to_representation(param.name)
            native_value = super(JSONRPCBodyDataSerializer, self).to_representation(param.value)
            if not self.is_need_to_skip(param, native_value):
                params[native_name] = native_value
        return {
            'jsonrpc': '2.0',
            'method': service_context_instance.subscription.json_rpc.method,
            'params': params,
            'id': service_context_instance.notification_unique_id,
        }


class JSONRPCContextProcessor(ContextProcessorBase):
    url = RenderedCharField(source='subscription.http_url')
    headers = JSONRPCHeadersSerializer(source='subscription.headers')  # todo: test me
    body_data = JSONRPCBodyDataSerializer(source='subscription.json_rpc.params')  # todo: test me
    tvm2_client_id = SafeCharField(source='subscription.tvm2_client_id', allow_blank=True)

    class Meta(ContextProcessorBase.Meta):
        fields = ContextProcessorBase.Meta.fields + [
            'url',
            'headers',
            'body_data',
            'tvm2_client_id',
        ]
