# coding: utf-8
import ylock
from django.conf import settings
from requests.exceptions import HTTPError

from events.common_app.wiki import get_wiki_client
from events.surveyme.models import Survey
from events.surveyme_integration.exceptions import WikiSupertagError, WikiPermissionError
from events.surveyme_integration.services.base.action_processors import ActionProcessorBase


class WikiActionProcessor(ActionProcessorBase):
    LOCK_TIMEOUT = 30  # полминуты

    def _append_to_page_for_survey(self, survey, supertag, text):
        dir_id = survey.org.dir_id if survey.org else None
        client = get_wiki_client(dir_id)
        try:
            client.append_to_wiki_page(supertag, text)
        except HTTPError as e:
            if e.response.status_code == 404:
                raise WikiSupertagError(supertag, e)
            elif e.response.status_code == 403:
                username = survey.user.get_yandex_username()
                raise WikiPermissionError(username, e)
            raise

    def do_action(self):
        survey = (
            Survey.objects.select_related('org', 'user')
            .get(pk=self.data['survey_id'])
        )
        supertag = self.data['supertag']
        lock_name = supertag.split('#')[0]
        text = self.data['text']

        dir_id = survey.org.dir_id if survey.org else None
        if dir_id:
            lock_name += ':{}'.format(dir_id)

        manager = ylock.create_manager(**settings.YLOCK)
        with manager.lock(
            lock_name,
            block=True,
            timeout=self.LOCK_TIMEOUT,
            block_timeout=5,
            common_lock_name=True,
        ) as acquired:
            if acquired:
                self._append_to_page_for_survey(survey, supertag, text)
                return {
                    'status': 'success',
                    'response': {}
                }
            else:
                return {
                    'status': 'error',
                    'error': 'could not acquire lock for {}'.format(supertag)
                }
