# -*- coding: utf-8 -*-
import logging
import sys
import traceback
from collections import defaultdict

from datetime import timedelta
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.core.cache import caches
from django.db import connection
from django.db.models import Prefetch, Q
from django.utils import timezone
from django.utils.translation import override as override_lang

from requests.exceptions import HTTPError
from celery.exceptions import RetryTaskError

from events.celery_app import app
from events.common_app.redis import redis
from events.common_app.sender.client import send_email
from events.common_app.utils import get_backoffice_url_for_obj, get_backoffice_url
from events.followme.models import ContentFollower
from events.surveyme.utils import send_email_with_retry
from events.surveyme_integration.statuses import statuses_by_name
from events.surveyme_integration.exceptions import RetriableError
from events.surveyme.models import ProfileSurveyAnswer, Survey
from events.surveyme_integration.models import (
    SurveyHook,
    SurveyHookCondition,
    ServiceSurveyHookSubscription,
    HookSubscriptionNotification,
)
from events.surveyme_integration.utils import parse_response, round_datetime_five_minutes_down
from library.python.redis_lock import RedisLock


logger = logging.getLogger(__name__)


class RedisLockException(Exception):
    pass


def get_classname(obj):
    if obj is not None:
        return '.'.join([obj.__class__.__module__, obj.__class__.__name__])


def get_response_data(exc):
    response = None
    if isinstance(exc, HTTPError):
        response = exc.response
    elif hasattr(exc, 'cause') and isinstance(exc.cause, HTTPError):
        response = exc.cause.response
    if response is not None:
        return parse_response(response)


def get_error_data(exc):
    return {
        'classname': get_classname(exc),
        'cause': get_classname(getattr(exc, 'cause', None)),
        'message': str(exc),
        'traceback': ''.join(traceback.format_tb(sys.exc_info()[2]))  # todo: test me
    }


@app.task(bind=True, default_retry_delay=60 * 3, max_retries=6, time_limit=60, soft_time_limit=30)
def send_notification(self, notification_id, queue=None):
    from events.surveyme_integration.models import HookSubscriptionNotification
    from ylog.context import log_context
    queue = queue or settings.QUEUE_INTEGRATIONS
    try:
        notification = (
            HookSubscriptionNotification.objects.using(settings.DATABASE_ROLOCAL)
            .select_related(
                'subscription__service_type_action__service_type',
                'answer',
            )
            .get(id=notification_id)
        )
    except HookSubscriptionNotification.DoesNotExist as exc:
        raise self.retry(exc=exc, countdown=3)  # in 3 seconds

    notification_status = statuses_by_name[notification.status]
    if not notification_status.could_be_processed:  # todo: test me
        return

    notification.celery_task_id = self.request.id
    subscription = notification.subscription
    service_class = subscription.service_type_action.service_type.get_service_class()

    with log_context(subscription_id=subscription.id, notification_service_class=service_class.__name__):
        try:
            notification_date_updated = notification.date_updated or timezone.now()
            if not notification.context or subscription.date_updated > notification_date_updated:
                calculate_notification_context(service_class, notification)
            action_response = subscription.service_type_action.do_action(
                data=notification.context,
                status=notification.status,
            )
        except RetryTaskError:
            raise  # preserve retry
        except service_class.retriable_exceptions as exc:
            notification.error = get_error_data(exc)
            notification.response = get_response_data(exc)
            notification.max_retries = self.max_retries + 1

            if queue != settings.QUEUE_RETRIES:
                countdown = settings.DEFAULT_COUNTDOWN
                notification.date_retry = timezone.now() + timedelta(seconds=countdown)
                notification.save()

                start_notification(notification.pk, settings.QUEUE_RETRIES, countdown)
            else:
                # with max_retries=60 it's going to be 29 hours retry window sum([i for i in range(1, 60)]) / 60
                countdown = settings.DEFAULT_COUNTDOWN * (self.request.retries + 2)
                notification.date_retry = timezone.now() + timedelta(seconds=countdown)
                notification.retries = self.request.retries + 1
                notification.save()

                try:
                    raise self.retry(exc=exc, countdown=countdown)
                except service_class.retriable_exceptions as exc2:
                    logger.warn('Max retries exceeded for notification %s', notification.pk)
                    mark_notification_as_failed(notification, exc2)
        except Exception as exc:
            mark_notification_as_failed(notification, exc)
        else:
            notification.date_finished = timezone.now()
            notification.status = action_response['status']
            notification.response = action_response['response']
            notification.save()


def calculate_notification_context(service_class, notification):
    if notification.subscription.context_language == 'from_request':
        lang = notification.answer.get_answer_language()
    else:
        lang = notification.subscription.context_language
    with override_lang(lang):
        notification.context = service_class().get_context_processor(
            subscription=notification.subscription,
            answer=notification.answer,
            trigger_data=notification.trigger_data,
            notification_unique_id=str(notification.pk),
            force_render=True,
        ).data

    notification.date_updated = timezone.now()
    notification.save()


def mark_notification_as_failed(notification, exc):
    change_counter = False
    if notification.status == 'pending' or not notification.is_visible:
        change_counter = True
    if change_counter:
        notification.increment_counter()

    logger.info(f'Marking notification {notification.id} as failed')

    notification.status = 'error'
    notification.is_visible = True
    notification.date_finished = timezone.now()
    notification.error = get_error_data(exc)
    notification.response = get_response_data(exc)
    notification.save()

    if settings.USE_NEW_FOLLOWERS:
        send_information_about_failed_notification(notification)
    else:
        send_information_about_failed_notification_old(notification)


def send_information_about_failed_notification(notification):
    survey = notification.survey
    subscription = notification.subscription

    error_type = notification.error.get('classname', '').rsplit('.', maxsplit=1)[-1]

    if survey.followers:
        if survey.follow_type is None and not settings.IS_BUSINESS_SITE:
            if settings.IS_BUSINESS_SITE and survey.org_id is not None:
                admin_url = '/cloud/admin'
            else:
                admin_url = '/admin'
            ct = ContentType.objects.get_for_model(Survey)
            followers_qs = ContentFollower.objects.using(settings.DATABASE_ROLOCAL).filter(
                content_type=ct,
                object_id=survey.id,
            ).values_list('email', 'user__email', 'secret_code', named=True)
            follower_emails = set((it.email or it.user__email, it.secret_code) for it in followers_qs)
            email_json = {
                'subject': '[Ошибка интеграции] {}'.format(survey.name),
                'survey_name': survey.name,
                'survey_href': f'{admin_url}/{survey.id}',
                'error': error_type,
                'subscription_id': subscription.id,
                'notification_json': {
                    'notification_href': (
                        f'/admin/notifications-history?status=error&integration='
                        f'{subscription.id}&survey={survey.id}&notification={notification.id}'
                    ),
                    'subscription_href': f'{admin_url}/{survey.id}/notifications?integration={subscription.id}',
                },
            }
            campaign = f'{settings.APP_TYPE.removesuffix("_admin")}_failed_integration'
            for email, secret_code in follower_emails:
                email_json['secret_code_href'] = f'{admin_url}/unsubscribe/{secret_code}'
                send_email(campaign, email_json, [email], has_ugc=False)
            return
        now = timezone.now()
        match survey.follow_type:
            case '5m':
                round_dt = round_datetime_five_minutes_down(now).strftime('%H%M')
                key = f'follow:{survey.follow_type}:{round_dt}:{survey.id}:{subscription.id}:{error_type}'
            case '1h':
                hour = now.strftime('%H')
                key = f'follow:{survey.follow_type}:{hour}:{survey.id}:{subscription.id}:{error_type}'
            case '1d':
                key = f'follow:{survey.follow_type}::{survey.id}:{subscription.id}:{error_type}'
            case '_':
                logger.error(f'Unknown follow_type {survey.follow_type}')
                return
        redis.hset(key, 'notification_id', str(notification.id))
        redis.hincrby(key, 'notifications_count')


def send_information_about_failed_notification_old(notification):
    # todo: test me
    from events.surveyme_integration.models import SurveyHookTrigger
    from django.contrib.contenttypes.models import ContentType
    from django.template.loader import get_template

    survey = notification.survey
    trigger = SurveyHookTrigger.objects.get(slug=notification.trigger_slug)
    service_type_action = notification.subscription.service_type_action
    answer_content_type = ContentType.objects.get_for_model(ProfileSurveyAnswer)

    subject = '[Ошибка интеграции] {}'.format(survey.name)
    context_params = dict(
        trigger_title=trigger.title.lower(),
        answer_href=get_backoffice_url(
            content_type_id=answer_content_type.id,
            object_id=notification.answer_id
        ),
        survey_href=get_backoffice_url_for_obj(survey),
        survey_title=survey.name,
        notification_href=get_backoffice_url_for_obj(notification),
        notification_title='%s. %s' % (service_type_action.service_type.title, service_type_action.title),
        notification=notification,
    )

    admin_emails = set(email for _, email in settings.ADMINS)
    if admin_emails and not settings.IS_BUSINESS_SITE:
        template = get_template('surveyme_integration/admin.html')
        content = template.render(context_params)
        send_email_with_retry(subject, content, admin_emails, from_email='robot-forms@yandex-team.ru')

    follower_emails = set(follower.get_email() for follower in survey.followers.all()) - admin_emails
    if follower_emails and not settings.IS_BUSINESS_SITE:
        template = get_template('surveyme_integration/follower.html')
        content = template.render(context_params)
        send_email_with_retry(subject, content, follower_emails)


def build_key_glob_for_get_aggregated_notification_info(follow_type: str):
    now = timezone.now()
    match follow_type:
        case '5m':
            round_dt = round_datetime_five_minutes_down(now - timedelta(minutes=5)).strftime('%H%M')
            return f'follow:{follow_type}:{round_dt}'
        case '1h':
            hour = (now - timedelta(hours=1)).strftime('%H')
            return f'follow:{follow_type}:{hour}'
        case '1d':
            return f'follow:{follow_type}:'


def get_aggregated_notification_info_by_key_glob_from_redis_and_send(key_glob: str):
    from events.surveyme_integration.models import HookSubscriptionNotification

    survey_to_subscription_error = defaultdict(list)
    for key in redis.keys(f'{key_glob}:*'):
        _, survey_id, subscription, error_type = key.rsplit(':', maxsplit=3)
        survey_to_subscription_error[survey_id].append((subscription, error_type))
    keys = []
    ct = ContentType.objects.get_for_model(Survey)
    followers_qs = (
        ContentFollower.objects.using(settings.DATABASE_ROLOCAL)
            .filter(content_type=ct, object_id__in=survey_to_subscription_error.keys())
            .values_list('object_id', 'email', 'user__email', 'secret_code', named=True)
    )
    followers = defaultdict(set)
    for it in followers_qs:
        followers[str(it.object_id)].add((it.email or it.user__email, it.secret_code))
    surveys_qs = (
        Survey.objects.using(settings.DATABASE_ROLOCAL)
            .filter(id__in=survey_to_subscription_error.keys())
            .values_list('id', 'name', 'org_id', named=True)
    )
    surveys = dict()
    for it in surveys_qs:
        surveys[str(it.id)] = (it.name, it.org_id)

    for survey_id, (survey_name, survey_org_id) in surveys.items():
        follower_emails = followers[survey_id]
        if not follower_emails:
            return
        email_json = dict()
        if settings.IS_BUSINESS_SITE and survey_org_id is not None:
            admin_url = '/cloud/admin'
        else:
            admin_url = '/admin'
        email_json['admin_url'] = admin_url
        email_json['survey_href'] = f'{admin_url}/{survey_id}'
        email_json['notifications_history_href'] = f'{admin_url}/notifications-history/?survey={survey_id}&status=error'
        email_json['subject'] = '[Ошибка интеграции] {}'.format(survey_name)
        email_json['survey_name'] = survey_name
        email_json['subscription_errors'] = dict()
        for subscription_id, error in survey_to_subscription_error[survey_id]:
            key = f'{key_glob}:{survey_id}:{subscription_id}:{error}'
            notification_json = redis.hgetall(key)
            keys.append(key)
            if not notification_json:
                continue
            try:
                secret_code = (
                    HookSubscriptionNotification.objects.using(settings.DATABASE_ROLOCAL)
                        .select_related('answer__secret_code')
                        .values_list('answer__secret_code', named=True)
                        .get(id=notification_json['notification_id'])
                ).answer__secret_code
            except HookSubscriptionNotification.DoesNotExist:
                continue
            notification_json['answer_href'] = f'/admin/answer-details/{secret_code}'
            # notification_json['answer_href'] = f'/admin/answers/{answer_id}/'
            notification_json['notification_href'] = (
                f'/admin/notifications-history?status=error&integration='
                f'{subscription_id}&survey={survey_id}&notification={notification_json["notification_id"]}'
            )
            notification_json['subscription_href'] = (
                f'{admin_url}/{survey_id}/notifications?integration={subscription_id}'
            )

            email_json['subscription_errors'][f'{subscription_id}:{error}'] = notification_json
        if email_json['subscription_errors']:
            campaign = f'{settings.APP_TYPE.removesuffix("_admin")}_failed_integration_aggregated'
            for email, secret_code in follower_emails:
                email_json['secret_code_href'] = f'{admin_url}/unsubscribe/{secret_code}'
                send_email(campaign, email_json, [email], has_ugc=False)
    redis.delete(*keys)


def get_aggregated_notification_info_from_redis_and_send(follow_type: str):
    key_glob = build_key_glob_for_get_aggregated_notification_info(follow_type)
    if key_glob is None:
        logger.error(f'Unknown follow_type {follow_type}')
        return
    try:
        with RedisLock(
            f'get_aggregated_notification_info_from_redis_and_send_{key_glob}',
            caches['ylock_redis'],
            RedisLockException,
        ):
            get_aggregated_notification_info_by_key_glob_from_redis_and_send(key_glob)
    except RedisLockException:
        logger.info('Same celery task is already running')


@app.task
def get_aggregated_notification_info_from_redis_and_send_5m():
    get_aggregated_notification_info_from_redis_and_send('5m')


@app.task
def get_aggregated_notification_info_from_redis_and_send_1h():
    get_aggregated_notification_info_from_redis_and_send('1h')


@app.task
def get_aggregated_notification_info_from_redis_and_send_1d():
    get_aggregated_notification_info_from_redis_and_send('1d')


@app.task
def update_hook_subscription_notification_counter():
    if connection.vendor == 'postgresql':
        sql = '''
            insert into surveyme_integration_hooksubscriptionnotificationcounter as f(
                survey_id, subscription_id, errors_count, date_updated
            )
            select t.survey_id, t.subscription_id, count(t.id), %s
            from surveyme_integration_hooksubscriptionnotification t
            join surveyme_integration_hooksubscriptionnotificationcounter c on(
                c.subscription_id = t.subscription_id
            )
            where t.date_finished > c.date_updated
                and t.status = 'error'
                and t.is_visible
            group by t.survey_id, t.subscription_id
            on conflict(subscription_id) do update set
                errors_count = f.errors_count + excluded.errors_count,
                date_updated = excluded.date_updated
        '''
    elif connection.vendor == 'sqlite':
        sql = '''
            update surveyme_integration_hooksubscriptionnotificationcounter set
                errors_count = errors_count + (
                    select count(t.id)
                    from surveyme_integration_hooksubscriptionnotification t
                    where t.status = 'error'
                        and t.is_visible
                        and t.subscription_id = surveyme_integration_hooksubscriptionnotificationcounter.subscription_id
                        and t.date_finished > surveyme_integration_hooksubscriptionnotificationcounter.date_updated
                ),
                date_updated = %s
            where exists(
                select null
                from surveyme_integration_hooksubscriptionnotification t
                where t.status = 'error'
                    and t.is_visible
                    and t.subscription_id = surveyme_integration_hooksubscriptionnotificationcounter.subscription_id
                    and t.date_finished > surveyme_integration_hooksubscriptionnotificationcounter.date_updated
            )
        '''
    else:
        raise NotImplementedError('Task not implemented for `%s` database vendor' % connection.vendor)
    with connection.cursor() as c:
        c.execute(sql, [timezone.now()])


def start_integrations(survey_id, answer_id, trigger_slug, trigger_data=None, answer_obj=None):
    answer = (
        ProfileSurveyAnswer.objects
        .select_related(
            'user',
            'survey',
        )
        .get(pk=answer_id)
    )
    subscription_qs = (
        ServiceSurveyHookSubscription.objects.using(settings.DATABASE_ROLOCAL)
        .select_related(
            'service_type_action__service_type',
        )
    )
    hook_conditions_qs = (
        SurveyHookCondition.objects.using(settings.DATABASE_ROLOCAL)
        .select_related(
            'content_type_attribute__content_type',
            'survey_question',
        )
    )
    hooks_qs = (
        SurveyHook.objects.using(settings.DATABASE_ROLOCAL)
        .filter(
            survey_id=survey_id,
            is_active=True,
        )
        .filter(
            Q(triggers__isnull=True)
            | Q(triggers__slug__in=[trigger_slug])
        )
        .prefetch_related(
            Prefetch('subscriptions', queryset=subscription_qs),
            Prefetch('condition_nodes__items', queryset=hook_conditions_qs),
        )
    )
    subscriptions_data, notifications = [], []
    for hook in hooks_qs:
        if hook.is_true(profile_survey_answer=answer):
            sd, n = hook.notify_subscriptions(answer, trigger_slug, trigger_data=trigger_data)
            subscriptions_data.extend(sd)
            notifications.extend(n)

    if notifications:
        HookSubscriptionNotification.objects.bulk_create(notifications)
        start_notifications.delay(answer_id)

    return subscriptions_data


def start_notification(notification_id, queue=None, countdown=None):
    queue = queue or settings.QUEUE_INTEGRATIONS
    args = (notification_id, queue)
    send_notification.apply_async(args, queue=queue, countdown=countdown)


@app.task
def start_notifications(answer_id):
    notifications = list(HookSubscriptionNotification.objects.filter(answer_id=answer_id, status='pending'))

    if notifications:
        notification, *delayed = notifications
        for it in delayed:
            start_notification(it.pk)
        try:
            send_notification(notification.pk)
        except RetriableError:
            notification.status = 'pending'
            notification.save(update_fields=['status'])
            start_notification(notification.pk, settings.QUEUE_RETRIES, settings.DEFAULT_COUNTDOWN)
