# -*- coding: utf-8 -*-
import base64
import re
import logging
import itertools
from collections import defaultdict

from django.conf import settings
from django.utils.encoding import force_str
from django.utils.translation import ugettext as _
from django.utils import timezone

from events.common_app.directory import CachedDirectoryClient
from events.common_app.utils import lazy_re_compile, import_from_string

logger = logging.getLogger(__name__)


def render_variable(config, answer, trigger_data, notification_id, force_render=False, variable_data=None, filters=None):
    from events.surveyme_integration.filters import filters_by_name
    from events.surveyme_integration.variables import variables_by_name

    variable_class = variables_by_name.get(config.var)
    if not variable_class:
        logger.warning('Found unknown macros "%s"', config.var)
        return ''
    variable_kwargs = config.arguments or {}
    variable_kwargs['answer'] = answer
    variable_kwargs['trigger_data'] = trigger_data
    variable_kwargs['notification_id'] = notification_id
    variable_kwargs['variable_data'] = variable_data
    format_name = config.format_name
    value = None
    try:
        value = variable_class(**variable_kwargs).get_value(format_name=format_name)
    except Exception as exc:
        logger.exception('Macros "%s" was rendered with error %s', config.var, exc)
    filters = filters or []
    if config.filters:
        filters = itertools.chain(filters, config.filters)
    for filter_name in filters:
        value = filters_by_name[filter_name]().apply_filter(value)
    if value is None:
        value = ''
    return force_str(value)


class FormatError(Exception):
    """ Ошибка в формате строки. """


macros_re = lazy_re_compile(r'\{([\da-fA-F]+)\}')


def get_variables_keys(template_text):
    return set(macros_re.findall(template_text))


def render_variables(value,
                     object_with_variables,
                     answer,
                     trigger_data,
                     notification_unique_id,
                     force_render=False,
                     filters=None,
                     ):
    from events.surveyme_integration.variables import variables_by_name
    template_text = value or ''
    variables = object_with_variables.variables_map
    bulk_variables = defaultdict(list)
    for variable_key in get_variables_keys(template_text):
        var = variables.get(variable_key)

        if var is not None:
            variable_class = variables_by_name.get(var.var)
            if not variable_class:
                template_text = template_text.replace(f'{{{variable_key}}}', _('Данные не найдены'))
                continue

            var_group = get_var_group(var)
            if is_bulk_group(var_group) and variable_class.is_bulk:
                bulk_variables[var_group].append((variable_key, var))

            else:
                template_text = perform_render(
                    var, variable_key, template_text, answer,
                    trigger_data, notification_unique_id, force_render,
                    filters=filters,
                )
        elif not force_render:
            raise FormatError(_('Найден несуществующий макрос %s. '
                                'Удалите его и перезапустите интеграцию.') % variable_key)

    template_text = render_bulk_variables(bulk_variables, answer, trigger_data,
                                          notification_unique_id, force_render,
                                          template_text, filters,
                                          )
    return template_text


def render_bulk_variables(bulk_variables, answer, trigger_data, notification_unique_id,
                          force_render, template_text, filters,
                          ):
    """
    Заменяем в template_text балковые макросы на их значения

    Получаем для каждой группы балковых макросов данные,
    которые затем используем для получения значений макросов
    позволяя избежать повторного получения/вычисления данных,
    которые необходимы макросам

    Пример:
    bulk_variables = {'staff': [('staff.birth_date', <variable_data>),
                                ('staff.personal.family', <variable_data>),
                                ]
    получаем BulkVariableRequester для staff, передаем в него данные
    всех нужных макросов, вызываем get_response(), BulkVariableRequester
    сформирует один запрос к стаффу из которого можно будет получить всю нужную
    для обоих макросов информацию, затем полученные данные передаются этим
    макросов для обработки после чего происходит замена макроса в template на
    результат работы данного макроса. Таким образом, вместо выполнения отдельных запросов
    на каждый макрос, делаем один запрос на всю группу макросов.
    """
    for group_name, group_data in bulk_variables.items():
        path = 'events.surveyme_integration.variables.{}.BulkVariableRequester'.format(group_name)
        group_requester = import_from_string(path, 'BulkVariableRequester')
        group_variable_data = group_requester(group_data, answer).get_response()
        for variable_key, var in group_data:
            template_text = perform_render(
                var, variable_key, template_text, answer,
                trigger_data, notification_unique_id, force_render,
                variable_data=group_variable_data, filters=filters,
            )
    return template_text


def perform_render(var, variable_key, template_text, answer, trigger_data,
                   notification_unique_id, force_render, variable_data=None, filters=None):
    variable_text = render_variable(
        var, answer, trigger_data, notification_unique_id,
        force_render, variable_data, filters=filters,
    )
    template_text = template_text.replace(''.join(['{', variable_key, '}']), variable_text)
    return template_text


def is_bulk_group(var_group):
    from events.surveyme_integration.variables import bulk_variable_groups
    return var_group in bulk_variable_groups


def get_var_group(var):
    return var.var.split('.')[0]


def get_short_dir_userinfo(org_dir_id, uid):
    client = CachedDirectoryClient()
    fields = 'contacts,department.name,groups.name,groups'
    short_userinfo = {}
    userinfo = client.get_user(org_dir_id, uid, fields=fields)
    if not userinfo:
        short_userinfo

    department = userinfo.get('department')
    if department is not None:
        short_userinfo['department'] = {'id': department.get('id')}
        department_name = department.get('name')  # dict with lower-case language code as a keys
        if department_name:
            short_userinfo['department']['name'] = department_name

    phones = []
    for contact in userinfo.get('contacts', []):
        if contact['type'] == 'phone':
            phone = contact['value']
            if phone:
                phones.append(phone)
    if phones:
        short_userinfo['phones'] = phones

    groups = []
    for group in userinfo.get('groups', []):
        group_info = {}
        group_name = group.get('name')  # dict with lower-case language code as a keys
        if group_name:
            group_info['name'] = group_name
        if group_info:
            groups.append(group_info)
    if groups:
        short_userinfo['groups'] = groups

    return short_userinfo


def get_short_dir_department_info(org_dir_id, id):
    client = CachedDirectoryClient()
    depinfo = client.get_department(org_dir_id, id, fields='id,head.name')
    short_depinfo = {}
    if not depinfo:
        return short_depinfo

    head = depinfo.get('head', {})
    if head is not None:
        name = head.get('name')
        if name:
            short_depinfo['head'] = {'name': name}

    return short_depinfo


def update_notification_counter(subscription):
    from events.surveyme_integration.models import (
        HookSubscriptionNotification,
        HookSubscriptionNotificationCounter,
    )
    notifications_queryset = (
        HookSubscriptionNotification.objects
        .filter(subscription=subscription, status='error', is_visible=True)
    )
    update_queryset = (
        HookSubscriptionNotificationCounter.objects
        .filter(subscription=subscription)
    )
    update_queryset.update(errors_count=notifications_queryset.count(), date_updated=timezone.now())


def get_message_id(data):
    app_type = settings.APP_TYPE
    hostname = settings.HOSTNAME
    if isinstance(data, dict):
        survey_id = data['survey_id']
        answer_id = data['answer_id']
        notification_id = data['notification_unique_id']
    else:
        survey_id = data.answer.survey_id
        answer_id = data.answer.pk
        notification_id = data.notification_unique_id
    return f'<{app_type}.{survey_id}.{answer_id}.{notification_id}@{hostname}>'


def get_email_with_name(long_email):
    m = re.search(r'(.*)<([^>]+)>', long_email)
    if not m:
        return long_email, ''
    email = m.group(2)
    name = m.group(1).strip(' "')
    if not name:
        return email, ''
    return email, name


def encode_string(s):
    if not s:
        return s
    b64_s = base64.b64encode(s.encode()).decode()
    return f'=?utf-8?B?{b64_s}?='


def encode_long_email(long_email):
    email, name = get_email_with_name(long_email)
    if not name:
        return email
    b64_name = base64.b64encode(f'"{name}"'.encode()).decode()
    return f'=?utf-8?B?{b64_name}?= <{email}>'


def decode_long_email(long_email):
    m = re.match(r'=\?utf-8\?B\?([^?]+)\?= <([^>]+)>', long_email)
    if not m:
        return long_email, ''
    return m.group(2), base64.b64decode(m.group(1).encode()).decode()


def get_valid_headers(headers):
    re_valid_key = re.compile(r'^(x|content)-', re.I)
    return {
        key: value
        for key, value in headers.items()
        if re_valid_key.search(key)
    }


def parse_response(response):
    return {
        'headers': get_valid_headers(response.headers),
        'status_code': response.status_code,
        'content': force_str(response.content),
        'method': response.request.method,
        'url': response.url,
    }


def round_datetime_five_minutes_down(datetime_obj):
    new_datetime_obj = datetime_obj.replace(minute=datetime_obj.minute // 5 * 5, second=0, microsecond=0)
    return new_datetime_obj
