# -*- coding: utf-8 -*-
import json

from django.utils.translation import ugettext_lazy as _, override as override_lang
from django.utils.timezone import get_default_timezone

from events.surveyme_integration.variables.base import (
    BaseVariableCategory,
    BaseVariable,
    BaseRenderer,
)
from events.surveyme.answer_value import (
    get_value_for_question,
    get_value_for_questions,
    find_question_data,
)


class FormVariableCategory(BaseVariableCategory):
    name = 'form'
    title = _('Форма')
    order = 1


class FormAnswerIdVariable(BaseVariable):
    name = 'form.answer_id'
    title = _('ID ответа')
    category = FormVariableCategory()

    def get_value(self, format_name=None):
        return self.answer.id


class FormDateVariable(BaseVariable):
    name = 'form.date_updated'
    title = _('Дата ответа')
    category = FormVariableCategory()

    def get_value(self, format_name=None):
        local_dt = self.answer.date_updated.astimezone(get_default_timezone())
        return local_dt.strftime('%d.%m.%Y')


class FormIdVariable(BaseVariable):
    name = 'form.id'
    title = _('ID формы')
    category = FormVariableCategory()

    def get_value(self, format_name=None):
        return self.answer.survey.id


class FormNameVariable(BaseVariable):
    name = 'form.name'
    title = _('Название')
    category = FormVariableCategory()

    def get_value(self, format_name=None):
        language = self.answer.get_answer_language()
        with override_lang(language):
            return self.answer.survey.get_name()


class FormQuestionAnswerChoicesIdsVariable(BaseVariable):
    name = 'form.question_answer_id'
    title = _('ID варианта ответа на вопрос')
    category = FormVariableCategory()
    arguments = ['question']
    filters = {'question': {'answer_type': 'answer_choices'}}

    def __init__(self, question, **kwargs):
        super(FormQuestionAnswerChoicesIdsVariable, self).__init__(**kwargs)
        self.question = question

    def _get_single_value(self, question_data):
        slug = question_data.get('question', {}).get('answer_type', {}).get('slug')
        if slug == 'answer_choices':
            for it in question_data.get('value') or []:
                yield it.get('key')

    def _get_value(self, answer_data, question):
        for question_data in find_question_data(answer_data, question):
            yield from self._get_single_value(question_data)

    def get_value(self, format_name=None):
        answer_data = self.answer.data.get('data')
        return ','.join(self._get_value(answer_data, self.question))


class FormQuestionAnswerChoiceSlugVariable(BaseVariable):
    name = 'form.question_answer_choice_slug'
    title = _('Идентификатор варианта ответа на вопрос')
    category = FormVariableCategory()
    arguments = ['question']
    filters = {'question': {'answer_type': 'answer_choices'}}

    def __init__(self, question, **kwargs):
        super(FormQuestionAnswerChoiceSlugVariable, self).__init__(**kwargs)
        self.question = question

    def _get_single_value(self, question_data):
        slug = question_data.get('question', {}).get('answer_type', {}).get('slug')
        if slug == 'answer_choices':
            for it in question_data.get('value') or []:
                yield it.get('slug') or it.get('key')

    def _get_value(self, answer_data, question):
        for question_data in find_question_data(answer_data, question):
            yield from self._get_single_value(question_data)

    def get_value(self, format_name=None):
        answer_data = self.answer.data.get('data')
        return ','.join(self._get_value(answer_data, self.question))


class FormQuestionAnswerVariable(BaseVariable):
    name = 'form.question_answer'
    title = _('Ответ на вопрос')
    category = FormVariableCategory()
    arguments = ['question', 'show_filenames']

    def __init__(self, question, show_filenames=False, **kwargs):
        super(FormQuestionAnswerVariable, self).__init__(**kwargs)
        self.question = question
        self.show_filenames = show_filenames

    def get_value(self, format_name=None):
        if not self.question:
            return ''
        return get_value_for_question(
            self.answer, self.question,
            show_filenames=self.show_filenames,
        )


class QuestionAnswersTxtRenderer(BaseRenderer):
    format_name = 'txt'
    format_title = _('Простой текст')

    def render(self, data):
        return '\n\n'.join(['%s:\n%s' % (title, value or 'Нет ответа') for title, value in data])


class QuestionAnswersStartrekRenderer(BaseRenderer):
    format_name = 'startrek'
    format_title = _('Для стартрека')

    def render(self, data):
        return '\n'.join(
            [
                '**{title}**\n%%\n{value}\n%%\n'.format(
                    title=title,
                    value=value or 'Нет ответа'
                )
                for title, value in data
            ]
        )


class QuestionAnswersJsonRenderer(BaseRenderer):
    format_name = 'json'
    format_title = _('JSON')

    def render(self, data):
        return json.dumps({title: value or 'Нет ответа' for title, value in data}, ensure_ascii=False)


class FormQuestionsAnswersVariable(BaseVariable):
    name = 'form.questions_answers'
    title = _('Ответы на вопросы')
    category = FormVariableCategory()
    arguments = ['questions', 'is_all_questions', 'only_with_value', 'show_filenames']
    renderer_classes = [QuestionAnswersTxtRenderer, QuestionAnswersStartrekRenderer, QuestionAnswersJsonRenderer]

    def __init__(self, questions=None, is_all_questions=False, only_with_value=True, show_filenames=False, **kwargs):
        super(FormQuestionsAnswersVariable, self).__init__(**kwargs)
        self.questions = questions
        self.is_all_questions = is_all_questions
        self.only_with_value = only_with_value
        self.show_filenames = show_filenames

    def get_value(self, format_name=None):
        if self.is_all_questions:
            self.questions = None

        data = get_value_for_questions(
            self.answer, self.questions,
            show_filename=self.show_filenames,
            only_with_value=self.only_with_value,
        )

        renderer_class = self.get_renderer_class(format_name=format_name) or self.renderer_classes[0]
        return renderer_class().render(data=data)


# Костыль для FORM-666
# Поэтому отдельной категорией и переменной.

class LawyersVariableCategory(BaseVariableCategory):
    name = 'lawyers'
    title = _('Юристы')


class FormQuestionAnswerWithPrefixVariable(BaseVariable):
    """
    """
    name = 'lawyers.question_answer_with_prefix'
    title = _('Текст + ответ на вопрос')
    category = LawyersVariableCategory()
    arguments = ['question', 'show_filenames', 'prefix', 'only_with_value']

    def __init__(self, question, show_filenames=False, only_with_value=False, prefix=None, **kwargs):
        super(FormQuestionAnswerWithPrefixVariable, self).__init__(**kwargs)
        self.question = question
        self.only_with_value = only_with_value
        self.show_filenames = show_filenames
        self.prefix = prefix

    def get_value(self, format_name=None):
        if not self.question:
            return ''
        value = get_value_for_question(
            self.answer, self.question,
            show_filenames=self.show_filenames,
        )
        if not value and self.only_with_value:
            return ''
        else:
            return '%s %s' % (self.prefix, value)


class FormValidationVariable(BaseVariable):
    name = 'form.validation_status'
    title = _('Статус валидации')
    category = FormVariableCategory()

    def get_value(self, format_name=None):
        return self.answer.source_request['validation_status']


class FormAuthorEmailVariable(BaseVariable):
    name = 'form.author_email'
    title = _('Email автора формы')
    category = FormVariableCategory()

    def get_value(self, format_name=None):
        return self.answer.survey.user.get_default_email()
