# -*- coding: utf-8 -*-
import logging

from django.utils.translation import get_language
from django.utils.translation import ugettext_lazy as _
from ids.exceptions import BackendError
from ylog.context import log_context

from events.staff.client import get_staff_repository
from events.staff.utils import find_heads_by_type
from events.surveyme_integration.variables.base import (
    BaseVariableCategory,
    BaseVariable,
    BaseRenderer,
    BulkVariableRequester as BaseBulkVariableRequester,
)
from events.surveyme_integration.filters import LoginFilter
from events.surveyme.answer_value import get_value_for_question


logger = logging.getLogger(__name__)


def format_login_key(variable):
    return '{}_{}'.format(variable.question,
                          variable.format_name,
                          )


def get_staff_response(fields, *logins):
    logins = set(logins)
    with log_context(logins=repr(logins), fields=repr(fields)):
        try:
            staff_repository = get_staff_repository('person')
            variable_data = staff_repository.get(
                lookup={
                    'login': ','.join(logins),
                    '_fields': 'login,{}'.format(fields),
                    '_limit': len(logins)
                },
            )
        except BackendError:
            logger.exception('Got exception from staff-api')
            return dict()
        else:
            if len(variable_data) != len(logins):
                logger.warning('Don\'t get response for all logins')
            return {
                person_data['login']: person_data
                for person_data in variable_data
            }


class BulkVariableRequester(BaseBulkVariableRequester):
    def get_response(self):
        from events.surveyme_integration.variables import variables_by_name
        variables_instances = [
            variables_by_name[config.var](
                question=(config.arguments or {}).get('question'),
                answer=self.answer,
            ).get_renderer_variable_instance(
                format_name=config.format_name
            )
            for _, config in self.group_data
        ]
        users_logins = self.get_logins(variables_instances)
        if users_logins:
            fields = {
                variable_instance.field
                for variable_instance in variables_instances
            }
            fields = self.format_fields(fields)
            return {
                'response': get_staff_response(fields, *iter(users_logins.values())),
                'logins_data': users_logins,
            }

    def get_logins(self, variables_instances):
        logins_data = {}
        for variable_instance in variables_instances:
            variable_login = variable_instance.get_login()
            if variable_login:
                logins_data[format_login_key(variable_instance)] = variable_login
        return logins_data

    def format_fields(self, fields):
        """
        Формируем строку для поля _fields в апи стафф

        >>> fields = ('office.city', 'office.location')
        >>> BulkVariableRequester.format_fields(fields)
        'office.city,office.location'

        >>> fields = ('office.city.name', 'office.city')
        >>> BulkVariableRequester.format_fields(fields)
        'office.city'

        >>> fields = ('office.city.name', 'office.location')
        >>> BulkVariableRequester.format_fields(fields)
        'office.city.name,office.location'

        :param fields: iterable
        :return: str
        """
        prepared_fields = [field + '.' for field in fields]
        result_fields = list()
        for index, field in enumerate(prepared_fields):
            other_fields = prepared_fields[:index] + prepared_fields[index + 1:]
            for field_to_check in other_fields:
                if field.find(field_to_check) == 0:
                    break
            else:
                result_fields.append(field[:-1])
        return ','.join(result_fields)


class StaffLanguageMixin(object):
    """
    Следует добавлять в базовые классы для
    макросов стаффа, которые возвращают значения
    в нескольких языках

    Учитываем, что ответ апи стаффа на нужном
    языке может быть пуст, отдаем ответ на
    дефолтном в таком случае
    """
    ACCEPTED_LANGUAGES = {'en', 'ru', }
    DEFAULT_LANGUAGE = 'ru'

    def get_value(self, format_name=None):
        result = self.get_response_from_staff(self.field)
        lang = get_language()
        lang = lang if lang in self.ACCEPTED_LANGUAGES else self.DEFAULT_LANGUAGE
        translated_result = result.get(lang)
        if not translated_result and lang != self.DEFAULT_LANGUAGE:
            translated_result = result.get(self.DEFAULT_LANGUAGE)
        return self.parse_result(translated_result)


class StaffVariableCategory(BaseVariableCategory):
    name = 'staff'
    title = _('Данные со Стаффа')
    order = 3


class StaffBaseVariable(BaseVariable):
    category = StaffVariableCategory()
    is_bulk = True

    def _get_external_login(self, email):
        external_login, *domain = email.split('@', 1)
        if domain and domain[0].startswith('yandex.'):
            return external_login

    def _get_internal_login_by_external(self, external_login):
        try:
            staff_repository = get_staff_repository('person')
            internal_login = staff_repository.get_one(
                lookup={
                    '_query': 'yandex.login==regex("^%s$","i")' % external_login,
                    '_fields': 'login',
                },
            )
            return internal_login['login']
        except (BackendError, KeyError):
            pass


class StaffBaseVariableRenderer(BaseRenderer, BaseVariable):

    def __init__(self, question=None, **kwargs):
        super(StaffBaseVariableRenderer, self).__init__(**kwargs)
        self.question = question

    def get_response_from_staff(self, field):
        login = self.login
        if not login:
            return
        user_variable_data = self.variable_data.get('response', {}).get(login)
        if not user_variable_data:
            user_variable_data = get_staff_response(field, login).get(login)
        if user_variable_data:
            return self.get_field_value(field.split('.'), user_variable_data)

    def get_login(self):
        login = self.variable_data.get('logins_data', {}).get(
            format_login_key(self)
        )
        if login is None:
            if self.question:
                result = get_value_for_question(self.answer, self.question)
                if result:
                    login = LoginFilter().get_login(result)
            elif not self.answer.user.is_anonymous:
                login = self.answer.user.get_yandex_username()
        if login:
            login = login.lower()
        return login

    def get_value(self, format_name=None):
        result = self.get_response_from_staff(self.field)
        return self.parse_result(result)

    def parse_result(self, result):
        if isinstance(result, dict):
            result = '\n'.join('{}: {}'.format(key, self.parse_result(value))
                               for key, value in result.items()
                               if key != 'id')
        elif isinstance(result, list):
            result = '\n'.join(self.parse_result(item)
                               for item in result
                               )
        return result

    def get_field_value(self, field_path, response, index=0):
        value = response[field_path[index]]
        index += 1
        if len(field_path) == index:
            return value
        return self.get_field_value(field_path, value, index)

    @property
    def login(self):
        attr_name = '_login'
        if not hasattr(self, attr_name):
            setattr(self, attr_name, self.get_login())
        return getattr(self, attr_name)


class StaffHeadBaseVariableRenderer(StaffBaseVariableRenderer):
    def get_value(self, format_name=None):
        result = self.get_response_from_staff(self.field)
        heads = find_heads_by_type(self.login, result,
                                   role=self.head_type,
                                   )
        return ','.join(heads)


class StaffBirthDateVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.birth_day'
    format_title = _('Дата рождения')
    field = 'personal.birthday'


class StaffLoginVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.login'
    format_title = _('Логин')
    field = 'login'


class StaffFirstNameVariableRenderer(StaffLanguageMixin, StaffBaseVariableRenderer):
    format_name = 'staff.first_name'
    format_title = _('Имя')
    field = 'name.first'


class StaffLastNameVariableRenderer(StaffLanguageMixin, StaffBaseVariableRenderer):
    format_name = 'staff.last_name'
    format_title = _('Фамилия')
    field = 'name.last'


class StaffMiddleNameVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.middle_name'
    format_title = _('Отчество')
    field = 'name.middle'


class StaffChildrenVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.children'
    format_title = _('Дети')
    field = 'personal.children'


class StaffFamilyStatusVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.family_status'
    format_title = _('Статус семьи')
    field = 'personal.family_status'


class StaffTshirtSizeVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.tshirt_size'
    format_title = _('Размер футболки')
    field = 'personal.tshirt_size'


class StaffCityVariableRenderer(StaffLanguageMixin, StaffBaseVariableRenderer):
    format_name = 'staff.city'
    format_title = _('Город')
    field = 'location.office.city.name'


class StaffOfficeVariableRenderer(StaffLanguageMixin, StaffBaseVariableRenderer):
    format_name = 'staff.office'
    format_title = _('Офис')
    field = 'location.office.name'


class StaffCarsVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.cars'
    format_title = _('Данные о машинах')
    field = 'cars'


class StaffContactsVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.contacts'
    format_title = _('Контакты')
    field = 'contacts'

    def parse_result(self, result):
        return '\n'.join('{}: {}'.format(contact['type'], contact['value'])
                         for contact in result)


class StaffDismissedVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.is_dismissed'
    format_title = _('Уволен')
    field = 'official.is_dismissed'


class StaffJoinAtVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.join_at'
    format_title = _('Дата начала работы')
    field = 'official.join_at'


class StaffQuitAtVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.quit_at'
    format_title = _('Дата увольнения')
    field = 'official.quit_at'


class StaffEmploymentVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.employment'
    format_title = _('Тип занятости')
    field = 'official.employment'


class StaffTableNumberVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.table_number'
    format_title = _('Номер стола')
    field = 'location.table.number'


class StaffHeadVariableRenderer(StaffHeadBaseVariableRenderer):
    format_name = 'staff.head'
    format_title = _('Руководитель')
    field = 'department_group'
    head_type = 'chief'


class StaffHrPartnerVariableRenderer(StaffHeadBaseVariableRenderer):
    format_name = 'staff.hr_partner'
    format_title = _('HR Партнеры')
    field = 'department_group'
    head_type = 'hr_partner'


class StaffDepartmentVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.department'
    format_title = _('Департамент')
    field = 'department_group.name'


class StaffDivisionVariableRenderer(StaffBaseVariableRenderer):
    format_name = 'staff.division'
    format_title = _('Подразделение')
    field = 'department_group.parent.name'


class StaffEducationVariableRenderer(StaffLanguageMixin, StaffBaseVariableRenderer):
    format_name = 'staff.education'
    format_title = _('Образование')
    field = 'education.place'


class StaffMetaUserVariable(StaffBaseVariable):
    name = 'staff.meta_user'
    title = _('Инфо о пользователе')
    arguments = []

    renderer_classes = [
        StaffFirstNameVariableRenderer,
        StaffLastNameVariableRenderer,
        StaffMiddleNameVariableRenderer,
        StaffBirthDateVariableRenderer,
        StaffCarsVariableRenderer,
        StaffChildrenVariableRenderer,
        StaffCityVariableRenderer,
        StaffContactsVariableRenderer,
        StaffDepartmentVariableRenderer,
        StaffDismissedVariableRenderer,
        StaffDivisionVariableRenderer,
        StaffEmploymentVariableRenderer,
        StaffFamilyStatusVariableRenderer,
        StaffHeadVariableRenderer,
        StaffHrPartnerVariableRenderer,
        StaffJoinAtVariableRenderer,
        StaffLoginVariableRenderer,
        StaffOfficeVariableRenderer,
        StaffQuitAtVariableRenderer,
        StaffTableNumberVariableRenderer,
        StaffTshirtSizeVariableRenderer,
        StaffEducationVariableRenderer,
    ]

    def __init__(self, question=None, **kwargs):
        super(StaffMetaUserVariable, self).__init__(**kwargs)
        self.kwargs = kwargs
        self.question = question

    def get_value(self, format_name=None):
        render_variable_instance = self.get_renderer_variable_instance(format_name=format_name)
        return render_variable_instance.get_value()

    def get_renderer_variable_instance(self, format_name):
        renderer_class = self.get_renderer_class(format_name=format_name) or self.renderer_classes[0]
        return renderer_class(question=self.question, **self.kwargs)


class StaffMetaQuestionVariable(StaffMetaUserVariable):
    name = 'staff.meta_question'
    title = _('Инфо из ответа на вопрос')
    filters = {'question': {'param_data_source': 'staff_login', }}
    arguments = ['question', ]


class StaffExternalLoginVariable(StaffBaseVariable):
    is_bulk = False
    name = 'staff.external_login'
    title = _('Логин на стаффе из вопроса')
    arguments = ['question', ]

    def __init__(self, question, **kwargs):
        super(StaffExternalLoginVariable, self).__init__(**kwargs)
        self.question = question

    def get_value(self, format_name=None):
        answer_email = get_value_for_question(self.answer, self.question)
        internal_login = None
        if answer_email:
            external_login = self._get_external_login(answer_email)
            if external_login:
                internal_login = self._get_internal_login_by_external(external_login)
        return internal_login


class StaffExternalLoginFromRequestVariable(StaffBaseVariable):
    is_bulk = False
    name = 'staff.external_login_from_request'
    title = _('Логин на стаффе из запроса')

    def get_value(self, format_name=None):
        user = self.answer.user
        internal_login = None
        if user:
            external_login = self._get_external_login(user.email)
            if external_login:
                internal_login = self._get_internal_login_by_external(external_login)
        return internal_login
