# -*- coding: utf-8 -*-
from rest_framework import serializers
from events.accounts.models import User

from events.rest_framework_contrib.serializers import InternalModelSerializerV2Mixin
from events.surveyme_keys.models import SurveyKeysBundle, Key


class SurveyKeysBundleSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    survey_id = serializers.IntegerField()
    keys_count = serializers.SerializerMethodField()

    def get_keys_count(self, obj):
        if hasattr(obj, 'keys_count'):
            return obj.keys_count  # needs annotation in view's queryset!

    class Meta:
        model = SurveyKeysBundle
        read_only_fields = ('date_created', 'date_updated')
        fields = (
            'id',
            'survey_id',
            'name',
            'allow_access_to_unpublished_form',
            'survey_frontend_base_url',
            'keys_count',
            'date_created',
            'date_updated',
        )


class KeyAuthorSerializer(serializers.ModelSerializer):
    name_and_surname_with_fallback = serializers.CharField(source='get_name_and_surname_with_fallback')

    class Meta:
        model = User
        fields = [
            'id',
            'name_and_surname_with_fallback',
        ]


class KeySerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    author = KeyAuthorSerializer(read_only=True, source='user')
    bundle_id = serializers.IntegerField()
    frontend_url = serializers.SerializerMethodField()

    def get_frontend_url(self, obj):
        return obj.get_frontend_url()

    class Meta:
        model = Key
        read_only_fields = ('date_created', 'date_updated')
        fields = (
            'id',
            'author',
            'bundle_id',
            'value',
            'is_active',
            'date_created',
            'date_updated',
            'date_deactivated',
            'deactivated_by_answer',
            'frontend_url',
            'is_available',
            'note',
            'source',
            'integration_id',
        )


class XLSXKeySerializer(KeySerializer):
    field_verbose_names = {
        'value': 'Ключ',
        'frontend_url': 'Адрес формы с ключем',
        'is_active': 'Активный',
        'is_available': 'Свободен',
    }

    class Meta(KeySerializer.Meta):
        fields = (
            'value',
            'frontend_url',
            'is_active',
            'is_available',
        )
