# -*- coding: utf-8 -*-
from django.db import models
from django.utils import timezone
from events.common_app.fields import SimpleTextField

from events.common_app.mixins import TimeStampMixin
from events.surveyme.models import Survey, ProfileSurveyAnswer
from events.surveyme_keys.managers import KeyManager
from events.common_app.utils import generate_code


KEY_SOURCE_CHOICES = (
    ('admin', 'Админка'),
    ('integration', 'Интеграция'),
)


class SurveyKeysBundle(TimeStampMixin, models.Model):
    survey = models.ForeignKey(Survey, related_name='keys_bundles', on_delete=models.DO_NOTHING)
    name = models.CharField(max_length=200)
    allow_access_to_unpublished_form = models.BooleanField(default=False)
    survey_frontend_base_url = models.CharField(max_length=200, blank=True)

    class Meta:
        ordering = ('-date_created',)

    def generate_keys(self, count, user_id, is_available, **kwargs):
        # todo: test me
        return Key.objects.bulk_create([
            Key(
                bundle=self,
                value=generate_code(),
                user_id=user_id,
                is_available=is_available,
                **kwargs
            )
            for _ in range(count)
        ])


class Key(TimeStampMixin, models.Model):
    user = models.ForeignKey('accounts.User', null=True, blank=True, on_delete=models.CASCADE)
    bundle = models.ForeignKey(SurveyKeysBundle, related_name='keys', on_delete=models.DO_NOTHING)
    value = models.CharField(max_length=200, db_index=True)
    # активность
    is_active = models.BooleanField(default=True)
    date_deactivated = models.DateTimeField(blank=True, null=True)
    deactivated_by_answer = models.ForeignKey(ProfileSurveyAnswer, blank=True, null=True, related_name='keys', on_delete=models.DO_NOTHING)
    # доступность
    is_available = models.BooleanField(default=True)
    note = SimpleTextField(blank=True, verbose_name='примечание')
    source = models.CharField(max_length=100, choices=KEY_SOURCE_CHOICES, default='admin')
    integration_id = models.CharField(max_length=100, blank=True)

    objects = KeyManager()

    @property
    def survey(self):
        return self.bundle.survey

    def deactivate_with_answer(self, answer):
        # todo: test me
        self.date_deactivated = timezone.now()
        self.deactivated_by_answer = answer
        self.is_active = False
        self.save()

    def get_frontend_url(self):
        # todo: test me
        base_url = self.bundle.survey_frontend_base_url or self.bundle.survey.get_frontend_url(
            is_with_site=True,
            external=True
        )
        return '{url}?key={key}'.format(
            url=base_url,
            key=self.value
        )

    class Meta:
        ordering = ('-date_created',)
        unique_together = (
            ('bundle', 'value'),
        )
