# -*- coding: utf-8 -*-
import uuid

from django.conf import settings
from django.db import transaction

from events.surveyme.models import ProfileSurveyAnswer
from events.takeout.models import TakeoutOperation

MAX_UPDATE_SIZE = 100


def relink_user_answers(user_id, new_user_id, max_update_size=MAX_UPDATE_SIZE):
    while True:
        queryset = ProfileSurveyAnswer.objects.filter(
            pk__in=ProfileSurveyAnswer.objects.filter(user_id=user_id)[:max_update_size]
        )
        with transaction.atomic():
            rs = queryset.update(user_id=new_user_id)
        if rs == 0:
            break


def create_operation(user_id, task_id, **params):
    return TakeoutOperation.objects.create(user_id=user_id, task_id=task_id, **params)


def change_operation_status(task_id, status, error=None):
    TakeoutOperation.objects.filter(task_id=task_id).update(status=status, error=error)


class TakeoutDeleteAnswers:
    def __init__(self, user, request_id=None):
        self.user = user
        self.request_id = request_id

    def _get_last_operation(self):
        return (
            TakeoutOperation.objects.using(settings.DATABASE_ROLOCAL)
            .filter(user_id=self.user.pk).order_by('-pk').first()
        )

    def _check_if_answers_exist(self):
        return bool(
            ProfileSurveyAnswer.objects.using(settings.DATABASE_ROLOCAL)
            .filter(user_id=self.user.pk).values_list('pk').order_by()[:1]
        )

    def check(self):
        operation = self._get_last_operation()
        if operation:
            if operation.status == TakeoutOperation.ERROR:
                return {
                    'status': 'error',
                    'errors': [{'code': 'internal', 'message': operation.error}],
                }

        task = {'id': '1', 'slug': 'answers'}
        if self._check_if_answers_exist():
            task['state'] = 'ready_to_delete'
        else:
            task['state'] = 'empty'

        if operation:
            if operation.status == TakeoutOperation.SUCCESS:
                task['update_date'] = operation.created.strftime('%Y-%m-%dT%H:%M:%SZ')
            elif operation.status == TakeoutOperation.INPROGRESS:
                task['state'] = 'delete_in_progress'
        return {'status': 'ok', 'data': [task]}

    def delete(self):
        from events.takeout import tasks
        task_id = str(uuid.uuid4())
        args = (self.user.pk, settings.MOCK_PROFILE_ID)
        kwargs = {'request_id': self.request_id, 'task_id': task_id}
        tasks.relink_user_answers.apply_async(args, kwargs, task_id=task_id)
        return {'status': 'ok'}
