# -*- coding: utf-8 -*-
import logging

from django.db import transaction
from django.conf import settings
from django.contrib.contenttypes.models import ContentType

from events.celery_app import app
from events.surveyme.models import Survey
from events.tanker import utils
from events.tanker.client import TankerClient
from events.tanker.models import TankerKeyset

logger = logging.getLogger(__name__)


@app.task
def run_translation_process(survey_id, languages):
    client = TankerClient(settings.TANKER_PROJECT)
    utils.run_translation_process(client, survey_id, languages)


@app.task
def update_all_translations():
    client = TankerClient(settings.TANKER_PROJECT)
    for tk in utils.get_tankerkeysets_for_update(client):
        with transaction.atomic():
            try:
                locked_ts = TankerKeyset.objects.select_for_update(skip_locked=True).get(pk=tk.pk)
            except TankerKeyset.DoesNotExist:
                pass
            else:
                utils.update_translations(client, locked_ts)


@app.task
def update_translations(survey_id):
    client = TankerClient(settings.TANKER_PROJECT)
    ct = ContentType.objects.get_for_model(Survey)
    try:
        tk = TankerKeyset.objects.get(content_type=ct, object_id=survey_id)
    except TankerKeyset.DoesNotExist:
        logger.warning('Keyset for object %s was not found', survey_id)
    else:
        utils.update_translations(client, tk)
