# -*- coding: utf-8 -*-
import operator

from functools import reduce

from django import forms
from django.db import connection
from django.db.models import Q
from django.db.models.expressions import RawSQL
from django_filters import Filter
from django_filters.fields import Lookup
from django_filters.filters import EMPTY_VALUES

from events.common_app.utils import get_lang_with_fallback


class TranslationFilterMixin:
    def _get_languages(self):
        return [lang for lang in get_lang_with_fallback() if lang]

    def filter_postgresql_exact(self, qs, value):
        langs = self._get_languages()
        raw_sql = ' OR '.join(
            "(translations -> '{field}' ->> '{lang}' = %s)".format(field=self.name, lang=lang)
            for lang in langs
        )
        raw_args = (value,) * len(langs)
        return qs.annotate(_exact=RawSQL(raw_sql, raw_args)).filter(_exact=True)

    def filter_postgresql_startswith(self, qs, value):
        langs = self._get_languages()
        raw_sql = ' OR '.join(
            "(translations -> '{field}' ->> '{lang}' ILIKE %s)".format(field=self.name, lang=lang)
            for lang in langs
        )
        raw_args = (value + '%',) * len(langs)
        return qs.annotate(_startswith=RawSQL(raw_sql, raw_args)).filter(_startswith=True)

    def filter_general_exact(self, qs, value):
        langs = self._get_languages()
        q_list = [
            Q(translations__icontains='"{lang}":"{text}"'.format(lang=lang, text=value))
            for lang in langs
        ]
        if q_list:
            qs = qs.filter(reduce(operator.or_, q_list))
        return qs

    def filter_general_startswith(self, qs, value):
        langs = self._get_languages()
        q_list = [
            Q(translations__icontains='"{lang}":"{text}'.format(lang=lang, text=value))
            for lang in langs
        ]
        if q_list:
            qs = qs.filter(reduce(operator.or_, q_list))
        return qs

    def filter(self, qs, value):
        if isinstance(value, Lookup):
            lookup = value.lookup_type
            value = value.value
        else:
            lookup = self.lookup_expr

        if value in EMPTY_VALUES:
            return qs

        vendor = 'postgresql' if connection.vendor == 'postgresql' else 'general'
        method_name = 'filter_{vendor}_{lookup}'.format(vendor=vendor, lookup=lookup)
        method = getattr(self, method_name, None)
        if method:
            qs = method(qs, value)

        return qs


class TranslationCharFilter(TranslationFilterMixin, Filter):
    field_class = forms.CharField
