# -*- coding: utf-8 -*-
from ninja.errors import HttpError as BaseHttpError

from events.v3.types import ErrorType
from events.v3.schemas import ErrorOut


class HttpError(BaseHttpError):
    status_code = None
    message = None

    def __init__(self, message: str = None):
        message = message or self.message
        super().__init__(status_code=self.status_code, message=message)


class BadRequest(HttpError):
    status_code = 400
    message = 'Bad request'


class AuthenticationRequired(HttpError):
    status_code = 401
    message = 'Authentication required'


class PermissionDenied(HttpError):
    status_code = 403
    message = 'Permission denied'


class ObjectNotFound(HttpError):
    status_code = 404
    message = 'Not found'


class ValidationError(Exception):
    def __init__(self, errors):
        self.errors = []
        if isinstance(errors, list):
            for error in errors:
                if isinstance(error, ErrorOut):
                    self.errors.append(error)
                elif isinstance(error, ValidationError):
                    self.errors.extend(error.errors)
        elif isinstance(errors, ErrorOut):
            self.errors = [errors]
        elif isinstance(errors, ValidationError):
            self.errors.extend(errors.errors)


class NotFoundError(ValidationError):
    msg = 'does not exist'

    def __init__(self, loc, value):
        error = ErrorOut(loc=loc, error_code=ErrorType.not_found, msg=self.msg, value=value)
        super().__init__(error)


class DuplicatedError(ValidationError):
    msg = 'duplicate value'

    def __init__(self, loc, value):
        error = ErrorOut(loc=loc, error_code=ErrorType.duplicated, msg=self.msg, value=value)
        super().__init__(error)


class MissingError(ValidationError):
    msg = 'field required'

    def __init__(self, loc):
        error = ErrorOut(loc=loc, error_code=ErrorType.missing, msg=self.msg)
        super().__init__(error)


class NotPermittedError(ValidationError):
    msg = 'does not have permission'

    def __init__(self, loc, value):
        error = ErrorOut(loc=loc, error_code=ErrorType.not_permitted, msg=self.msg, value=value)
        super().__init__(error)


class InvalidError(ValidationError):
    msg = 'invalid format'

    def __init__(self, loc, value):
        error = ErrorOut(loc=loc, error_code=ErrorType.invalid, msg=self.msg, value=value)
        super().__init__(error)
