# -*- coding: utf-8 -*-
from django.conf import settings
from guardian.shortcuts import get_objects_for_user
from typing import List, Union

from events.accounts.models import User
from events.surveyme.models import Survey, SurveyGroup
from events.v3.errors import PermissionDenied
from events.v3.schemas import PermissionIn, PermissionOut, UserOut, GroupOut
from events.v3.types import PermissionAccessType, PermissionActionType

CHANGE_SURVEY = 'surveyme.change_survey'
CHANGE_SURVEYGROUP = 'surveyme.change_surveygroup'


def logged_user(request):
    if not request.user.is_anonymous:
        return request.user


def has_perm(user, obj):
    if isinstance(obj, Survey):
        result = user.has_perm(settings.ROLE_FORM_MANAGER, obj)
        if not result and obj.group:
            result = has_perm(user, obj.group)
    elif isinstance(obj, SurveyGroup):
        result = user.has_perm(settings.ROLE_GROUP_MANAGER, obj)
    else:
        result = False
    return result


def check_perm(user, obj):
    if not has_perm(user, obj):
        raise PermissionDenied()


def _get_objects_for_user(user: User, permission: str, qs):
    return get_objects_for_user(
        user, permission, klass=qs,
        use_groups=True, any_perm=True,
        with_superuser=False, accept_global_perms=False,
    )


def filter_permitted_surveys(user: User, orgs: List[str], qs):  # qs: QuerySet[Survey]
    if user.is_superuser:
        return qs
    if settings.IS_BUSINESS_SITE:
        user_qs = qs.filter(org_id__isnull=True, user=user)
        if orgs:
            survey_qs = _get_objects_for_user(
                user, CHANGE_SURVEY,
                qs.filter(org__dir_id__in=orgs),
            )
            group_qs = _get_objects_for_user(
                user, CHANGE_SURVEYGROUP,
                SurveyGroup.objects.filter(org__dir_id__in=orgs),
            )
            qs = user_qs | survey_qs | qs.filter(group__in=group_qs)
        else:
            qs = user_qs
    else:
        survey_qs = _get_objects_for_user(user, CHANGE_SURVEY, qs)
        group_qs = _get_objects_for_user(user, CHANGE_SURVEYGROUP, None)
        qs = survey_qs | qs.filter(group__in=group_qs)
    return qs


def get_action_type(codename: str) -> PermissionActionType:
    pos = codename.find('_')
    if pos != -1:
        return PermissionActionType(codename[:pos])
    return PermissionActionType.change


def get_permission_out(codename: str, perminfo) -> PermissionOut:
    params = {
        'action': get_action_type(codename),
        'access': perminfo['type'],
    }
    if params['access'] == PermissionAccessType.restricted:
        users = [
            UserOut(
                id=user.pk, uid=user.uid, cloud_uid=user.cloud_uid,
                login=user.get_yandex_username(), display=user.get_name_and_surname_with_fallback(),
            )
            for user in perminfo.get('users') or []
        ]
        groups = [
            GroupOut(id=group.pk, name=group.name, url=group.get_info_url())
            for group in perminfo.get('groups') or []
        ]
        if not users and not groups:
            params['access'] = PermissionAccessType.owner
        if users:
            params['users'] = users
        if groups:
            params['groups'] = groups
    return PermissionOut(**params)


def get_permissions_out(obj) -> List[PermissionOut]:
    from events.surveyme.logic.access import get_survey_access
    result = get_survey_access(obj, optimize=False)
    return [
        get_permission_out(codename, perminfo)
        for (codename, perminfo) in result.items()
    ]


def change_permissions(user: User, obj: Union[Survey, SurveyGroup], data: List[PermissionIn]):
    from events.surveyme.logic.access import set_survey_access
    for it in data:
        set_survey_access(
            user, obj, access_type=it.access, action_type=it.action,
            users=it.users, groups=it.groups,
        )
