# -*- coding: utf-8 -*-
from ninja import Router
from ninja.constants import NOT_SET
from functools import wraps
from typing import List, Callable
from ylog.context import put_to_context


class LogContext:
    default_value = '-'

    def __init__(self, endpoint: bool = True, auth: bool = True, user: bool = True):
        self.endpoint = endpoint
        self.auth = auth
        self.user = user

    def get_endpoint(self, func, request):
        return {
            'name': func.__qualname__,
            'view_name': self.default_value,
        }

    def get_user(self, func, request):
        return {
            'uid': str(request.yauser.uid) if request.yauser.uid else self.default_value,
            'cloud_uid': getattr(request.yauser, 'cloud_uid', self.default_value),
            'orgs': ','.join(request.orgs) if request.orgs else self.default_value,
        }

    def get_auth(self, func, request):
        if not request.yauser.is_authenticated():
            mechanism = self.default_value
        else:
            mechanism = getattr(request.yauser.authenticated_by, 'mechanism_name', self.default_value)
        return {
            'mechanism': mechanism,
        }

    def put_log_context(self, func, request):
        if self.endpoint:
            put_to_context('endpoint', self.get_endpoint(func, request))
        if self.auth:
            put_to_context('auth', self.get_auth(func, request))
        if self.user:
            put_to_context('user', self.get_user(func, request))

    def __call__(self, func):
        @wraps(func)
        def wrapper(request, *args, **kwargs):
            self.put_log_context(func, request)
            return func(request, *args, **kwargs)
        return wrapper


class ContextRouter(Router):
    def add_api_operation(self, path: str, methods: List[str], view_func: Callable, **kwargs):
        auth = kwargs.get('auth')
        has_auth = auth and auth != NOT_SET
        view_func = LogContext(auth=has_auth, user=has_auth)(view_func)
        super().add_api_operation(path, methods, view_func, **kwargs)
