# -*- coding: utf-8 -*-
import re

from bson import ObjectId
from django.conf import settings
from enum import IntEnum, Enum
from pydantic import SetMaxLengthError


class StrEnum(str, Enum):
    pass


class StringBasedType(str):
    string_pattern = None
    string_re = None
    can_be_empty = False
    max_length = None

    @classmethod
    def __get_validators__(cls):
        yield cls.validate

    @classmethod
    def __modify_schema__(cls, field_schema):
        field_schema.update(pattern=cls.string_pattern)
        if cls.max_length:
            field_schema.update(maxLength=cls.max_length)

    @classmethod
    def validate(cls, v):
        if not isinstance(v, str):
            raise TypeError('string required')
        if cls.can_be_empty and not v:
            return v
        m = cls.string_re.fullmatch(v)
        if not m:
            raise ValueError('invalid format')
        if cls.max_length and len(v) > cls.max_length:
            raise SetMaxLengthError(limit_value=cls.max_length)
        return cls(v)


class StringId(StringBasedType):
    string_pattern = r'^[a-fA-F\d]{24}$'
    string_re = re.compile(string_pattern)

    @classmethod
    def create(cls) -> 'StringId':
        return StringId(ObjectId())


class TrackerQueue(StringBasedType):
    string_pattern = r'^[a-zA-Z][a-zA-Z\d]+$'
    string_re = re.compile(string_pattern)
    max_length = 255


class TrackerIssue(StringBasedType):
    string_pattern = r'^[a-zA-Z][a-zA-Z\d]+-\d+$'
    string_re = re.compile(string_pattern)
    can_be_empty = True
    max_length = 255


if settings.IS_BUSINESS_SITE:
    SurveyId = StringId
else:
    SurveyId = int


VariableId = StringId


class ActionType(IntEnum):
    email = 3
    tracker = 8
    wiki = 9
    jsonrpc = 6
    post = 4
    put = 10
    http = 11


class SubscriptionType(StrEnum):
    email = 'email'
    tracker = 'tracker'
    wiki = 'wiki'
    jsonrpc = 'jsonrpc'
    post = 'post'
    put = 'put'
    http = 'http'


class PostPutFormatType(StrEnum):
    json = 'json'
    xml = 'xml'


class HttpMethodType(StrEnum):
    get = 'get'
    post = 'post'
    patch = 'patch'
    put = 'put'
    delete = 'delete'


class OperatorType(StrEnum):
    and_ = 'and'
    or_ = 'or'


class ConditionType(StrEnum):
    eq = 'eq'
    neq = 'neq'
    lt = 'lt'
    gt = 'gt'


class ConditionItemType(StrEnum):
    question = 'question'
    language = 'language'
    origin = 'origin'


class ErrorType(StrEnum):
    not_found = 'value_error.not_found'
    missing = 'value_error.missing'
    not_permitted = 'value_error.not_permitted'
    duplicated = 'value_error.duplicated'
    invalid = 'value_error'


_languages = {x: x for x, _ in settings.LANGUAGES}
_languages['from_request'] = 'from_request'
LanguageType = StrEnum('LanguageType', _languages)


class NotificationType(StrEnum):
    pending = 'pending'
    success = 'success'
    error = 'error'
    canceled = 'canceled'


class NotificationFieldType(StrEnum):
    text = 'text'
    textarea = 'textarea'
    code = 'code'
    json = 'json'
    xml = 'xml'
    url = 'url'


class OrderingType(StrEnum):
    asc = 'asc'
    desc = 'desc'


class ResultType(StrEnum):
    ok = 'ok'
    fail = 'fail'
    operation = 'operation'
    skip = 'skip'


class OperationStatus(StrEnum):
    ok = 'ok'
    fail = 'fail'
    wait = 'wait'
    not_running = 'not_running'


class LayerType(StrEnum):
    survey = 'survey'
    hook = 'hook'
    subscription = 'subscription'


class UnsubscribeStatusType(StrEnum):
    unsubscribed = 'unsubscribed'
    not_subscribed = 'not_subscribed'


class PermissionActionType(StrEnum):
    change = 'change'
    viewfile = 'viewfile'


class PermissionAccessType(StrEnum):
    owner = 'owner'
    restricted = 'restricted'
    common = 'common'
