# -*- coding: utf-8 -*-
from collections import defaultdict
from django.db.models import Q
from lazy_object_proxy import Proxy as LazyProxy
from typing import List, Dict

from events.conditions.models import ContentTypeAttribute
from events.surveyme.models import Survey
from events.surveyme_integration.models import SurveyHookConditionNode, SurveyHookCondition, SurveyHook
from events.v3.errors import ObjectNotFound, MissingError, NotFoundError, InvalidError
from events.v3.perms import check_perm
from events.v3.schemas import ConditionItemIn, ConditionItemOut, ConditionIn, ConditionOut
from events.v3.types import SurveyId, ConditionItemType
from events.v3.utils import set_master_db, get_from_slave, get_from_master, get_from_db, get_survey
from events.v3.views.subscriptions import get_question_slugs_with_type


def get_condition_item_type(attr, lookup_field) -> ConditionItemType:
    if attr.startswith('answer_'):
        return ConditionItemType.question
    elif attr == 'source_request':
        if lookup_field == 'accept-language':
            return ConditionItemType.language
        elif lookup_field == 'parent_origin':
            return ConditionItemType.origin


def get_condition_item_out(item) -> ConditionItemOut:
    type_ = get_condition_item_type(
        item.content_type_attribute__attr,
        item.content_type_attribute__lookup_field,
    )
    if type_:
        return ConditionItemOut(
            operator=item.operator,
            condition=item.condition,
            value=item.value,
            type=type_,
            question=item.survey_question__param_slug,
        )


def get_condition_out(node, condition_items) -> ConditionOut:
    return ConditionOut(
        id=node.pk,
        operator='or',
        items=condition_items.get(node.pk),
    )


def get_one_condition_out(survey: Survey, hook_id: int, condition_id: int) -> ConditionOut:
    fields = ('pk', 'hook_id')
    qs = (
        get_from_db(SurveyHookConditionNode, 'default')
        .filter(hook__survey=survey, hook_id=hook_id, pk=condition_id)
        .values_list(*fields, named=True)
    )
    condition_items = LazyProxy(lambda: get_condition_items_out(survey, hook_id))
    for it in qs:
        return get_condition_out(it, condition_items)
    raise ObjectNotFound


def get_conditions_out(survey: Survey, hook_id: int = None) -> Dict[int, List[ConditionOut]]:
    fields = ('pk', 'hook_id')
    qs = get_from_db(SurveyHookConditionNode, 'default').filter(hook__survey=survey)
    if hook_id:
        qs = qs.filter(hook_id=hook_id)
    qs = qs.order_by('position', 'pk').values_list(*fields, named=True)
    condition_items = LazyProxy(lambda: get_condition_items_out(survey, hook_id))
    out = defaultdict(list)
    for it in qs:
        out[it.hook_id].append(get_condition_out(it, condition_items))
    return out


def get_condition_items_out(survey: Survey, hook_id: int = None) -> Dict[int, List[ConditionItemOut]]:
    fields = (
        'pk', 'condition', 'operator', 'value', 'condition_node_id', 'survey_question__param_slug',
        'content_type_attribute__attr', 'content_type_attribute__lookup_field',
    )
    qs = (
        get_from_db(SurveyHookCondition, 'default')
        .filter(condition_node__hook__survey=survey)
        .filter(Q(survey_question_id__isnull=True) | Q(survey_question__is_deleted=False))
    )
    if hook_id:
        qs = qs.filter(condition_node__hook_id=hook_id)
    qs = qs.order_by('position', 'pk').values_list(*fields, named=True)
    out = defaultdict(list)
    for it in qs:
        condition_item = get_condition_item_out(it)
        if condition_item:
            out[it.condition_node_id].append(condition_item)
    return out


def get_cta():
    fields = ('pk', 'attr', 'lookup_field')
    cta = {}
    for it in get_from_slave(ContentTypeAttribute).values_list(*fields, named=True):
        if it.attr.startswith('answer_'):
            cta[it.attr] = it.pk
        elif it.attr == 'source_request':
            if it.lookup_field == 'accept-language':
                cta['language'] = it.pk
            elif it.lookup_field == 'parent_origin':
                cta['origin'] = it.pk
    return cta


def get_condition_items_obj(
    survey: Survey,
    items: List[ConditionItemIn],
    condition_id: int = None,
) -> List[SurveyHookCondition]:
    condition_items = []
    if items:
        cta = get_cta()
        questions = get_question_slugs_with_type(survey)
        position = 1
        for item in items:
            params = {}
            if condition_id:
                params['condition_node_id'] = condition_id
            if item.type == 'question':
                if item.question is None:
                    raise MissingError(loc=['items', 'question'])
                elif item.question not in questions:
                    raise NotFoundError(loc=['items', 'question'], value=item.question)
                question_id, attr = questions[item.question]
                if attr == 'answer_date':
                    attr = 'answer_date.date_start'
                if attr not in cta:
                    raise InvalidError(loc=['items', 'type'], value=attr)
                params['survey_question_id'] = question_id
                params['content_type_attribute_id'] = cta[attr]
            else:
                if item.type not in cta:
                    raise InvalidError(loc=['items', 'type'], value=item.type)
                params['content_type_attribute_id'] = cta[item.type]
            params['operator'] = item.operator
            params['condition'] = item.condition
            params['value'] = item.value
            params['position'] = position
            condition_items.append(SurveyHookCondition(**params))
            position += 1
    return condition_items


def create_condition(survey: Survey, hook_id: int, data: ConditionIn) -> int:
    hook = get_from_slave(SurveyHook).get(survey=survey, pk=hook_id)
    condition_items = get_condition_items_obj(survey, items=data.items)
    condition = SurveyHookConditionNode.objects.create(hook=hook)
    if condition_items:
        for item in condition_items:
            item.condition_node = condition
        SurveyHookCondition.objects.bulk_create(condition_items)
    return condition.pk


def modify_condition(survey: Survey, hook_id: int, condition_id: int, data: ConditionIn) -> int:
    condition = (
        get_from_slave(SurveyHookConditionNode, 'default')
        .get(hook__survey=survey, hook_id=hook_id, pk=condition_id)
    )
    qs = (
        get_from_master(SurveyHookCondition, 'default')
        .filter(condition_node_id=condition.pk)
    )
    qs.delete()
    condition_items = get_condition_items_obj(survey, condition_id=condition.pk, items=data.items)
    if condition_items:
        SurveyHookCondition.objects.bulk_create(condition_items)
    return condition.pk


def delete_condition(survey: Survey, hook_id: int, condition_id: int) -> None:
    condition = (
        get_from_slave(SurveyHookConditionNode, 'default')
        .get(hook__survey=survey, hook_id=hook_id, pk=condition_id)
    )
    condition.delete()


def get_conditions_view(
    request,
    survey_id: SurveyId, hook_id: int,
) -> List[ConditionOut]:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    out = get_conditions_out(survey, hook_id)
    return out.get(hook_id) or []


def get_condition_view(
    request,
    survey_id: SurveyId, hook_id: int, condition_id: int,
) -> ConditionOut:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return get_one_condition_out(survey, hook_id, condition_id)


def post_condition_view(
    request,
    survey_id: SurveyId, hook_id: int,
    data: ConditionIn,
) -> ConditionOut:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    with set_master_db():
        return get_one_condition_out(survey, hook_id, create_condition(survey, hook_id, data))


def patch_condition_view(
    request,
    survey_id: SurveyId, hook_id: int, condition_id: int,
    data: ConditionIn,
) -> ConditionOut:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    with set_master_db():
        return get_one_condition_out(survey, hook_id, modify_condition(survey, hook_id, condition_id, data))


def delete_condition_view(
    request,
    survey_id: SurveyId, hook_id: int, condition_id: int,
) -> None:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    delete_condition(survey, hook_id, condition_id)
