# -*- coding: utf-8 -*-
import json
import re

from enum import IntEnum
from datetime import datetime
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db import transaction
from django.db.models import Q, F, Max
from django.db.models.expressions import Func
from django.utils import timezone
from django.utils.translation import ugettext as _
from ninja import Query
from typing import List, Dict, Tuple
from urllib.parse import urljoin

from events.accounts.models import User
from events.common_app.utils import flatlist
from events.common_app.startrek.client import get_startrek_client
from events.history.models import HistoryRawEntry
from events.surveyme.models import Survey
from events.surveyme_integration.models import (
    HookSubscriptionNotification,
    HookSubscriptionNotificationCounter,
)
from events.surveyme_integration.models import StartrekSubscriptionData
from events.surveyme_integration.tasks import send_notification
from events.v3.perms import has_perm, check_perm
from events.v3.pagination import get_pagination_out, DEFAULT_PAGE_SIZE
from events.v3.schemas import (
    DetailedNotificationOut, NotificationOut, ListNotificationOut, NotificationFieldOut,
    NotificationsIn, RestartNotificationOut, StatusNotificationOut,
    ResultOut, ResultOkOut, ResultSkipOut, ResultOperationOut,
)
from events.v3.types import (
    SurveyId, NotificationType, NotificationFieldType,
    OrderingType, ResultType, SubscriptionType,
)
from events.v3.utils import (
    get_from_slave,
    get_from_master,
    get_survey,
    get_subscription_type_by_action_id,
    get_action_ids_by_subscription_type,
)


class GetNotificationComment(Func):
    function = 'GET_NOTIFICATION_COMMENT'

    def as_sqlite(self, compiler, connection):
        return ('NULL', [])


def get_notification(notification_id: int) -> HookSubscriptionNotification:
    return (
        get_from_slave(HookSubscriptionNotification)
        .select_related(
            'subscription',
            'subscription__survey_hook',
            'survey', 'survey__org',
        )
        .annotate(comment=GetNotificationComment(F('context'), F('response')))
        .only(
            'id', 'status', 'date_created', 'date_finished', 'context', 'response', 'error',
            'survey_id', 'survey__name', 'survey__org__dir_id',
            'answer_id', 'user_id', 'subscription_id',
            'subscription__survey_hook__id', 'subscription__survey_hook__name',
            'subscription__service_type_action_id', 'subscription__http_format_name',
        )
        .get(pk=notification_id)
    )


def get_notification_status(notification_id: int) -> Tuple:
    return (
        get_from_slave(HookSubscriptionNotification)
        .values_list('pk', 'status', 'survey_id', named=True)
        .get(pk=notification_id)
    )


def get_notifications(
    user: User, orgs: List[str],
    survey_id: SurveyId = None, hook_id: int = None, subscription_id: int = None,
    answer_id: int = None, status: List[NotificationType] = None,
    created_lt: datetime = None, created_gt: datetime = None,
    finished_lt: datetime = None, finished_gt: datetime = None,
    visible: bool = None, type: SubscriptionType = None,
    id: int = None, page_size: int = None, ordering: OrderingType = OrderingType.asc,
) -> List[HookSubscriptionNotification]:
    qs = (
        get_from_slave(HookSubscriptionNotification)
        .select_related(
            'subscription',
            'subscription__survey_hook',
            'survey',
        )
        .annotate(comment=GetNotificationComment(F('context'), F('response')))
        .only(
            'id', 'status', 'date_created', 'date_finished',
            'survey_id', 'survey__name',
            'answer_id', 'user_id', 'subscription_id',
            'subscription__survey_hook__id', 'subscription__survey_hook__name',
            'subscription__service_type_action_id',
        )
    )
    if survey_id:
        qs = qs.filter(survey_id=survey_id)
    if hook_id:
        qs = qs.filter(subscription__survey_hook_id=hook_id)
    if subscription_id:
        qs = qs.filter(subscription_id=subscription_id)
    if answer_id:
        qs = qs.filter(answer_id=answer_id)
    if status:
        qs = qs.filter(status__in=status)
    if created_lt:
        qs = qs.filter(date_created__lt=created_lt)
    if created_gt:
        qs = qs.filter(date_created__gt=created_gt)
    if finished_lt:
        qs = qs.filter(date_finished__lt=finished_lt)
    if finished_gt:
        qs = qs.filter(date_finished__gt=finished_gt)
    if visible is not None:
        qs = qs.filter(is_visible=visible)
    if type:
        action_ids = get_action_ids_by_subscription_type(type)
        qs = qs.filter(subscription__service_type_action_id__in=action_ids)
    if ordering == OrderingType.desc:
        if id:
            qs = qs.filter(pk__lt=id)
        qs = qs.order_by('-id')
    else:
        if id:
            qs = qs.filter(pk__gt=id)
        qs = qs.order_by('id')
    if not page_size:
        return qs
    return qs[:page_size]


def get_notification_out(notification: HookSubscriptionNotification) -> NotificationOut:
    params = {
        'id': notification.pk,
        'status': notification.status,
        'created': notification.date_created,
        'finished': notification.date_finished,
        'answer_id': notification.answer_id,
        'user_id': notification.user_id,
        'comment': notification.comment,
    }
    if notification.survey:
        params['survey_id'] = notification.survey_id
        params['survey_name'] = notification.survey.name
    if notification.subscription:
        params['subscription_id'] = notification.subscription_id
        params['type'] = get_subscription_type_by_action_id(notification.subscription.service_type_action_id)
        params['hook_id'] = notification.subscription.survey_hook.id
        params['hook_name'] = notification.subscription.survey_hook.name
    return NotificationOut(**params)


def notification_text_field(name, value) -> NotificationFieldOut:
    return NotificationFieldOut(name=name, value=value, type=NotificationFieldType.text)


def notification_textarea_field(name, value) -> NotificationFieldOut:
    return NotificationFieldOut(name=name, value=value, type=NotificationFieldType.textarea)


def notification_code_field(name, value) -> NotificationFieldOut:
    return NotificationFieldOut(name=name, value=value, type=NotificationFieldType.code)


def notification_json_field(name, value) -> NotificationFieldOut:
    return NotificationFieldOut(name=name, value=value, type=NotificationFieldType.json)


def notification_xml_field(name, value) -> NotificationFieldOut:
    return NotificationFieldOut(name=name, value=value, type=NotificationFieldType.xml)


def notification_url_field(name, value) -> NotificationFieldOut:
    return NotificationFieldOut(name=name, value=value, type=NotificationFieldType.url)


class NotificationAcl(IntEnum):
    none = 0
    business_user = 1
    intranet_user = 2
    support = 4
    superuser = 8


class NotificationField:
    acl = NotificationAcl.business_user + NotificationAcl.intranet_user

    def __init__(self, user: User, notification: HookSubscriptionNotification, **kwargs):
        self.user = user
        self.notification = notification
        self.kwargs = kwargs

    def has_perm(self) -> bool:
        user_acl = NotificationAcl.none
        if settings.IS_BUSINESS_SITE:
            user_acl += NotificationAcl.business_user
        else:
            user_acl += NotificationAcl.intranet_user
        if self.user.is_staff:
            user_acl += NotificationAcl.support
        if self.user.is_superuser:
            user_acl += NotificationAcl.superuser
        return bool(self.acl & user_acl)

    def get(self) -> NotificationFieldOut:
        if self.has_perm():
            return self._get()

    def _get(self) -> NotificationFieldOut:
        pass


class ErrorClassName(NotificationField):
    def _get(self) -> NotificationFieldOut:
        error = self.notification.error
        if not isinstance(error, dict):
            return
        value = error.get('classname')
        if value:
            value = value[value.rfind('.') + 1:]
            return notification_text_field(_('Класс ошибки'), value)


class ErrorMessage(NotificationField):
    def _get(self) -> NotificationFieldOut:
        error = self.notification.error
        if not isinstance(error, dict):
            return
        value = error.get('message')
        if value:
            return notification_text_field(_('Сообщение о ошибке'), value)


class ErrorTraceBack(NotificationField):
    acl = NotificationAcl.superuser

    def _get(self) -> NotificationFieldOut:
        error = self.notification.error
        if not isinstance(error, dict):
            return
        value = error.get('traceback')
        if value:
            return notification_code_field(_('Трейс'), value)


class ResponseStatusCode(NotificationField):
    acl = NotificationAcl.intranet_user + NotificationAcl.support + NotificationAcl.superuser

    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('status_code')
        if value:
            return notification_text_field(_('Статус операции'), str(value))


class ResponseMethod(NotificationField):
    acl = NotificationAcl.intranet_user + NotificationAcl.support + NotificationAcl.superuser

    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('method')
        if value:
            return notification_text_field(_('Метод'), value.upper())


class ResponseUrl(NotificationField):
    acl = NotificationAcl.intranet_user + NotificationAcl.support + NotificationAcl.superuser

    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('url')
        if value:
            return notification_text_field(_('Url'), value)


class ResponseContent(NotificationField):
    acl = NotificationAcl.intranet_user + NotificationAcl.support + NotificationAcl.superuser
    max_content_length = 1000

    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('content')
        if value:
            name = _('Ответ')
            try:
                value = json.dumps(json.loads(value), ensure_ascii=False, indent=2, sort_keys=True)
                return notification_json_field(name, value)
            except (json.JSONDecodeError, TypeError):
                if len(value) > self.max_content_length:
                    content_bound = (self.max_content_length - 50) // 2
                    value = f'{value[:content_bound]}\n...\n{value[-content_bound:]}'
                return notification_code_field(name, value)


class ContextAttachments(NotificationField):
    def _get(self) -> List[NotificationFieldOut]:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        result = []
        for attach in context.get('attachments') or []:
            url = attach.get('frontend_url')
            if url:
                result.append(url)
        for attach in context.get('static_attachments') or []:
            url = attach.get('frontend_url')
            if url:
                result.append(url)
        if result:
            name_format = _('Файл %(index)s')
            return [
                notification_url_field(name_format % {'index': index}, url)
                for (index, url) in enumerate(result, start=1)
            ]


class TrackerResponseIssue(NotificationField):
    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('issue') or {}
        if value:
            issue_key = value.get('key')
            if issue_key:
                issue_url = urljoin(settings.STARTREK_HOST, issue_key)
                return notification_url_field(_('Тикет'), issue_url)


class TrackerContextSummary(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('summary')
        if value:
            return notification_text_field(_('Заголовок'), value)


class TrackerContextDescription(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('description')
        if value:
            return notification_textarea_field(_('Описание'), value)


class TrackerContextQueue(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('queue')
        if value:
            return notification_text_field(_('Очередь'), value)


class TrackerContextParent(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('parent')
        if value:
            parent_url = urljoin(settings.STARTREK_HOST, value)
            return notification_url_field(_('Родительский тикет'), parent_url)


class TrackerContextIssueType(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('type')
        if value:
            dir_id = self.kwargs.get('dir_id')
            client = get_startrek_client(dir_id)
            out = client.get_issuetype(value)
            if out:
                value = out.get('name')
            return notification_text_field(_('Тип тикета'), value)


class TrackerContextPriority(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('priority')
        if value:
            dir_id = self.kwargs.get('dir_id')
            client = get_startrek_client(dir_id)
            out = client.get_priority(value)
            if out:
                value = out.get('name')
            return notification_text_field(_('Приоритет'), value)


class TrackerContextAuthor(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('author')
        if value:
            return notification_text_field(_('Автор'), value)


class TrackerContextAssignee(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('assignee')
        if value:
            return notification_text_field(_('Исполнитель'), value)


class TrackerContextFields(NotificationField):
    def _get(self) -> List[NotificationFieldOut]:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        fields = context.get('fields') or {}
        if fields:
            result = []
            tracker_fields = self._get_tracker_fields()
            for name, value in fields.items():
                name = tracker_fields.get(name, name)
                if isinstance(value, str):
                    result.append(notification_text_field(name, value))
                else:
                    if isinstance(value, list):
                        value = list(flatlist(value))
                    result.append(notification_json_field(name, json.dumps(value, ensure_ascii=False)))
            return result

    def _get_tracker_fields(self) -> Dict[str, str]:
        try:
            st = (
                get_from_slave(StartrekSubscriptionData)
                .only('fields')
                .get(subscription_id=self.notification.subscription_id)
            )
        except StartrekSubscriptionData.DoesNotExist:
            st_fields = []
        else:
            st_fields = st.fields or []
        fields = {}
        for field in st_fields:
            key = field.get('key') or {}
            if key:
                fields[key.get('id') or key.get('slug')] = key.get('name')
        return fields


class EmailResponseMessageId(NotificationField):
    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('message_id')
        if not value:
            try:
                content = json.loads(response.get('content'))
            except (json.JSONDecodeError, TypeError):
                content = None
            if isinstance(content, dict):
                value = (content.get('result') or {}).get('message_id')
        if value:
            return notification_text_field(_('Код сообщения'), value)


class EmailResponseTaskId(NotificationField):
    def _get(self) -> NotificationFieldOut:
        response = self.notification.response
        if not isinstance(response, dict):
            return
        value = response.get('task_id')
        if not value:
            try:
                content = json.loads(response.get('content'))
            except (json.JSONDecodeError, TypeError):
                content = None
            if isinstance(content, dict):
                value = (content.get('result') or {}).get('task_id')
        if value:
            return notification_text_field(_('Код операции'), value)


class EmailContextToAddress(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('to_address')
        if value:
            return notification_text_field(_('Кому'), value)


class EmailContextFromAddress(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('from_address')
        if value:
            return notification_text_field(_('От кого'), value)


class EmailContextSubject(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('subject')
        if value:
            return notification_text_field(_('Тема'), value)


class EmailContextBody(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('body')
        if value:
            return notification_textarea_field(_('Текст'), value)


class EmailContextReplyTo(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('headers') or {}
        if value:
            for header_name, header_value in value.items():
                if header_name.lower() == 'reply-to':
                    return notification_text_field(_('Обратный адрес'), header_value)


class EmailContextHeaders(NotificationField):
    acl = NotificationAcl.intranet_user + NotificationAcl.support + NotificationAcl.superuser

    def _get(self) -> List[NotificationFieldOut]:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('headers') or {}
        if value:
            header_format = _('Заголовок %(name)s')
            result = [
                notification_text_field(header_format % {'name': header_name}, header_value)
                for header_name, header_value in value.items()
                if header_name.lower() != 'reply-to'
            ]
            if result:
                return result


class WikiContextSupertag(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('supertag') or {}
        if value:
            value = urljoin(settings.WIKI_HOST, value)
            return notification_url_field(_('Адрес страницы'), value)


class WikiContextText(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('text') or {}
        if value:
            return notification_textarea_field(_('Текст'), value)


class JsonrpcContextMethod(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        body_data = context.get('body_data') or {}
        value = body_data.get('method')
        if value:
            return notification_text_field(_('JSON-RPC метод'), value)


class JsonrpcContextParams(NotificationField):
    def _get(self) -> List[NotificationFieldOut]:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        body_data = context.get('body_data') or {}
        value = body_data.get('params') or {}
        if value:
            param_format = _('Параметр %(name)s')
            result = [
                notification_text_field(param_format % {'name': param_name}, param_value)
                for param_name, param_value in value.items()
            ]
            if result:
                return result


class HttpContextUrl(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('url')
        if value:
            return notification_url_field(_('Url'), value)


class HttpContextMethod(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('method')
        if value:
            return notification_text_field(_('Метод'), value.upper())


class HttpContextBody(NotificationField):
    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('body_data')
        if value:
            name = _('Запрос')
            try:
                value = json.dumps(json.loads(value), ensure_ascii=False, indent=2, sort_keys=True)
                return notification_json_field(name, value)
            except (json.JSONDecodeError, TypeError):
                return notification_code_field(name, value)


class HttpContextTvm2ClientId(NotificationField):
    acl = NotificationAcl.intranet_user

    def _get(self) -> NotificationFieldOut:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('tvm2_client_id')
        if value:
            return notification_text_field(_('Код клиента TVM'), value)


class HttpContextHeaders(NotificationField):
    def _get(self) -> List[NotificationFieldOut]:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        value = context.get('headers') or {}
        if value:
            header_format = _('Заголовок %(name)s')
            result = [
                notification_text_field(
                    header_format % {'name': header_name},
                    self._sanitize(header_name, header_value),
                )
                for header_name, header_value in value.items()
            ]
            if result:
                return result

    def _sanitize(self, name, value) -> str:
        if name.lower() == 'authorization':
            value = re.sub(r'^\s*(\w+)\s+.*', r'\1 xxxxx', value)
        return value


class HttpResponseStatusCode(ResponseStatusCode):
    acl = NotificationAcl.intranet_user + NotificationAcl.business_user


class HttpResponseMethod(ResponseMethod):
    acl = NotificationAcl.intranet_user + NotificationAcl.business_user


class HttpResponseUrl(ResponseUrl):
    acl = NotificationAcl.intranet_user + NotificationAcl.business_user


class HttpResponseContent(ResponseContent):
    acl = NotificationAcl.intranet_user + NotificationAcl.business_user


class PostPutContextMethod(NotificationField):
    def _get(self) -> NotificationFieldOut:
        subscription = self.notification.subscription
        if subscription:
            value = get_subscription_type_by_action_id(subscription.service_type_action_id)
            if value in (SubscriptionType.post, SubscriptionType.put):
                return notification_text_field(_('Метод'), value.value.upper())


class PostPutContextFields(NotificationField):
    def _get_format(self) -> str:
        subscription = self.notification.subscription
        if subscription:
            return subscription.http_format_name

    def _get(self) -> List[NotificationFieldOut]:
        context = self.notification.context
        if not isinstance(context, dict):
            return
        body_data = context.get('body_data') or {}
        if body_data:
            result = []
            format = self._get_format()
            field_format = _('Поле %(index)s')
            for (index, value) in enumerate(body_data.values(), 1):
                if value:
                    name = field_format % {'index': index}
                    if format == 'json':
                        result.append(notification_json_field(name, value))
                    elif format == 'xml':
                        result.append(notification_xml_field(name, value))
                    else:
                        result.append(notification_code_field(name, value))
            if result:
                return result


class NotificationFields:
    context = []
    error_response = [
        ResponseStatusCode, ResponseMethod,
        ResponseUrl, ResponseContent,
    ]
    response = []
    error = [ErrorClassName, ErrorMessage, ErrorTraceBack]

    def __init__(self, user: User, notification: HookSubscriptionNotification):
        self.user = user
        self.notification = notification

    def _get_category(self, class_list, **kwargs):
        result = []
        for clazz in class_list:
            inst = clazz(self.user, self.notification, **kwargs)
            value = inst.get()
            if isinstance(value, list):
                result.extend(value)
            elif value:
                result.append(value)
        if result:
            return result

    def get_context(self, **kwargs) -> List[NotificationFieldOut]:
        return self._get_category(self.context, **kwargs)

    def get_response(self, **kwargs) -> List[NotificationFieldOut]:
        if self.notification.status != NotificationType.success:
            return self._get_category(self.error_response, **kwargs)
        return self._get_category(self.response, **kwargs)

    def get_error(self, **kwargs) -> List[NotificationFieldOut]:
        if self.notification.status != NotificationType.success:
            return self._get_category(self.error, **kwargs)


class TrackerNotificationFields(NotificationFields):
    context = [
        TrackerContextSummary, TrackerContextDescription,
        TrackerContextQueue, TrackerContextParent,
        TrackerContextIssueType, TrackerContextPriority,
        TrackerContextAuthor, TrackerContextAssignee,
        TrackerContextFields, ContextAttachments,
    ]
    response = [TrackerResponseIssue]


class EmailNotificationFields(NotificationFields):
    context = [
        EmailContextToAddress, EmailContextFromAddress, EmailContextReplyTo,
        EmailContextSubject, EmailContextBody, ContextAttachments,
        EmailContextHeaders,
    ]
    response = [EmailResponseMessageId, EmailResponseTaskId]


class WikiNotificationFields(NotificationFields):
    context = [
        WikiContextSupertag, WikiContextText,
    ]


class JsonrpcNotificationFields(NotificationFields):
    context = [
        HttpContextUrl, JsonrpcContextMethod, HttpContextTvm2ClientId,
        JsonrpcContextParams, HttpContextHeaders,
    ]
    response = [
        HttpResponseStatusCode, HttpResponseMethod, HttpResponseUrl,
        HttpResponseContent,
    ]


class HttpNotificationFields(NotificationFields):
    context = [
        HttpContextUrl, HttpContextMethod, HttpContextTvm2ClientId,
        HttpContextBody, HttpContextHeaders,
    ]
    response = [
        HttpResponseStatusCode, HttpResponseMethod, HttpResponseUrl,
        HttpResponseContent,
    ]


class PostPutNotificationFields(NotificationFields):
    context = [
        HttpContextUrl, PostPutContextMethod, HttpContextTvm2ClientId,
        PostPutContextFields, HttpContextHeaders,
    ]
    response = [
        HttpResponseStatusCode, HttpResponseMethod, HttpResponseUrl,
        HttpResponseContent,
    ]


def get_notification_fields(user: User, notification: HookSubscriptionNotification) -> NotificationFields:
    if not notification.subscription:
        return
    match get_subscription_type_by_action_id(notification.subscription.service_type_action_id):
        case SubscriptionType.tracker:
            return TrackerNotificationFields(user, notification)
        case SubscriptionType.email:
            return EmailNotificationFields(user, notification)
        case SubscriptionType.wiki:
            return WikiNotificationFields(user, notification)
        case SubscriptionType.jsonrpc:
            return JsonrpcNotificationFields(user, notification)
        case SubscriptionType.http:
            return HttpNotificationFields(user, notification)
        case SubscriptionType.post | SubscriptionType.put:
            return PostPutNotificationFields(user, notification)


def get_detailed_notification_out(
    user: User,
    notification: HookSubscriptionNotification,
) -> DetailedNotificationOut:
    params = {
        'id': notification.pk,
        'status': notification.status,
        'created': notification.date_created,
        'finished': notification.date_finished,
        'answer_id': notification.answer_id,
        'user_id': notification.user_id,
        'comment': notification.comment,
    }
    fields = get_notification_fields(user, notification)
    if fields:
        dir_id = None
        if notification.survey and notification.survey.org:
            dir_id = notification.survey.org.dir_id
        params['error'] = fields.get_error(dir_id=dir_id)
        params['context'] = fields.get_context(dir_id=dir_id)
        params['response'] = fields.get_response(dir_id=dir_id)
    if notification.survey:
        params['survey_id'] = notification.survey_id
        params['survey_name'] = notification.survey.name
    if notification.subscription:
        params['subscription_id'] = notification.subscription_id
        params['type'] = get_subscription_type_by_action_id(notification.subscription.service_type_action_id)
        params['hook_id'] = notification.subscription.survey_hook_id
        params['hook_name'] = notification.subscription.survey_hook.name
    return DetailedNotificationOut(**params)


def get_notifications_out(notifications: List[HookSubscriptionNotification]) -> List[NotificationOut]:
    for notification in notifications:
        yield get_notification_out(notification)


def hide_errors(survey: Survey, subscription_id: int) -> None:
    notifications_qs = (
        get_from_master(HookSubscriptionNotification)
        .filter(survey_id=survey.pk, status='error', is_visible=True)
    )
    counters_qs = (
        get_from_master(HookSubscriptionNotificationCounter)
        .filter(survey_id=survey.pk)
    )
    if subscription_id:
        notifications_qs = notifications_qs.filter(subscription_id=subscription_id)
        counters_qs = counters_qs.filter(subscription_id=subscription_id)
    with transaction.atomic():
        notifications_qs.update(is_visible=False)
        counters_qs.update(errors_count=0, date_updated=timezone.now())
    return ResultOkOut(status=ResultType.ok)


def get_visible_errors(survey: Survey) -> List[int]:
    qs = (
        get_from_slave(HookSubscriptionNotification)
        .filter(
            survey_id=survey.pk,
            status='error',
            is_visible=True,
            subscription_id__isnull=False,
        )
        .values_list('subscription_id', flat=True)
        .annotate(last_id=Max('id'))
    )
    return list(qs)


def get_notification_view(request, notification_id: int) -> DetailedNotificationOut:
    notification = get_notification(notification_id)
    survey = get_survey(notification.survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return get_detailed_notification_out(request.user, notification)


def find_survey(
    survey_id: SurveyId = None, hook_id: int = None,
    subscription_id: int = None, answer_id: int = None, only=None,
) -> Survey:
    qs = get_from_slave(Survey)
    if survey_id:
        qs = qs.filter(pk=survey_id)
    elif hook_id:
        qs = qs.filter(hooks__pk=hook_id)
    elif subscription_id:
        qs = qs.filter(hooks__subscriptions__pk=subscription_id)
    elif answer_id:
        qs = qs.filter(profilesurveyanswer__pk=answer_id)
    else:
        qs = Survey.objects.none()
    if only:
        qs = qs.only(*only)
    out = list(qs[:1])
    if not out:
        return None
    return out[0]


def get_notifications_view(
    request,
    survey_id: SurveyId = None, hook_id: int = None, subscription_id: int = None, answer_id: int = None,
    status: List[NotificationType] = Query(None), created_lt: datetime = None, created_gt: datetime = None,
    finished_lt: datetime = None, finished_gt: datetime = None, visible: bool = None,
    id: int = None, page_size: int = DEFAULT_PAGE_SIZE, ordering: OrderingType = OrderingType.asc,
    type: SubscriptionType = None,
) -> ListNotificationOut:
    result = []
    if survey_id or hook_id or subscription_id or answer_id:
        survey = find_survey(survey_id, hook_id, subscription_id, answer_id, only=['pk'])
        if survey and has_perm(request.user, survey):
            notifications = get_notifications(
                request.user, request.orgs,
                survey.pk, hook_id, subscription_id, answer_id, status,
                created_lt, created_gt, finished_lt, finished_gt, visible, type,
                id, page_size, ordering,
            )
            result = list(get_notifications_out(notifications))
    return get_pagination_out(request, result, page_size)


def hide_subscription_errors_view(
    request,
    survey_id: SurveyId, hook_id: int, subscription_id: int,
) -> ResultOut:
    user = request.user
    survey = get_survey(survey_id, user=user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return hide_errors(survey, subscription_id)


def hide_survey_errors_view(request, survey_id: SurveyId) -> ResultOut:
    user = request.user
    survey = get_survey(survey_id, user=user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return hide_errors(survey, None)


def get_visible_errors_view(request, survey_id: SurveyId) -> List[int]:
    user = request.user
    survey = get_survey(survey_id, user=user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return get_visible_errors(survey)


def restart_notification(notification: HookSubscriptionNotification) -> RestartNotificationOut:
    if notification.status in (NotificationType.error, NotificationType.canceled):
        notification.status = NotificationType.pending
        notification.save(update_fields=['status'])
        task = send_notification.delay(notification.pk)
        result = ResultOperationOut(status=ResultType.operation, operation_id=task.task_id)
    else:
        result = ResultSkipOut(status=ResultType.skip)
    return RestartNotificationOut(
        id=notification.pk, survey_id=notification.survey_id,
        subscription_id=notification.subscription_id, result=result,
    )


def _create_history_entry(request, obj):
    ct = ContentType.objects.get_for_model(obj)
    HistoryRawEntry.objects.create_raw_entry(request, ct, obj.pk)


def restart_notification_view(request, notification_id: int) -> RestartNotificationOut:
    notification = get_notification(notification_id)
    survey = get_survey(notification.survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    out = restart_notification(notification)
    if out.result.status != ResultType.skip:
        _create_history_entry(request, notification)
    return out


def _get_surveys(ids: List[int], user, orgs=None, only=None):  # QuerySet[Survey]
    if not ids:
        return Survey.objects.none()
    qs = (
        get_from_slave(Survey).select_related('org', 'group')
        .filter(hooksubscriptionnotification__pk__in=ids)
    )
    if settings.IS_BUSINESS_SITE and not user.is_superuser:
        conditions = Q(org_id__isnull=True, user=user)
        if orgs:
            conditions |= Q(org__dir_id__in=orgs)
        qs = qs.filter(conditions)
    if only:
        if 'org__dir_id' not in only:
            only.append('org__dir_id')
        if 'group__id' not in only:
            only.append('group__id')
        qs = qs.only(*only)
    return qs.distinct()


def _get_notifications(notification_ids: List[int], survey_ids: List[SurveyId], only=None):
    if not notification_ids or not survey_ids:
        return HookSubscriptionNotification.objects.none()
    only = only or ('pk', 'survey_id', 'subscription_id')
    qs = (
        get_from_slave(HookSubscriptionNotification)
        .filter(pk__in=notification_ids, survey_id__in=survey_ids)
    )
    return qs.only(*only)


def restart_notifications_view(request, data: NotificationsIn) -> List[RestartNotificationOut]:
    survey_ids = [
        survey.pk
        for survey in _get_surveys(data.ids, user=request.user, orgs=request.orgs, only=['pk'])
        if has_perm(request.user, survey)
    ]
    out = []
    for notification in _get_notifications(data.ids, survey_ids):
        _out = restart_notification(notification)
        if _out.result.status != ResultType.skip:
            _create_history_entry(request, notification)
        out.append(_out)
    return out


def status_notification_view(request, notification_id: int) -> StatusNotificationOut:
    notification = get_notification_status(notification_id)
    survey = get_survey(notification.survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return StatusNotificationOut(id=notification.pk, status=notification.status)


def status_notifications_view(request, data: NotificationsIn) -> List[StatusNotificationOut]:
    survey_ids = [
        survey.pk
        for survey in _get_surveys(data.ids, user=request.user, orgs=request.orgs, only=['pk'])
        if has_perm(request.user, survey)
    ]
    return [
        StatusNotificationOut(id=notification.pk, status=notification.status)
        for notification in _get_notifications(data.ids, survey_ids, only=('id', 'status'))
    ]
