# -*- coding: utf-8 -*-
import re

from collections import defaultdict
from django.conf import settings
from django.db import transaction
from django.db.utils import IntegrityError
from django.utils import timezone
from lazy_object_proxy import Proxy as LazyProxy
from typing import List, Union, Dict, Tuple

from events.abc.utils import has_role_tvm_or_form_manager
from events.accounts.models import User
from events.common_app.wiki import check_if_supertag_exist
from events.common_app.startrek.client import get_startrek_client
from events.common_storages.models import ProxyStorageModel
from events.common_storages.utils import get_mds_url
from events.surveyme.models import Survey, SurveyQuestion
from events.surveyme_integration.models import (
    JSONRPCSubscriptionData,
    JSONRPCSubscriptionParam,
    ServiceSurveyHookSubscription,
    StartrekSubscriptionData,
    SubscriptionAttachment,
    SubscriptionHeader,
    SurveyHook,
    SurveyVariable,
    WikiSubscriptionData,
)
from events.v3.errors import (
    ObjectNotFound, ValidationError, NotFoundError, DuplicatedError, MissingError,
    NotPermittedError, InvalidError,
)
from events.v3.perms import check_perm
from events.v3.types import SurveyId, ActionType, SubscriptionType, LanguageType
from events.v3.utils import (
    set_master_db, get_from_slave, get_from_db,
    json_loads, get_survey, get_subscription_type,
)
from events.v3.schemas import (
    QuestionsOut, VariableIn, VariableOut, VariablesOut, HeaderOut, ParamOut,
    TrackerFieldOut, QuestionAttachmentsOut, StaticFile, StaticAttachmentOut,
    TemplateAttachmentOut, AttachmentsOut, TrackerIn, TrackerExtraOut, TrackerOut,
    WikiIn, WikiExtraOut, WikiOut, EmailIn, EmailOut, JsonRpcIn, JsonRpcExtraOut,
    JsonRpcOut, PostIn, PutIn, PostOut, PutOut, HttpIn, HttpOut, SubscriptionIn,
    SubscriptionOut, TrackerFieldKeyIn, TrackerFieldKeyOut,
)


QuestionManyToMany = ServiceSurveyHookSubscription.questions.through


class TrackerValidator:
    def __init__(self, queue_name: str, dir_id: str):
        self.client = LazyProxy(lambda: get_startrek_client(dir_id=dir_id))
        self.queue_name = queue_name

    def validate_issue_type(self, issue_type: int) -> int:
        if issue_type == 0:
            queue_data = self.client.get_queue(self.queue_name)
            if queue_data:
                return int(queue_data['defaultType']['id'])
        for it in self.client.get_issuetypes(self.queue_name):
            if it['id'] == issue_type:
                return issue_type
        raise NotFoundError(loc=['issue_type'], value=issue_type)

    def validate_priority(self, priority: int) -> int:
        if priority == 0:
            queue_data = self.client.get_queue(self.queue_name)
            if queue_data:
                return int(queue_data['defaultPriority']['id'])
        for it in self.client.get_priorities():
            if it['id'] == priority:
                return priority
        raise NotFoundError(loc=['priority'], value=priority)

    def validate_queue(self, queue: str) -> str:
        if not queue:
            return queue
        elif bool(self.client.get_queue(queue)):
            return queue
        raise NotFoundError(loc=['queue'], value=queue)

    def validate_parent(self, parent: str) -> str:
        if not parent:
            return parent
        elif bool(self.client.get_issue(parent)):
            return parent
        raise NotFoundError(loc=['parent'], value=parent)

    def validate_fields(self, fields: List['TrackerFieldIn']) -> list:
        errors = []
        result = []
        for i, field in enumerate(fields):
            if isinstance(field.key, TrackerFieldKeyIn):
                field_key = field.key.slug
            else:
                field_key = field.key
            field_data = self.client.get_field(field_key)
            if field_data:
                result.append({
                    'key': self._get_field_key(field_data),
                    'value': field.value,
                    'add_only_with_value': field.only_with_value or False,
                })
            else:
                errors.append(NotFoundError(loc=['fields', i, 'key'], value=field.key))
        if errors:
            raise ValidationError(errors)
        return result

    def _get_field_type(self, schema):
        result = schema.get('type')
        if result == 'array':
            result = '/'.join((result, schema.get('items', 'string')))
        return result

    def _get_field_key(self, field):
        return {
            'name': field.get('name'),
            'slug': field.get('id') or field.get('key'),
            'type': self._get_field_type(field.get('schema', {})),
        }


def get_one_subscription_out(survey: Survey, hook_id: int, subscription_id: int) -> SubscriptionOut:
    fields = (
        'pk', 'survey_hook_id', 'is_active', 'context_language', 'follow_result',
        'title', 'body', 'is_all_questions',
        'http_url', 'http_method', 'http_format_name', 'tvm2_client_id',
        'email_to_address', 'email_from_title', 'email_from_address', 'email_spam_check',
        'service_type_action__slug', 'service_type_action__service_type__slug',
    )
    qs = (
        get_from_db(ServiceSurveyHookSubscription)
        .filter(survey_hook__survey=survey, survey_hook_id=hook_id, pk=subscription_id)
        .values_list(*fields, named=True)
    )
    variables = LazyProxy(lambda: get_variables_out(survey))
    params = {
        'variables': variables,
        'headers': LazyProxy(lambda: get_headers_out(survey)),
        'tracker_extras': LazyProxy(lambda: get_tracker_extras_out(survey)),
        'wiki_extras': LazyProxy(lambda: get_wiki_extras_out(survey)),
        'jsonrpc_extras': LazyProxy(lambda: get_jsonrpc_extras_out(survey)),
        'jsonrpc_params': LazyProxy(lambda: get_jsonrpc_params_out(survey)),
        'questions': LazyProxy(lambda: get_questions_out(survey)),
        'static_attachments': LazyProxy(lambda: get_static_attachments_out(survey)),
        'template_attachments': LazyProxy(lambda: get_template_attachments_out(survey, variables)),
    }
    for it in qs:
        return get_subscription_out(it, **params)
    raise ObjectNotFound


def get_subscriptions_out(survey: Survey, hook_id: int = None) -> Dict[int, List[SubscriptionOut]]:
    fields = (
        'pk', 'survey_hook_id', 'is_active', 'context_language', 'follow_result',
        'title', 'body', 'is_all_questions',
        'http_url', 'http_method', 'http_format_name', 'tvm2_client_id',
        'email_to_address', 'email_from_title', 'email_from_address', 'email_spam_check',
        'service_type_action__slug', 'service_type_action__service_type__slug',
    )
    qs = get_from_db(ServiceSurveyHookSubscription).filter(survey_hook__survey=survey)
    if hook_id:
        qs = qs.filter(survey_hook_id=hook_id)
    qs = qs.values_list(*fields, named=True)
    variables = LazyProxy(lambda: get_variables_out(survey))
    params = {
        'variables': variables,
        'headers': LazyProxy(lambda: get_headers_out(survey)),
        'tracker_extras': LazyProxy(lambda: get_tracker_extras_out(survey)),
        'wiki_extras': LazyProxy(lambda: get_wiki_extras_out(survey)),
        'jsonrpc_extras': LazyProxy(lambda: get_jsonrpc_extras_out(survey)),
        'jsonrpc_params': LazyProxy(lambda: get_jsonrpc_params_out(survey)),
        'questions': LazyProxy(lambda: get_questions_out(survey)),
        'static_attachments': LazyProxy(lambda: get_static_attachments_out(survey)),
        'template_attachments': LazyProxy(lambda: get_template_attachments_out(survey, variables)),
    }
    out = defaultdict(list)
    for it in qs:
        out[it.survey_hook_id].append(get_subscription_out(it, **params))
    return out


def get_subscription_out(
    subscription, variables, headers, tracker_extras,
    wiki_extras, jsonrpc_extras, jsonrpc_params,
    questions, static_attachments, template_attachments,
) -> SubscriptionOut:
    service_slug = subscription.service_type_action__service_type__slug
    action_slug = subscription.service_type_action__slug
    type_ = get_subscription_type(service_slug, action_slug)
    out = None
    if type_ == SubscriptionType.tracker:
        out = get_tracker_out(subscription, variables, tracker_extras, questions, static_attachments)
    elif type_ == SubscriptionType.email:
        out = get_email_out(
            subscription, variables, headers, questions,
            static_attachments, template_attachments,
        )
    elif type_ == SubscriptionType.wiki:
        out = get_wiki_out(subscription, variables, wiki_extras)
    elif type_ == SubscriptionType.jsonrpc:
        out = get_jsonrpc_out(subscription, variables, jsonrpc_extras, jsonrpc_params, headers)
    elif type_ == SubscriptionType.post:
        out = get_post_out(subscription, variables, headers, questions)
    elif type_ == SubscriptionType.put:
        out = get_put_out(subscription, variables, headers, questions)
    elif type_ == SubscriptionType.http:
        out = get_http_out(subscription, variables, headers)
    return out


def get_http_out(subscription, variables, headers) -> HttpOut:
    return HttpOut(
        id=subscription.pk,
        active=subscription.is_active,
        type=SubscriptionType.http,
        follow=subscription.follow_result,
        language=subscription.context_language,
        variables=variables.get_subscription(subscription.pk),
        url=subscription.http_url,
        method=subscription.http_method,
        body=subscription.body,
        tvm_client=subscription.tvm2_client_id,
        headers=headers.get(subscription.pk),
    )


def get_put_out(subscription, variables, headers, questions) -> PutOut:
    return PutOut(
        id=subscription.pk,
        active=subscription.is_active,
        type=SubscriptionType.put,
        follow=subscription.follow_result,
        language=subscription.context_language,
        variables=variables.get_subscription(subscription.pk),
        url=subscription.http_url,
        format=subscription.http_format_name,
        tvm_client=subscription.tvm2_client_id,
        questions=get_questions_extra_out(subscription, questions),
        headers=headers.get(subscription.pk),
    )


def get_post_out(subscription, variables, headers, questions) -> PostOut:
    return PostOut(
        id=subscription.pk,
        active=subscription.is_active,
        type=SubscriptionType.post,
        follow=subscription.follow_result,
        language=subscription.context_language,
        variables=variables.get_subscription(subscription.pk),
        url=subscription.http_url,
        format=subscription.http_format_name,
        tvm_client=subscription.tvm2_client_id,
        questions=get_questions_extra_out(subscription, questions),
        headers=headers.get(subscription.pk),
    )


def get_jsonrpc_out(subscription, variables, jsonrpc_extras, jsonrpc_params, headers) -> JsonRpcOut:
    extra = jsonrpc_extras.get(subscription.pk)
    if extra:
        return JsonRpcOut(
            id=subscription.pk,
            active=subscription.is_active,
            type=SubscriptionType.jsonrpc,
            follow=subscription.follow_result,
            language=subscription.context_language,
            tvm_client=subscription.tvm2_client_id,
            variables=variables.get_subscription(subscription.pk),
            url=subscription.http_url,
            headers=headers.get(subscription.pk),
            params=jsonrpc_params.get(subscription.pk),
            **extra.dict(),
        )


def get_wiki_out(subscription, variables, wiki_extras) -> WikiOut:
    extra = wiki_extras.get(subscription.pk)
    if extra:
        return WikiOut(
            id=subscription.pk,
            active=subscription.is_active,
            type=SubscriptionType.wiki,
            follow=subscription.follow_result,
            language=subscription.context_language,
            variables=variables.get_subscription(subscription.pk),
            **extra.dict(),
        )


def get_email_out(
    subscription, variables, headers, questions,
    static_attachments, template_attachments,
) -> EmailOut:
    email_from_address = subscription.email_from_address
    email_spam_check = subscription.email_spam_check
    if settings.IS_BUSINESS_SITE:
        email_from_address = None
        email_spam_check = None
    return EmailOut(
        id=subscription.pk,
        active=subscription.is_active,
        type=SubscriptionType.email,
        follow=subscription.follow_result,
        language=subscription.context_language,
        variables=variables.get_subscription(subscription.pk),
        subject=subscription.title,
        body=subscription.body,
        email_to_address=subscription.email_to_address,
        email_from_address=email_from_address,
        email_from_title=subscription.email_from_title or None,
        email_spam_check=email_spam_check,
        headers=headers.get(subscription.pk),
        attachments=get_attachments_out(
            subscription, questions, static_attachments, template_attachments,
        ),
    )


def get_tracker_out(subscription, variables, tracker_extras, questions, static_attachments) -> TrackerOut:
    extra = tracker_extras.get(subscription.pk)
    if extra:
        return TrackerOut(
            id=subscription.pk,
            active=subscription.is_active,
            type=SubscriptionType.tracker,
            follow=subscription.follow_result,
            language=subscription.context_language,
            variables=variables.get_subscription(subscription.pk),
            subject=subscription.title,
            body=subscription.body,
            attachments=get_attachments_out(subscription, questions, static_attachments),
            **extra.dict(),
        )


def get_wiki_extras_out(survey: Survey) -> Dict[int, WikiExtraOut]:
    fields = ('pk', 'supertag', 'text', 'subscription_id')
    qs = (
        get_from_db(WikiSubscriptionData)
        .filter(subscription__survey_hook__survey=survey).values_list(*fields, named=True)
    )
    return {
        it.subscription_id: get_wiki_extra_out(it)
        for it in qs
    }


def get_wiki_extra_out(wiki_extra) -> WikiExtraOut:
    return WikiExtraOut(
        supertag=wiki_extra.supertag,
        body=wiki_extra.text,
    )


def get_tracker_extras_out(survey: Survey) -> Dict[int, TrackerExtraOut]:
    fields = (
        'pk', 'queue', 'parent', 'author', 'assignee', 'type', 'priority',
        'fields', 'subscription_id',
    )
    qs = (
        get_from_db(StartrekSubscriptionData)
        .filter(subscription__survey_hook__survey=survey).values_list(*fields, named=True)
    )
    return {
        it.subscription_id: get_tracker_extra_out(it)
        for it in qs
    }


def get_tracker_extra_out(tracker_extra) -> TrackerExtraOut:
    fields = [
        get_tracker_field_out(field)
        for field in json_loads(tracker_extra.fields) or []
    ]
    return TrackerExtraOut(
        queue=tracker_extra.queue,
        parent=tracker_extra.parent or None,
        author=tracker_extra.author or None,
        assignee=tracker_extra.assignee or None,
        issue_type=tracker_extra.type,
        priority=tracker_extra.priority,
        fields=fields or None,
    )


def get_tracker_field_key_out(key_field) -> TrackerFieldKeyOut:
    return TrackerFieldKeyOut(
        slug=key_field.get('id') or key_field.get('slug'),
        name=key_field.get('name'),
        type=key_field.get('type'),
    )


def get_tracker_field_out(tracker_field) -> TrackerFieldOut:
    return TrackerFieldOut(
        key=get_tracker_field_key_out(tracker_field.get('key')),
        value=tracker_field.get('value'),
        only_with_value=tracker_field.get('add_only_with_value') or False,
    )


def get_jsonrpc_params_out(survey: Survey) -> Dict[int, List[ParamOut]]:
    fields = ('pk', 'name', 'value', 'add_only_with_value', 'subscription__subscription_id')
    qs = (
        get_from_db(JSONRPCSubscriptionParam)
        .filter(subscription__subscription__survey_hook__survey=survey).values_list(*fields, named=True)
    )
    out = defaultdict(list)
    for it in qs:
        out[it.subscription__subscription_id].append(get_jsonrpc_param_out(it))
    return out


def get_jsonrpc_param_out(param) -> ParamOut:
    return ParamOut(
        name=param.name,
        value=param.value,
        only_with_value=param.add_only_with_value,
    )


def get_jsonrpc_extras_out(survey: Survey) -> Dict[int, JsonRpcExtraOut]:
    fields = ('pk', 'method', 'subscription_id')
    qs = (
        get_from_db(JSONRPCSubscriptionData)
        .filter(subscription__survey_hook__survey=survey).values_list(*fields, named=True)
    )
    return {
        it.subscription_id: get_jsonrpc_extra_out(it)
        for it in qs
    }


def get_jsonrpc_extra_out(jsonrpc_extra) -> JsonRpcExtraOut:
    return JsonRpcExtraOut(
        method=jsonrpc_extra.method,
    )


def get_headers_out(survey: Survey) -> Dict[int, List[HeaderOut]]:
    fields = ('pk', 'name', 'value', 'add_only_with_value', 'subscription_id')
    qs = (
        get_from_db(SubscriptionHeader)
        .filter(subscription__survey_hook__survey=survey).values_list(*fields, named=True)
    )
    out = defaultdict(list)
    for it in qs:
        out[it.subscription_id].append(get_header_out(it))
    return out


def get_header_out(header) -> HeaderOut:
    return HeaderOut(
        name=header.name,
        value=header.value,
        only_with_value=header.add_only_with_value,
    )


def get_static_attachments_out(survey: Survey) -> Dict[int, List[StaticAttachmentOut]]:
    fields = ('subscription_id', 'file')
    qs = (
        get_from_db(SubscriptionAttachment)
        .filter(subscription__survey_hook__survey=survey)
        .order_by('file')
        .values_list(*fields, named=True)
    )
    files = defaultdict(list)
    for it in qs:
        files[it.file].append(it.subscription_id)
    if not files:
        return {}

    fields = ('pk', 'path', 'original_name')
    qs = (
        get_from_db(ProxyStorageModel)
        .filter(path__in=files.keys())
        .values_list(*fields, named=True)
    )
    meta = {
        it.path: {
            'name': it.original_name,
            'links': {
                'orig': get_mds_url(it.path),
            },
        }
        for it in qs
    }
    if not meta:
        return {}

    out = defaultdict(list)
    for (path, subscription_ids) in files.items():
        params = meta.get(path)
        if params:
            for subscription_id in subscription_ids:
                out[subscription_id].append(StaticAttachmentOut(**params))
    return out


def get_template_attachments_out(survey: Survey, variables) -> Dict[int, List[TemplateAttachmentOut]]:
    fields = (
        'servicesurveyhooksubscription_id', 'integrationfiletemplate_id',
        'integrationfiletemplate__name', 'integrationfiletemplate__template',
        'integrationfiletemplate__type',
    )
    TemplateAttachmentManyToMany = ServiceSurveyHookSubscription.attachment_templates.through
    qs = (
        get_from_db(TemplateAttachmentManyToMany)
        .filter(servicesurveyhooksubscription__survey_hook__survey=survey)
        .order_by('integrationfiletemplate_id')
        .values_list(*fields, named=True)
    )
    out = defaultdict(list)
    for it in qs:
        out[it.servicesurveyhooksubscription_id].append(get_template_attachment_out(it, variables))
    return out


def get_template_attachment_out(template_attachment, variables) -> TemplateAttachmentOut:
    mime = {
        'txt': 'text/plain',
        'pdf': 'application/pdf',
    }
    return TemplateAttachmentOut(
        name=template_attachment.integrationfiletemplate__name,
        body=template_attachment.integrationfiletemplate__template,
        mime=mime.get(template_attachment.integrationfiletemplate__type),
        variables=variables.get_template(template_attachment.integrationfiletemplate_id),
    )


def get_questions_out(survey: Survey) -> Dict[int, List[str]]:
    fields = ('servicesurveyhooksubscription_id', 'surveyquestion__param_slug')
    qs = (
        get_from_db(QuestionManyToMany)
        .filter(servicesurveyhooksubscription__survey_hook__survey=survey)
        .filter(surveyquestion__is_deleted=False)
        .values_list(*fields, named=True)
        .order_by('surveyquestion__page', 'surveyquestion__position')
    )
    out = defaultdict(list)
    for it in qs:
        out[it.servicesurveyhooksubscription_id].append(it.surveyquestion__param_slug)
    return out


def get_attachments_out(
    subscription, questions, static_attachments=None, template_attachments=None,
) -> AttachmentsOut:
    question = None
    _all = subscription.is_all_questions
    items = questions.get(subscription.pk) if not _all else None
    if _all or items:
        question = QuestionAttachmentsOut(all=_all, items=items)

    static = None
    if static_attachments:
        static = static_attachments.get(subscription.pk)

    template = None
    if template_attachments:
        template = template_attachments.get(subscription.pk)

    out = None
    if question or static or template:
        out = AttachmentsOut(question=question, static=static, template=template)
    return out


def get_questions_extra_out(subscription, questions) -> QuestionsOut:
    out = None
    is_all_questions = subscription.is_all_questions
    if is_all_questions:
        items = None
    else:
        items = questions.get(subscription.pk)
    if is_all_questions or items:
        out = QuestionsOut(all=is_all_questions, items=items)
    return out


def get_variables_out(survey: Survey) -> VariablesOut:
    fields = (
        'variable_id', 'var', 'format_name', 'arguments', 'filters',
        'hook_subscription_id', 'integration_file_template_id',
    )
    question_slugs = LazyProxy(lambda: get_question_slugs(survey))
    subscription_qs = (
        get_from_db(SurveyVariable)
        .filter(hook_subscription__survey_hook__survey=survey)
        .values_list(*fields, named=True)
    )
    template_qs = (
        get_from_db(SurveyVariable)
        .filter(integration_file_template__survey=survey)
        .values_list(*fields, named=True)
    )
    qs = subscription_qs.union(template_qs, all=True)

    subscriptions = defaultdict(list)
    templates = defaultdict(list)
    for it in qs:
        var = get_variable_out(it, question_slugs)
        if var:
            if it.hook_subscription_id:
                subscriptions[it.hook_subscription_id].append(var)
            elif it.integration_file_template_id:
                templates[it.integration_file_template_id].append(var)
    return VariablesOut(subscriptions=subscriptions, templates=templates)


def get_question_slugs_with_type(survey: Survey) -> Dict[str, Tuple[int, str]]:
    fields = ('pk', 'param_slug', 'answer_type__slug')
    qs = (
        get_from_slave(SurveyQuestion)
        .filter(survey=survey)
        .order_by()
        .values_list(*fields, named=True)
    )
    return {
        it.param_slug: (it.pk, it.answer_type__slug)
        for it in qs
    }


def get_question_slugs(survey: Survey) -> Dict[int, str]:
    fields = ('pk', 'param_slug')
    qs = (
        get_from_slave(SurveyQuestion)
        .filter(survey=survey)
        .order_by()
        .values_list(*fields, named=True)
    )
    return {
        it.pk: it.param_slug
        for it in qs
    }


def get_variable_out(var, question_slugs) -> VariableOut:
    params = {
        'id': var.variable_id,
        'type': var.var,
        'renderer': var.format_name or None,
    }

    args = json_loads(var.arguments) or None
    if isinstance(args, dict):
        if'question' in args:
            slug = question_slugs.get(args['question'])
            if slug:
                params['question'] = slug
            else:
                return None
        if 'questions' in args or 'is_all_questions' in args:
            if args['is_all_questions']:
                params['questions'] = QuestionsOut(all=True)
            else:
                items = []
                for question in args.get('questions') or []:
                    slug = question_slugs.get(question)
                    if slug:
                        items.append(slug)
                params['questions'] = QuestionsOut(items=items)
        params['only_with_value'] = args.get('only_with_value')
        params['show_filenames'] = args.get('show_filenames')
        params['name'] = args.get('name')

    filters = json_loads(var.filters) or None
    if isinstance(filters, list):
        params['filters'] = filters

    return VariableOut(**params)


def get_variable_obj(questions: dict, i: int, data: VariableIn) -> SurveyVariable:
    params = {
        'variable_id': data.id,
        'var': data.type.value,
        'filters': [f.value for f in data.filters or []],
    }
    if data.is_require_question(data.type):
        if data.question not in questions:
            raise NotFoundError(loc=['variables', i, 'question'], value=data.question)
        question_id, answer_type = questions[data.question]
        params['arguments'] = {'question': question_id}
        if answer_type == 'answer_files' and data.show_filenames is not None:
            params['arguments']['show_filenames'] = data.show_filenames
    elif data.is_require_questions(data.type):
        need_show_filenames = False
        if data.questions.all:
            params['arguments'] = {'is_all_questions': True}
            need_show_filenames = True
        else:
            params['arguments'] = {'is_all_questions': False}
            question_list = []
            for j, question in enumerate(data.questions.items or []):
                if question not in questions:
                    raise NotFoundError(loc=['variables', i, 'questions', 'items', j], value=question)
                question_id, answer_type = questions[question]
                question_list.append(question_id)
                if answer_type == 'answer_files':
                    need_show_filenames = True
            params['arguments']['questions'] = question_list
        if need_show_filenames and data.show_filenames is not None:
            params['arguments']['show_filenames'] = data.show_filenames
    else:
        params['arguments'] = {}
    if data.name is not None:
        params['arguments']['name'] = data.name
    if data.only_with_value is not None:
        params['arguments']['only_with_value'] = data.only_with_value
    if data.renderer is not None:
        params['format_name'] = data.renderer.value
    return SurveyVariable(**params)


def get_variables_obj(questions: dict, data: List[VariableIn]) -> List[SurveyVariable]:
    return [
        get_variable_obj(questions, i, it)
        for i, it in enumerate(data)
    ]


def create_variables(subscription_id: int, variables: List[SurveyVariable]):
    if variables:
        for it in variables:
            it.hook_subscription_id = subscription_id
        try:
            SurveyVariable.objects.bulk_create(variables)
        except IntegrityError as e:
            # duplicate key value violates unique constraint
            # "idx_surveyme_integration_surveyvariable_variable_id_pkey"
            # DETAIL:  Key (variable_id)=(61ee49642a626a048f93bbc7) already exists.
            errors = []
            orig_msg = str(e)
            if orig_msg.startswith('duplicate key'):
                m = re.search(r'\(variable_id\)=\((\w+)\)', orig_msg)
                if m:
                    var_id = m.group(1)
                    for i, var in enumerate(variables):
                        if var.variable_id == var_id:
                            break
                    errors.append(DuplicatedError(loc=['variables', i, 'id'], value=var_id))
            if not errors:
                errors.append(DuplicatedError(loc=['variables', 0, 'id'], value=None))
            raise ValidationError(errors)


def modify_variables(subscription_id: int, variables: List[SurveyVariable]):
    if variables is not None:
        SurveyVariable.objects.filter(hook_subscription_id=subscription_id).delete()
        create_variables(subscription_id, variables)


def create_headers(subscription_id: int, headers: List[SubscriptionHeader]):
    if headers:
        for it in headers:
            it.subscription_id = subscription_id
        SubscriptionHeader.objects.bulk_create(headers)


def modify_headers(subscription_id: int, headers: List[SubscriptionHeader]):
    if headers is not None:
        SubscriptionHeader.objects.filter(subscription_id=subscription_id).delete()
        create_headers(subscription_id, headers)


def create_attachments(subscription_id: int, attachments: List[QuestionManyToMany]):
    if attachments:
        for it in attachments:
            it.servicesurveyhooksubscription_id = subscription_id
        QuestionManyToMany.objects.bulk_create(attachments)


def modify_attachments(subscription_id: int, attachments: List[QuestionManyToMany]):
    if attachments is not None:
        QuestionManyToMany.objects.filter(servicesurveyhooksubscription_id=subscription_id).delete()
        create_attachments(subscription_id, attachments)


def create_static_attachments(subscription_id: int, static_attachments: List[SubscriptionAttachment]):
    if static_attachments:
        for it in static_attachments:
            it.subscription_id = subscription_id
        SubscriptionAttachment.objects.bulk_create(static_attachments)


def modify_static_attachments(subscription_id: int, static_attachments: List[SubscriptionAttachment]):
    if static_attachments is not None:
        SubscriptionAttachment.objects.filter(subscription_id=subscription_id).delete()
        create_static_attachments(subscription_id, static_attachments)


def create_email_obj(user: User, survey: Survey, hook_id: int, data: EmailIn) -> int:
    if not data.email_to_address:
        raise MissingError(loc=['email_to_address'])
    params = {
        'survey_hook_id': hook_id,
        'service_type_action_id': ActionType.email,
        'title': data.subject or '',
        'body': data.body or '',
        'is_all_questions': False,
        'email_to_address': data.email_to_address,
        'email_from_title': data.email_from_title or '',
        'context_language': LanguageType.from_request,
    }
    if data.email_from_address is not None:
        if not settings.IS_BUSINESS_SITE:
            params['email_from_address'] = data.email_from_address
    if settings.IS_BUSINESS_SITE:
        params['email_spam_check'] = True
    else:
        params['email_spam_check'] = data.email_spam_check or False
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    headers = [
        SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
        for it in data.headers or []
    ]
    questions = get_question_slugs_with_type(survey)
    attachments = []
    static_attachments = []
    if data.attachments:
        if data.attachments.question:
            if data.attachments.question.all:
                params['is_all_questions'] = True
            else:
                for i, it in enumerate(data.attachments.question.items or []):
                    if it in questions:
                        question_id, answer_type = questions[it]
                        if answer_type == 'answer_files':
                            attachments.append(QuestionManyToMany(surveyquestion_id=question_id))
                    else:
                        raise NotFoundError(loc=['attachments', 'question', 'items', i], value=it)
        if data.attachments.static:
            for it in data.attachments.static:
                static_attachments.append(SubscriptionAttachment(file=StaticFile(it)))
    variables = None
    if data.variables:
        variables = get_variables_obj(questions, data.variables)
    with transaction.atomic():
        subscription = ServiceSurveyHookSubscription.objects.create(**params)
        create_headers(subscription.pk, headers)
        create_attachments(subscription.pk, attachments)
        create_static_attachments(subscription.pk, static_attachments)
        create_variables(subscription.pk, variables)
        return subscription.pk


def create_tracker_obj(user: User, survey: Survey, hook_id: int, data: TrackerIn) -> int:
    params = {
        'survey_hook_id': hook_id,
        'service_type_action_id': ActionType.tracker,
        'title': data.subject or '',
        'body': data.body or '',
        'is_all_questions': False,
        'context_language': LanguageType.from_request,
    }
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language

    if not data.subject:
        raise MissingError(loc=['subject'])

    if not data.queue and not data.parent:
        raise ValidationError([
            MissingError(loc=['queue']),
            MissingError(loc=['parent']),
        ])
    queue_name = get_tracker_queue(data.queue, data.parent)
    dir_id = survey.org.dir_id if survey.org else None
    validator = TrackerValidator(queue_name, dir_id)
    extra = {
        'queue': '',
        'parent': '',
        'author': data.author or '',
        'assignee': data.assignee or '',
    }
    if data.queue:
        extra['queue'] = validator.validate_queue(data.queue)
    if data.parent:
        extra['parent'] = validator.validate_parent(data.parent)
    extra['type'] = validator.validate_issue_type(data.issue_type or 0)
    extra['priority'] = validator.validate_priority(data.priority or 0)
    if data.fields:
        extra['fields'] = validator.validate_fields(data.fields)

    questions = get_question_slugs_with_type(survey)
    attachments = []
    static_attachments = []
    if data.attachments:
        if data.attachments.question:
            if data.attachments.question.all:
                params['is_all_questions'] = True
            else:
                for i, it in enumerate(data.attachments.question.items or []):
                    if it in questions:
                        question_id, answer_type = questions[it]
                        if answer_type == 'answer_files':
                            attachments.append(QuestionManyToMany(surveyquestion_id=question_id))
                    else:
                        raise NotFoundError(loc=['attachments', 'question', 'items', i], value=it)

        if data.attachments.static:
            for it in data.attachments.static:
                static_attachments.append(SubscriptionAttachment(file=StaticFile(it)))

    variables = None
    if data.variables:
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        subscription = ServiceSurveyHookSubscription.objects.create(**params)
        StartrekSubscriptionData.objects.create(subscription=subscription, **extra)
        create_attachments(subscription.pk, attachments)
        create_static_attachments(subscription.pk, static_attachments)
        create_variables(subscription.pk, variables)
        return subscription.pk


def create_wiki_obj(user: User, survey: Survey, hook_id: int, data: WikiIn) -> int:
    if not data.supertag:
        raise MissingError(loc=['supertag'])
    dir_id = survey.org.dir_id if survey.org else None
    if not check_if_supertag_exist(data.supertag, dir_id):
        raise NotFoundError(loc=['supertag'], value=data.supertag)
    params = {
        'survey_hook_id': hook_id,
        'service_type_action_id': ActionType.wiki,
        'context_language': LanguageType.from_request,
    }
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    extra = {
        'supertag': data.supertag,
        'text': data.body or '',
    }
    variables = None
    if data.variables:
        questions = get_question_slugs_with_type(survey)
        variables = get_variables_obj(questions, data.variables)
    with transaction.atomic():
        subscription = ServiceSurveyHookSubscription.objects.create(**params)
        WikiSubscriptionData.objects.create(subscription=subscription, **extra)
        create_variables(subscription.pk, variables)
        return subscription.pk


def create_jsonrpc_obj(user: User, survey: Survey, hook_id: int, data: WikiIn) -> int:
    if not data.url:
        raise MissingError(loc=['url'])
    if data.tvm_client:
        if not has_role_tvm_or_form_manager(data.tvm_client, user.uid):
            raise NotPermittedError(loc=['tvm_client'], value=data.tvm_client)
    params = {
        'survey_hook_id': hook_id,
        'service_type_action_id': ActionType.jsonrpc,
        'http_url': data.url,
        'context_language': LanguageType.from_request,
    }
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.tvm_client is not None:
        params['tvm2_client_id'] = data.tvm_client
    extra = {
        'method': data.method or '',
    }
    jsonrpc_params = [
        JSONRPCSubscriptionParam(
            name=it.name,
            value=it.value,
            add_only_with_value=it.only_with_value,
        )
        for it in data.params or []
    ]
    headers = [
        SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
        for it in data.headers or []
    ]
    variables = None
    if data.variables:
        questions = get_question_slugs_with_type(survey)
        variables = get_variables_obj(questions, data.variables)
    with transaction.atomic():
        subscription = ServiceSurveyHookSubscription.objects.create(**params)
        jsonrpc = JSONRPCSubscriptionData.objects.create(subscription=subscription, **extra)
        if jsonrpc_params:
            for it in jsonrpc_params:
                it.subscription_id = jsonrpc.pk
            JSONRPCSubscriptionParam.objects.bulk_create(jsonrpc_params)
        create_headers(subscription.pk, headers)
        create_variables(subscription.pk, variables)
        return subscription.pk


def create_postput_obj(
    user: User, survey: Survey,
    hook_id: int, data: Union[PostIn, PutIn],
    action: ActionType,
) -> int:
    if not data.url:
        raise MissingError(loc=['url'])
    if data.tvm_client:
        if not has_role_tvm_or_form_manager(data.tvm_client, user.uid):
            raise NotPermittedError(loc=['tvm_client'], value=data.tvm_client)
    params = {
        'survey_hook_id': hook_id,
        'service_type_action_id': action.value,
        'http_url': data.url or '',
        'http_method': action.name,
        'is_all_questions': False,
        'context_language': LanguageType.from_request,
    }
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.tvm_client is not None:
        params['tvm2_client_id'] = data.tvm_client
    if data.format is not None:
        params['http_format_name'] = data.format
    questions = get_question_slugs_with_type(survey)
    postput_questions = []
    if data.questions:
        if data.questions.all:
            params['is_all_questions'] = True
        else:
            for i, it in enumerate(data.questions.items or []):
                if it in questions:
                    question_id, _ = questions[it]
                    postput_questions.append(question_id)
                else:
                    raise NotFoundError(loc=['questions', 'items', i], value=it)
    headers = [
        SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
        for it in data.headers or []
    ]
    variables = None
    if data.variables:
        variables = get_variables_obj(questions, data.variables)
    with transaction.atomic():
        subscription = ServiceSurveyHookSubscription.objects.create(**params)
        if postput_questions:
            subscription.questions.add(*postput_questions)
        create_headers(subscription.pk, headers)
        create_variables(subscription.pk, variables)
        return subscription.pk


def create_http_obj(user: User, survey: Survey, hook_id: int, data: HttpIn) -> int:
    if not data.url:
        raise MissingError(loc=['url'])
    if not data.method:
        raise MissingError(loc=['method'])
    if data.tvm_client:
        if not has_role_tvm_or_form_manager(data.tvm_client, user.uid):
            raise NotPermittedError(loc=['tvm_client'], value=data.tvm_client)
    params = {
        'survey_hook_id': hook_id,
        'service_type_action_id': ActionType.http,
        'http_url': data.url,
        'http_method': data.method,
        'body': data.body or '',
        'context_language': LanguageType.from_request,
    }
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.tvm_client is not None:
        params['tvm2_client_id'] = data.tvm_client
    headers = [
        SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
        for it in data.headers or []
    ]
    variables = None
    if data.variables:
        questions = get_question_slugs_with_type(survey)
        variables = get_variables_obj(questions, data.variables)
    with transaction.atomic():
        subscription = ServiceSurveyHookSubscription.objects.create(**params)
        create_headers(subscription.pk, headers)
        create_variables(subscription.pk, variables)
        return subscription.pk


def modify_email_obj(user: User, survey: Survey, subscription_id: int, data: EmailIn) -> int:
    params = {}
    if data.subject is not None:
        params['title'] = data.subject
    if data.body is not None:
        params['body'] = data.body
    if data.email_to_address is not None:
        if not data.email_to_address:
            raise MissingError(loc=['email_to_address'])
        params['email_to_address'] = data.email_to_address
    if data.email_from_title is not None:
        params['email_from_title'] = data.email_from_title
    if data.email_from_address is not None:
        if not settings.IS_BUSINESS_SITE:
            params['email_from_address'] = data.email_from_address
    if data.email_spam_check is not None:
        if not settings.IS_BUSINESS_SITE:
            params['email_spam_check'] = data.email_spam_check
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language

    headers = None
    if data.headers is not None:
        headers = [
            SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
            for it in data.headers or []
        ]

    questions = get_question_slugs_with_type(survey)

    attachments = None
    static_attachments = None
    if data.attachments is not None:
        if data.attachments.question is not None:
            attachments = []
            if data.attachments.question.all is not None and data.attachments.question.all:
                params['is_all_questions'] = True
            else:
                params['is_all_questions'] = False
                for i, it in enumerate(data.attachments.question.items or []):
                    if it in questions:
                        question_id, answer_type = questions[it]
                        if answer_type == 'answer_files':
                            attachments.append(QuestionManyToMany(surveyquestion_id=question_id))
                    else:
                        raise NotFoundError(loc=['attachments', 'question', 'items', i], value=it)

        if data.attachments.static is not None:
            static_attachments = [
                SubscriptionAttachment(file=StaticFile(it))
                for it in data.attachments.static
            ]

    variables = None
    if data.variables is not None:
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        if (
            params or headers is not None or variables is not None
            or attachments is not None or static_attachments is not None
        ):
            params['date_updated'] = timezone.now()
            ServiceSurveyHookSubscription.objects.filter(pk=subscription_id).update(**params)

        modify_attachments(subscription_id, attachments)
        modify_static_attachments(subscription_id, static_attachments)
        modify_headers(subscription_id, headers)
        modify_variables(subscription_id, variables)
        return subscription_id


def get_tracker_queue(queue: str, parent: str) -> str:
    if queue:
        return queue
    if parent:
        pos = parent.find('-')
        if pos >= 0:
            return parent[:pos]
        raise InvalidError(loc=['parent'], value=parent)
    raise InvalidError(loc=['queue'], value=queue)


def get_tracker_queue_from_subscription(subscription_id: int, data: TrackerIn) -> str:
    try:
        extra = (
            get_from_slave(StartrekSubscriptionData)
            .values_list('queue', 'parent', named=True)
            .get(subscription_id=subscription_id)
        )
    except StartrekSubscriptionData.DoesNotExits:
        raise NotFoundError(loc=['queue'], value=None)
    else:
        queue = extra.queue if data.queue is None else data.queue
        parent = extra.parent if data.parent is None else data.parent
        return get_tracker_queue(queue, parent)


def modify_tracker_obj(user: User, survey: Survey, subscription_id: int, data: TrackerIn) -> int:
    params = {}
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.subject is not None:
        if not data.subject:
            raise MissingError(loc=['subject'])
        params['title'] = data.subject
    if data.body is not None:
        params['body'] = data.body

    extra = {}
    queue_name = LazyProxy(lambda: get_tracker_queue_from_subscription(subscription_id, data))
    dir_id = survey.org.dir_id if survey.org else None
    validator = TrackerValidator(queue_name, dir_id)
    if data.queue is not None:
        extra['queue'] = validator.validate_queue(data.queue)
    if data.parent is not None:
        extra['parent'] = validator.validate_parent(data.parent)
    if data.issue_type is not None:
        extra['type'] = validator.validate_issue_type(data.issue_type)
    if data.priority is not None:
        extra['priority'] = validator.validate_priority(data.priority)
    if data.author is not None:
        extra['author'] = data.author
    if data.assignee is not None:
        extra['assignee'] = data.assignee
    if data.fields is not None:
        extra['fields'] = validator.validate_fields(data.fields)

    questions = get_question_slugs_with_type(survey)
    attachments = None
    static_attachments = None
    if data.attachments is not None:
        if data.attachments.question is not None:
            attachments = []
            if data.attachments.question.all is not None and data.attachments.question.all:
                params['is_all_questions'] = True
            else:
                params['is_all_questions'] = False
                for i, it in enumerate(data.attachments.question.items or []):
                    if it in questions:
                        question_id, answer_type = questions[it]
                        if answer_type == 'answer_files':
                            attachments.append(QuestionManyToMany(surveyquestion_id=question_id))
                    else:
                        raise NotFoundError(loc=['attachments', 'question', 'items', i], value=it)

        if data.attachments.static is not None:
            static_attachments = [
                SubscriptionAttachment(file=StaticFile(it))
                for it in data.attachments.static
            ]

    variables = None
    if data.variables is not None:
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        if (
            params or extra or variables is not None
            or attachments is not None or static_attachments is not None
        ):
            params['date_updated'] = timezone.now()
            ServiceSurveyHookSubscription.objects.filter(pk=subscription_id).update(**params)
        if extra:
            StartrekSubscriptionData.objects.filter(subscription_id=subscription_id).update(**extra)
        modify_attachments(subscription_id, attachments)
        modify_static_attachments(subscription_id, static_attachments)
        modify_variables(subscription_id, variables)
        return subscription_id


def modify_wiki_obj(user: User, survey: Survey, subscription_id: int, data: WikiIn) -> int:
    params = {}
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language

    extra = {}
    if data.supertag is not None:
        if not data.supertag:
            raise MissingError(loc=['supertag'])
        dir_id = survey.org.dir_id if survey.org else None
        if not check_if_supertag_exist(data.supertag, dir_id):
            raise NotFoundError(loc=['supertag'], value=data.supertag)
        extra['supertag'] = data.supertag
    if data.body is not None:
        extra['text'] = data.body

    variables = None
    if data.variables is not None:
        questions = get_question_slugs_with_type(survey)
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        if params or extra or variables is not None:
            params['date_updated'] = timezone.now()
            ServiceSurveyHookSubscription.objects.filter(pk=subscription_id).update(**params)

        if extra:
            WikiSubscriptionData.objects.filter(subscription_id=subscription_id).update(**extra)

        modify_variables(subscription_id, variables)
        return subscription_id


def modify_jsonrpc_obj(user: User, survey: Survey, subscription_id: int, data: JsonRpcIn) -> int:
    params = {}
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.url is not None:
        if not data.url:
            raise MissingError(loc=['url'])
        params['http_url'] = data.url
    if data.tvm_client is not None:
        if data.tvm_client:
            if not has_role_tvm_or_form_manager(data.tvm_client, user.uid):
                raise NotPermittedError(loc=['tvm_client'], value=data.tvm_client)
        params['tvm2_client_id'] = data.tvm_client or None

    extra = {}
    if data.method is not None:
        extra['method'] = data.method

    headers = None
    if data.headers is not None:
        headers = [
            SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
            for it in data.headers or []
        ]

    jsonrpc_params = None
    if data.params is not None:
        jsonrpc_params = [
            JSONRPCSubscriptionParam(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
            for it in data.params or []
        ]

    variables = None
    if data.variables is not None:
        questions = get_question_slugs_with_type(survey)
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        if params or extra or jsonrpc_params is not None or headers is not None or variables is not None:
            params['date_updated'] = timezone.now()
            ServiceSurveyHookSubscription.objects.filter(pk=subscription_id).update(**params)

        if extra:
            JSONRPCSubscriptionData.objects.filter(subscription_id=subscription_id).update(**extra)

        if jsonrpc_params is not None:
            JSONRPCSubscriptionParam.objects.filter(subscription__subscription_id=subscription_id).delete()
            if jsonrpc_params:
                jsonrpc, _ = JSONRPCSubscriptionData.objects.get_or_create(subscription_id=subscription_id)
                for it in jsonrpc_params:
                    it.subscription_id = jsonrpc.pk
                JSONRPCSubscriptionParam.objects.bulk_create(jsonrpc_params)

        modify_headers(subscription_id, headers)
        modify_variables(subscription_id, variables)
        return subscription_id


def modify_postput_obj(user: User, survey: Survey, subscription_id: int, data: Union[PostIn, PutIn]) -> int:
    params = {}
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.url is not None:
        if not data.url:
            raise MissingError(loc=['url'])
        params['http_url'] = data.url
    if data.format is not None:
        params['http_format_name'] = data.format
    if data.tvm_client is not None:
        if data.tvm_client:
            if not has_role_tvm_or_form_manager(data.tvm_client, user.uid):
                raise NotPermittedError(loc=['tvm_client'], value=data.tvm_client)
        params['tvm2_client_id'] = data.tvm_client or None

    headers = None
    if data.headers is not None:
        headers = [
            SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
            for it in data.headers or []
        ]

    questions = get_question_slugs_with_type(survey)

    postput_questions = None
    if data.questions is not None:
        postput_questions = []
        if data.questions.all is not None and data.questions.all:
            params['is_all_questions'] = True
        else:
            params['is_all_questions'] = False
            for i, it in enumerate(data.questions.items or []):
                if it in questions:
                    question_id, _ = questions[it]
                    postput_questions.append(QuestionManyToMany(surveyquestion_id=question_id))
                else:
                    raise NotFoundError(loc=['questions', 'items', i], value=it)

    variables = None
    if data.variables is not None:
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        if params or postput_questions is not None or headers is not None or variables is not None:
            params['date_updated'] = timezone.now()
            ServiceSurveyHookSubscription.objects.filter(pk=subscription_id).update(**params)

        if postput_questions is not None:
            QuestionManyToMany.objects.filter(servicesurveyhooksubscription_id=subscription_id).delete()
            if postput_questions:
                for it in postput_questions:
                    it.servicesurveyhooksubscription_id = subscription_id
                QuestionManyToMany.objects.bulk_create(postput_questions)

        modify_headers(subscription_id, headers)
        modify_variables(subscription_id, variables)
        return subscription_id


def modify_http_obj(user: User, survey: Survey, subscription_id: int, data: HttpIn) -> int:
    params = {}
    if data.active is not None:
        params['is_active'] = data.active
    if data.follow is not None:
        params['follow_result'] = data.follow
    if data.language is not None:
        params['context_language'] = data.language
    if data.url is not None:
        if not data.url:
            raise MissingError(loc=['url'])
        params['http_url'] = data.url
    if data.method is not None:
        if not data.method:
            raise MissingError(loc=['method'])
        params['http_method'] = data.method
    if data.body is not None:
        params['body'] = data.body
    if data.tvm_client is not None:
        if data.tvm_client:
            if not has_role_tvm_or_form_manager(data.tvm_client, user.uid):
                raise NotPermittedError(loc=['tvm_client'], value=data.tvm_client)
        params['tvm2_client_id'] = data.tvm_client or None

    headers = None
    if data.headers is not None:
        headers = [
            SubscriptionHeader(name=it.name, value=it.value, add_only_with_value=it.only_with_value)
            for it in data.headers or []
        ]

    variables = None
    if data.variables is not None:
        questions = get_question_slugs_with_type(survey)
        variables = get_variables_obj(questions, data.variables)

    with transaction.atomic():
        if params or headers is not None or variables is not None:
            params['date_updated'] = timezone.now()
            ServiceSurveyHookSubscription.objects.filter(pk=subscription_id).update(**params)

        modify_headers(subscription_id, headers)
        modify_variables(subscription_id, variables)
        return subscription_id


def create_subscription(user: User, survey: Survey, hook_id: int, data: SubscriptionIn) -> int:
    hook = get_from_slave(SurveyHook).get(survey=survey, pk=hook_id)
    data = data.__root__
    if data.type == SubscriptionType.email:
        return create_email_obj(user, survey, hook.pk, data)
    elif data.type == SubscriptionType.tracker:
        return create_tracker_obj(user, survey, hook.pk, data)
    elif data.type == SubscriptionType.wiki:
        return create_wiki_obj(user, survey, hook.pk, data)
    elif data.type == SubscriptionType.jsonrpc:
        return create_jsonrpc_obj(user, survey, hook.pk, data)
    elif data.type == SubscriptionType.post:
        return create_postput_obj(user, survey, hook.pk, data, ActionType.post)
    elif data.type == SubscriptionType.put:
        return create_postput_obj(user, survey, hook.pk, data, ActionType.put)
    elif data.type == SubscriptionType.http:
        return create_http_obj(user, survey, hook.pk, data)


def modify_subscription(
    user: User,
    survey: Survey, hook_id: int, subscription_id: int,
    data: SubscriptionIn,
) -> int:
    subscription = (
        get_from_slave(ServiceSurveyHookSubscription)
        .select_related('service_type_action', 'service_type_action__service_type')
        .only('pk', 'service_type_action__service_type__slug', 'service_type_action__slug')
        .get(survey_hook__survey=survey, survey_hook_id=hook_id, pk=subscription_id)
    )
    data = data.__root__
    service_slug = subscription.service_type_action.service_type.slug
    action_slug = subscription.service_type_action.slug
    type_ = get_subscription_type(service_slug, action_slug)
    if data.type != type_:
        raise InvalidError(loc=['type'], value=data.type)
    if data.type == SubscriptionType.email:
        return modify_email_obj(user, survey, subscription.pk, data)
    elif data.type == SubscriptionType.wiki:
        return modify_wiki_obj(user, survey, subscription.pk, data)
    elif data.type == SubscriptionType.tracker:
        return modify_tracker_obj(user, survey, subscription.pk, data)
    elif data.type == SubscriptionType.jsonrpc:
        return modify_jsonrpc_obj(user, survey, subscription.pk, data)
    elif data.type == SubscriptionType.post:
        return modify_postput_obj(user, survey, subscription.pk, data)
    elif data.type == SubscriptionType.put:
        return modify_postput_obj(user, survey, subscription.pk, data)
    elif data.type == SubscriptionType.http:
        return modify_http_obj(user, survey, subscription.pk, data)


def delete_subscription(user: User, survey: Survey, hook_id: int, subscription_id: int) -> None:
    subscription = (
        get_from_slave(ServiceSurveyHookSubscription)
        .get(survey_hook__survey=survey, survey_hook_id=hook_id, pk=subscription_id)
    )
    subscription.delete()


def get_subscriptions_view(
    request,
    survey_id: SurveyId, hook_id: int,
) -> List[SubscriptionOut]:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    out = get_subscriptions_out(survey, hook_id)
    return out.get(hook_id) or []


def get_subscription_view(
    request,
    survey_id: SurveyId, hook_id: int, subscription_id: int,
) -> SubscriptionOut:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    return get_one_subscription_out(survey, hook_id, subscription_id)


def post_subscription_view(
    request,
    survey_id: SurveyId, hook_id: int,
    data: SubscriptionIn,
) -> SubscriptionOut:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    with set_master_db():
        subscription_id = create_subscription(request.user, survey, hook_id, data)
        return get_one_subscription_out(survey, hook_id, subscription_id)


def patch_subscription_view(
    request,
    survey_id: SurveyId, hook_id: int, subscription_id: int,
    data: SubscriptionIn,
) -> SubscriptionOut:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    with set_master_db():
        modify_subscription(request.user, survey, hook_id, subscription_id, data)
        return get_one_subscription_out(survey, hook_id, subscription_id)


def delete_subscription_view(
    request,
    survey_id: SurveyId, hook_id: int, subscription_id: int,
) -> None:
    survey = get_survey(survey_id, user=request.user, orgs=request.orgs, only=['pk'])
    check_perm(request.user, survey)
    delete_subscription(request.user, survey, hook_id, subscription_id)
