from django.db.models import Q
from typing import List

from events.accounts.models import User
from events.surveyme.models import Survey
from events.v3.schemas import SurveySuggestOut, HookSuggestOut, SubscriptionSuggestOut, SuggestOut
from events.v3.pagination import DEFAULT_PAGE_SIZE
from events.v3.perms import filter_permitted_surveys
from events.v3.types import OrderingType, LayerType, SurveyId
from events.v3.utils import get_from_slave, get_subscription_type_by_action_id


def get_surveys(
    user: User, orgs: List[str], text: str,
    limit: int, ordering: OrderingType,
) -> List[SurveySuggestOut]:
    if not text:
        return []
    qs = get_from_slave(Survey).filter(Q(pk__icontains=text) | Q(name__icontains=text))
    qs = filter_permitted_surveys(user, orgs, qs)
    if ordering == OrderingType.asc:
        qs = qs.order_by('name', 'pk')
    elif ordering == OrderingType.desc:
        qs = qs.order_by('-name', '-pk')
    if limit:
        qs = qs[:limit]
    qs = qs.values_list('pk', 'name', named=True)
    return [
        SurveySuggestOut(layer=LayerType.survey, id=it.pk, name=it.name)
        for it in qs
    ]


def get_hooks(
    user: User, orgs: List[str], text: str, survey_id: SurveyId,
    limit: int, ordering: OrderingType,
) -> List[HookSuggestOut]:
    if not text and not survey_id:
        return []
    qs = get_from_slave(Survey)
    if text:
        qs = qs.filter(Q(hooks__pk__icontains=text) | Q(hooks__name__icontains=text))
    if survey_id:
        qs = qs.filter(pk=survey_id)
    qs = filter_permitted_surveys(user, orgs, qs)
    if ordering == OrderingType.asc:
        qs = qs.order_by('hooks__name', 'hooks__pk')
    elif ordering == OrderingType.desc:
        qs = qs.order_by('-hooks__name', '-hooks__pk')
    if limit:
        qs = qs[:limit]
    qs = qs.values_list('pk', 'name', 'hooks__pk', 'hooks__name', named=True)
    return [
        HookSuggestOut(
            layer=LayerType.hook, id=it.hooks__pk, name=it.hooks__name,
            survey_id=it.pk, survey_name=it.name,
        )
        for it in qs
    ]


def get_subscriptions(
    user: User, orgs: List[str], text: str, survey_id: SurveyId, hook_id: int,
    limit: int, ordering: OrderingType,
) -> List[SubscriptionSuggestOut]:
    if not text and not survey_id and not hook_id:
        return []
    qs = get_from_slave(Survey)
    if text:
        qs = qs.filter(hooks__subscriptions__pk__icontains=text)
    if survey_id:
        qs = qs.filter(pk=survey_id)
    if hook_id:
        qs = qs.filter(hooks__pk=hook_id)
    qs = filter_permitted_surveys(user, orgs, qs)
    if ordering == OrderingType.asc:
        qs = qs.order_by('hooks__subscriptions__pk')
    elif ordering == OrderingType.desc:
        qs = qs.order_by('-hooks__subscriptions__pk')
    if limit:
        qs = qs[:limit]
    qs = qs.values_list(
        'pk', 'name',
        'hooks__pk', 'hooks__name',
        'hooks__subscriptions__pk', 'hooks__subscriptions__service_type_action_id',
        named=True,
    )
    return [
        SubscriptionSuggestOut(
            layer=LayerType.subscription, id=it.hooks__subscriptions__pk,
            type=get_subscription_type_by_action_id(it.hooks__subscriptions__service_type_action_id),
            hook_id=it.hooks__pk, hook_name=it.hooks__name,
            survey_id=it.pk, survey_name=it.name,
        )
        for it in qs
        if it.hooks__subscriptions__pk
    ]


def get_suggest_view(
    request,
    text: str = '', layer: LayerType = None,
    survey_id: SurveyId = None, hook_id: int = None,
    limit: int = DEFAULT_PAGE_SIZE, ordering: OrderingType = OrderingType.asc,
) -> List[SuggestOut]:
    match layer:
        case LayerType.survey:
            return get_surveys(request.user, request.orgs, text, limit, ordering)
        case LayerType.hook:
            return get_hooks(request.user, request.orgs, text, survey_id, limit, ordering)
        case LayerType.subscription:
            return get_subscriptions(request.user, request.orgs, text, survey_id, hook_id, limit, ordering)
        case _:
            return []
