# -*- coding: utf-8 -*-
from django.test import TestCase, override_settings
from rest_framework.test import APIClient

from events.yauth_contrib.mechanisms.tvm2 import Mechanism as TVMMechanism
from events.yauth_contrib.mechanisms.cookie import Mechanism as CookieMechanism
from events.yauth_contrib.mechanisms.http_header import Mechanism as HeaderMechanism
from events.yauth_contrib.mechanisms.oauth import Mechanism as OAuthMechanism
from events.yauth_contrib.mechanisms.iam_token import Mechanism as IamTokenMechanism


class MockSession:
    def __init__(self, uid, email=None, **fields):
        self.uid = uid
        self.lite_uid = None
        self.fields = fields
        self.redirect = False
        self.emails = []
        self.default_email = email
        self.new_session = None
        self.oauth = None
        self.secure = False
        self.new_sslsession = None
        self.url = None
        self.valid = True
        self.error = 'OK'

    def __contains__(self, item):
        return item in dir(self)


class MockXmlBlackbox:
    def __init__(self):
        self.url = 'http://blackbox.yandex.ru/'

    def sessionid(self, session_id, *args, **kwargs):
        return MockSession(session_id)

    def oauth(self, oauth_token, *args, **kwargs):
        return MockSession(oauth_token)

    def userinfo(self, uid, *args, **kwargs):
        return MockSession(uid)


class MockBlackboxMixin:
    @property
    def blackbox(self):
        return MockXmlBlackbox()


class MockServiceTicket:
    def __init__(self, src):
        self.src = src


class MockUserTicket:
    def __init__(self, uid):
        self.uids = [uid]
        self.default_uid = uid


class MockTVM:
    def parse_service_ticket(self, service_ticket):
        if service_ticket:
            return MockServiceTicket(service_ticket)

    def parse_user_ticket(self, user_ticket):
        if user_ticket:
            return MockUserTicket(user_ticket)

    def get_service_ticket(self, tvm_client_id):
        return str(tvm_client_id)


class MockTVMMechanism(MockBlackboxMixin, TVMMechanism):
    def get_tvm2_client(self):
        return MockTVM()


class MockCookieMechanism(MockBlackboxMixin, CookieMechanism):
    pass


class MockHeaderMechanism(MockBlackboxMixin, HeaderMechanism):
    pass


class MockOAuthMechanism(MockBlackboxMixin, OAuthMechanism):
    pass


class MockAccessClient:
    def get_cloud_uid(self, iam_token):
        return iam_token


class MockIamTokenMechanism(IamTokenMechanism):
    def _get_access_client(self):
        return MockAccessClient()

    def _get_organizations(self, uid, cloud_uid):
        parts = cloud_uid.split(':')
        return parts[1:]


class TVMClient(APIClient):
    auth_backends = [
        'guardian.backends.ObjectPermissionBackend',
        'events.yauth_contrib.helpers.MockTVMMechanism',
    ]

    def set_service_ticket(self, tvm_client_id):
        self.defaults['HTTP_X_YA_SERVICE_TICKET'] = str(tvm_client_id)

    def set_user_ticket(self, uid):
        self.defaults['HTTP_X_YA_USER_TICKET'] = str(uid)


class CookieClient(APIClient):
    auth_backends = [
        'guardian.backends.ObjectPermissionBackend',
        'events.yauth_contrib.helpers.MockCookieMechanism',
    ]

    def set_cookie(self, uid):
        self.cookies['Session_id'] = str(uid)

    def remove_cookie(self):
        self.cookies.pop('Session_id', None)


class HeaderClient(APIClient):
    auth_backends = [
        'guardian.backends.ObjectPermissionBackend',
        'events.yauth_contrib.helpers.MockHeaderMechanism',
    ]

    def set_header(self, uid):
        self.defaults['HTTP_AUTHORIZATION'] = f'SessionId {uid}'


class OAuthClient(APIClient):
    auth_backends = [
        'guardian.backends.ObjectPermissionBackend',
        'events.yauth_contrib.helpers.MockOAuthMechanism',
    ]

    def set_header(self, uid):
        self.defaults['HTTP_AUTHORIZATION'] = f'OAuth {uid}'


class IamTokenClient(APIClient):
    auth_backends = [
        'guardian.backends.ObjectPermissionBackend',
        'events.yauth_contrib.helpers.MockIamTokenMechanism',
    ]

    def set_token(self, iam_token):
        self.defaults['HTTP_AUTHORIZATION'] = f'Bearer {iam_token}'


@override_settings(AUTHENTICATION_BACKENDS=TVMClient.auth_backends)
class TVMAuthTestCase(TestCase):
    client_class = TVMClient


@override_settings(AUTHENTICATION_BACKENDS=CookieClient.auth_backends)
class CookieAuthTestCase(TestCase):
    client_class = CookieClient


@override_settings(AUTHENTICATION_BACKENDS=HeaderClient.auth_backends)
class HeaderAuthTestCase(TestCase):
    client_class = HeaderClient


@override_settings(AUTHENTICATION_BACKENDS=OAuthClient.auth_backends)
class OAuthTestCase(TestCase):
    client_class = OAuthClient


@override_settings(AUTHENTICATION_BACKENDS=IamTokenClient.auth_backends)
class IamTokenTestCase(TestCase):
    client_class = IamTokenClient
