# -*- coding: utf-8 -*-
import grpc
import logging
import re
import ssl

from django.conf import settings
from django_yauth.authentication_mechanisms.base import BaseMechanism
from django_yauth.user import BaseYandexUser
from yc_as_client.client import YCAccessServiceClient
from yc_as_client.exceptions import UnauthenticatedException

from events.common_app.directory import CachedDirectoryClient
from events.yauth_contrib.user import FormsYandexUserMixin, FormsAnonymousYandexUser
from events.yauth_contrib.mechanisms.base import BaseFormsMixin

logger = logging.getLogger(__name__)


class AccessClient:
    def __init__(self):
        self._host = settings.CLOUD_ACCESS_SERVICE_HOST
        self._creds = None

    def _get_creds(self):
        if not self._creds:
            host, port = self._host.split(':')
            cert = ssl.get_server_certificate((host, port))
            self._creds = grpc.ssl_channel_credentials(cert.encode())
        return self._creds

    def get_cloud_uid(self, iam_token):
        with grpc.secure_channel(self._host, self._get_creds()) as channel:
            account = YCAccessServiceClient(channel).authenticate(iam_token)
            return account.id


class FormsIamTokenYandexUser(FormsYandexUserMixin, BaseYandexUser):
    def __init__(self, cloud_uid, orgs, mechanism=None):
        self.cloud_uid = cloud_uid
        self.orgs = orgs
        self.uid = None
        self.default_email = None
        self.fields = {}
        self.authenticated_by = mechanism

    def is_authenticated(self):
        return bool(self.cloud_uid)


class Mechanism(BaseFormsMixin, BaseMechanism):
    YandexUser = FormsIamTokenYandexUser
    AnonymousYandexUser = FormsAnonymousYandexUser

    def extract_params(self, request):
        iam_token = self._get_iam_token(request)
        if not iam_token:
            return None

        return {
            'request': request,
            'iam_token': iam_token,
        }

    def _get_iam_token(self, request):
        auth = request.META.get('HTTP_AUTHORIZATION')
        if not auth:
            return None

        m = re.match(r'bearer\s+(.+)', auth, re.I)
        if not m:
            return None

        return m.group(1)

    def _get_access_client(self):
        return AccessClient()

    def _get_organizations(self, uid, cloud_uid):
        client = CachedDirectoryClient()
        return [
            str(org['id'])
            for org in client.get_organizations(uid, cloud_uid=cloud_uid)
        ]

    def apply(self, request, iam_token):
        cloud_uid = None
        client = self._get_access_client()
        try:
            cloud_uid = client.get_cloud_uid(iam_token)
        except UnauthenticatedException as e:
            logger.warn('Cloud auth error: %s', e)

        if not cloud_uid:
            return self.anonymous()

        if not request.has_orgs:
            orgs = self._get_organizations(request.META.get(settings.YAUTH_UID_HEADER), cloud_uid)
            request.orgs = orgs

        return self.YandexUser(cloud_uid, request.orgs, mechanism=self)

    def anonymous(self):
        return self.AnonymousYandexUser(mechanism=self)
